"""风控引擎骨架。"""

from __future__ import annotations

from typing import Iterable, List, Optional, Sequence

import structlog

from lacopro.agent.events import AgentEvent, EventPriority, EventType
from lacopro.notifications.base import NotificationLevel, Notifier
from lacopro.tools.portfolio_tool import PortfolioSnapshot

from .models import RiskAlert, RiskSeverity
from .rules import RiskRule

logger = structlog.get_logger("risk_engine")


SEVERITY_TO_PRIORITY = {
    RiskSeverity.CRITICAL: EventPriority.CRITICAL,
    RiskSeverity.WARNING: EventPriority.HIGH,
    RiskSeverity.INFO: EventPriority.NORMAL,
}

SEVERITY_TO_LEVEL = {
    RiskSeverity.CRITICAL: NotificationLevel.CRITICAL,
    RiskSeverity.WARNING: NotificationLevel.WARNING,
    RiskSeverity.INFO: NotificationLevel.INFO,
}


class RiskEngine:
    """汇集多条规则，输出告警与自动事件。"""

    def __init__(
        self,
        rules: Sequence[RiskRule],
        *,
        notifier: Optional[Notifier] = None,
        event_source: str = "risk_engine",
    ) -> None:
        self.rules = list(rules)
        self.notifier = notifier
        self.event_source = event_source

    async def evaluate(self, snapshot: PortfolioSnapshot) -> List[RiskAlert]:
        alerts: List[RiskAlert] = []
        for rule in self.rules:
            try:
                result = await rule.evaluate(snapshot)
            except Exception as exc:  # noqa: BLE001
                logger.exception("risk.rule_failed", rule_id=getattr(rule, "rule_id", "unknown"), error=str(exc))
                continue
            alerts.extend(result)

        if alerts:
            logger.warning(
                "risk.alerts_detected",
                count=len(alerts),
                rules=[alert.rule_id for alert in alerts],
            )
        return alerts

    async def build_auto_events(self, alerts: Iterable[RiskAlert]) -> List[AgentEvent]:
        events: List[AgentEvent] = []
        for alert in alerts:
            priority = SEVERITY_TO_PRIORITY.get(alert.severity, EventPriority.NORMAL)
            events.append(
                AgentEvent.create(
                    EventType.AUTO_SIGNAL,
                    source=self.event_source,
                    payload={
                        "rule_id": alert.rule_id,
                        "severity": alert.severity.value,
                        "message": alert.message,
                        "details": alert.details,
                    },
                    priority=priority,
                )
            )
        return events

    async def dispatch_notifications(self, alerts: Iterable[RiskAlert]) -> None:
        if not self.notifier:
            return
        for alert in alerts:
            level = SEVERITY_TO_LEVEL.get(alert.severity, NotificationLevel.INFO)
            try:
                await self.notifier.send(
                    subject=f"风险告警: {alert.rule_id}",
                    content=alert.message,
                    level=level,
                )
            except Exception as exc:  # noqa: BLE001
                logger.exception("risk.notify_failed", rule_id=alert.rule_id, error=str(exc))


__all__ = ["RiskEngine"]
