"""LLM模型管理器 - 支持运行时动态切换模型"""

from __future__ import annotations

import json
from pathlib import Path
from typing import Dict, List, Optional
from datetime import datetime
import structlog

logger = structlog.get_logger("model_manager")

# 支持的模型配置
SUPPORTED_MODELS = {
    "gpt-5-mini": {
        "name": "gpt-5-mini-2025-08-07",
        "provider": "yunwu",
        "description": "GPT-5 Mini (云雾AI) - 快速高效",
        "max_tokens": 8192,
        "temperature": 0.7,
        "supports_function_calling": True,
        "is_default": True
    },
    "grok-4-fast-reasoning": {
        "name": "grok-4-fast-reasoning",
        "provider": "yunwu",
        "description": "Grok 4 Fast Reasoning (云雾AI) - 超快响应推理",
        "max_tokens": 8192,
        "temperature": 0.7,
        "supports_function_calling": True,
        "is_default": False
    },
    "doubao-1.6": {
        "name": "doubao-seed-1-6-250615",
        "provider": "doubao",
        "description": "豆包1.6主力模型",
        "max_tokens": 4096,
        "temperature": 0.7,
        "supports_function_calling": True,
        "is_default": False
    },
    "doubao-flash": {
        "name": "doubao-seed-1-6-flash-250715",
        "provider": "doubao",
        "description": "豆包Flash快速模型",
        "max_tokens": 4096,
        "temperature": 0.7,
        "supports_function_calling": True,
        "is_default": False
    },
    "deepseek-v3": {
        "name": "deepseek-v3-1-250821",
        "provider": "doubao",  # 使用火山引擎API
        "description": "DeepSeek V3.1模型",
        "max_tokens": 8192,
        "temperature": 0.7,
        "supports_function_calling": True,
        "is_default": False
    }
}


class ModelManager:
    """LLM模型管理器，支持运行时切换模型"""

    def __init__(self, state_file: Optional[Path] = None):
        """
        初始化模型管理器

        Args:
            state_file: 状态文件路径，用于持久化当前模型选择
        """
        self.state_file = state_file or Path.home() / ".lacopro" / "model_state.json"
        self.state_file.parent.mkdir(parents=True, exist_ok=True)

        # 加载或初始化状态
        self.current_model = self._load_state()
        logger.info(
            "model_manager.initialized",
            current_model=self.current_model,
            state_file=str(self.state_file)
        )

    def _load_state(self) -> str:
        """从状态文件加载当前模型"""
        if self.state_file.exists():
            try:
                with open(self.state_file, 'r', encoding='utf-8') as f:
                    state = json.load(f)
                    model_key = state.get("current_model", "gpt-5-mini")
                    if model_key in SUPPORTED_MODELS:
                        logger.info("model_manager.state_loaded", model=model_key)
                        return model_key
            except Exception as e:
                logger.warning("model_manager.state_load_failed", error=str(e))

        # 默认返回主力模型
        return "gpt-5-mini"

    def _save_state(self):
        """保存当前模型到状态文件"""
        try:
            state = {
                "current_model": self.current_model,
                "updated_at": datetime.now().isoformat(),
                "model_info": SUPPORTED_MODELS[self.current_model]
            }
            with open(self.state_file, 'w', encoding='utf-8') as f:
                json.dump(state, f, ensure_ascii=False, indent=2)
            logger.info("model_manager.state_saved", model=self.current_model)
        except Exception as e:
            logger.error("model_manager.state_save_failed", error=str(e))

    def get_current_model(self) -> Dict[str, any]:
        """获取当前模型配置"""
        return {
            "key": self.current_model,
            **SUPPORTED_MODELS[self.current_model]
        }

    def get_current_model_name(self) -> str:
        """获取当前模型的完整名称"""
        return SUPPORTED_MODELS[self.current_model]["name"]

    def switch_model(self, model_key: str) -> Dict[str, any]:
        """
        切换模型

        Args:
            model_key: 模型简称 (doubao-1.6, doubao-flash, deepseek-v3)

        Returns:
            切换后的模型配置

        Raises:
            ValueError: 如果模型不存在
        """
        if model_key not in SUPPORTED_MODELS:
            raise ValueError(
                f"不支持的模型: {model_key}. "
                f"支持的模型: {', '.join(SUPPORTED_MODELS.keys())}"
            )

        old_model = self.current_model
        self.current_model = model_key
        self._save_state()

        logger.info(
            "model_manager.model_switched",
            from_model=old_model,
            to_model=model_key,
            model_name=SUPPORTED_MODELS[model_key]["name"]
        )

        return self.get_current_model()

    def list_models(self) -> List[Dict[str, any]]:
        """列出所有支持的模型"""
        return [
            {
                "key": key,
                "current": key == self.current_model,
                **config
            }
            for key, config in SUPPORTED_MODELS.items()
        ]

    def get_model_info(self, model_key: str) -> Optional[Dict[str, any]]:
        """获取指定模型的详细信息"""
        if model_key not in SUPPORTED_MODELS:
            return None
        return {
            "key": model_key,
            "current": model_key == self.current_model,
            **SUPPORTED_MODELS[model_key]
        }

    def reset_to_default(self) -> Dict[str, any]:
        """重置为默认模型"""
        default_model = next(
            (key for key, config in SUPPORTED_MODELS.items() if config.get("is_default")),
            "gpt-5-mini"
        )
        return self.switch_model(default_model)


# 全局单例
_model_manager: Optional[ModelManager] = None


def get_model_manager() -> ModelManager:
    """获取全局模型管理器单例"""
    global _model_manager
    if _model_manager is None:
        _model_manager = ModelManager()
    return _model_manager


__all__ = ["ModelManager", "get_model_manager", "SUPPORTED_MODELS"]
