"""事件模型与优先队列接口。

该模块定义了代理系统内部使用的统一事件结构，负责描述事件类型、优先级和上下文。
"""

from __future__ import annotations

import asyncio
from dataclasses import dataclass, field
from datetime import datetime, timezone
from enum import Enum, IntEnum
from typing import Any, Dict, Optional
from uuid import uuid4


class EventType(str, Enum):
    """事件类型枚举。"""

    MANUAL_COMMAND = "manual_command"
    SCHEDULED_TICK = "scheduled_tick"
    AUTO_SIGNAL = "auto_signal"


class EventPriority(IntEnum):
    """事件优先级（值越小优先级越高）。"""

    CRITICAL = 0
    HIGH = 10
    NORMAL = 20
    LOW = 30


DEFAULT_PRIORITY_BY_TYPE = {
    EventType.MANUAL_COMMAND: EventPriority.HIGH,
    EventType.SCHEDULED_TICK: EventPriority.NORMAL,
    EventType.AUTO_SIGNAL: EventPriority.CRITICAL,
}


@dataclass(order=True)
class AgentEvent:
    """Agent 内部流转的事件模型。"""

    priority: EventPriority
    event_type: EventType = field(compare=False)
    source: str = field(compare=False)
    event_id: str = field(default_factory=lambda: uuid4().hex, compare=False)
    payload: Dict[str, Any] = field(default_factory=dict, compare=False)
    created_at: datetime = field(
        default_factory=lambda: datetime.now(timezone.utc), compare=False
    )

    def __post_init__(self) -> None:
        if not isinstance(self.priority, EventPriority):
            self.priority = EventPriority(self.priority)
        if not isinstance(self.event_type, EventType):
            self.event_type = EventType(self.event_type)

    @classmethod
    def create(
        cls,
        event_type: EventType,
        source: str,
        payload: Optional[Dict[str, Any]] = None,
        priority: Optional[EventPriority] = None,
    ) -> "AgentEvent":
        """基于类型与来源快速构建事件。"""

        effective_priority = priority or DEFAULT_PRIORITY_BY_TYPE[event_type]
        return cls(
            priority=effective_priority,
            event_type=event_type,
            source=source,
            payload=payload or {},
        )


class EventQueue:
    """事件优先队列封装。"""

    def __init__(self, maxsize: int = 0) -> None:
        self._queue: asyncio.PriorityQueue[AgentEvent] = asyncio.PriorityQueue(maxsize=maxsize)

    async def put(self, event: AgentEvent) -> None:
        await self._queue.put(event)

    async def get(self) -> AgentEvent:
        return await self._queue.get()

    def empty(self) -> bool:
        return self._queue.empty()

    def qsize(self) -> int:
        return self._queue.qsize()

    def task_done(self) -> None:
        self._queue.task_done()


__all__ = [
    "AgentEvent",
    "EventQueue",
    "EventPriority",
    "EventType",
]
