/**
 * Unified Wallet Component for JuxGame - JuChain Edition
 * Handles wallet connection, display, and management across all pages
 * Version: 2.0.0 - JuChain Network Support
 */

class UnifiedWallet {
    constructor() {
        this.isConnected = false;
        this.account = null;
        this.agBalance = '0';
        this.juBalance = '0';
        this.networkName = 'JuChain';
        
        this.init();
    }
    
    async init() {
        // Wait for web3 to be ready
        await this.waitForWeb3();
        
        // Setup event listeners
        this.setupEventListeners();
        
        // Check if wallet was previously connected
        await this.checkPreviousConnection();
        
        // Start balance update timer
        this.startBalanceUpdater();
    }
    
    async waitForWeb3() {
        return new Promise((resolve) => {
            const checkWeb3 = () => {
                if (window.juxGameWeb3) {
                    resolve();
                } else {
                    setTimeout(checkWeb3, 100);
                }
            };
            checkWeb3();
        });
    }
    
    setupEventListeners() {
        // Connect button click
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('connect-wallet-btn')) {
                e.preventDefault();
                this.handleConnectClick();
            }
        });
        
        // Disconnect button click
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('disconnect-wallet-btn')) {
                e.preventDefault();
                this.handleDisconnect();
            }
        });
        
        // Wallet dropdown toggle
        document.addEventListener('click', (e) => {
            if (e.target.closest('.wallet-dropdown-toggle')) {
                e.preventDefault();
                this.toggleWalletDropdown();
            }
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', (e) => {
            const dropdown = document.querySelector('.wallet-dropdown');
            const toggle = e.target.closest('.wallet-dropdown-toggle');
            
            if (dropdown && !toggle && !dropdown.contains(e.target)) {
                dropdown.classList.remove('show');
            }
        });
        
        // Listen for wallet connection events
        window.addEventListener('walletConnected', (e) => {
            this.onWalletConnected(e.detail);
        });
        
        window.addEventListener('walletDisconnected', () => {
            this.onWalletDisconnected();
        });
    }
    
    async checkPreviousConnection() {
        try {
            // Check if web3State exists and is connected
            if (window.web3State && window.web3State.connected && window.web3State.account) {
                await this.onWalletConnected({ account: window.web3State.account });
            } else if (window.juxGameWeb3 && window.juxGameWeb3.isConnected()) {
                const account = window.juxGameWeb3.getAccount();
                if (account) {
                    await this.onWalletConnected({ account });
                }
            }
        } catch (error) {
            console.warn('Failed to check previous connection:', error);
        }
    }
    
    async handleConnectClick() {
        try {
            if (!window.juxGameWeb3) {
                throw new Error('Web3 not loaded yet');
            }
            
            const result = await window.juxGameWeb3.connectWallet();
            if (result.success) {
                await this.onWalletConnected({ account: result.account });
            }
        } catch (error) {
            console.error('Wallet connection failed:', error);
            this.showNotification('error', error.message);
        }
    }
    
    async onWalletConnected(data) {
        this.isConnected = true;
        this.account = data.account;
        
        // Update balances
        await this.updateBalances();
        
        // Update UI
        this.updateWalletUI();
        
        console.log('Unified Wallet: Connected to', this.account);
    }
    
    onWalletDisconnected() {
        this.isConnected = false;
        this.account = null;
        this.agBalance = '0';
        this.juBalance = '0';
        
        // Update UI
        this.updateWalletUI();
        
        console.log('Unified Wallet: Disconnected');
    }
    
    async handleDisconnect() {
        try {
            // Clear localStorage if any
            localStorage.removeItem('walletconnect');
            
            // Reset state
            this.onWalletDisconnected();
            
            // Refresh page to ensure clean state
            window.location.reload();
        } catch (error) {
            console.error('Disconnect failed:', error);
        }
    }
    
    async updateBalances() {
        if (!this.isConnected || !window.juxGameWeb3) return;
        
        try {
            // Get $Juxie balance
            const agBalance = await window.juxGameWeb3.getAGBalance();
            this.agBalance = parseFloat(agBalance).toLocaleString();
            
            // Get JU balance - simplified approach using MetaMask directly
            if (window.ethereum && this.account) {
                try {
                    const provider = new window.ethers.BrowserProvider(window.ethereum);
                    const juBalance = await provider.getBalance(this.account);
                    this.juBalance = parseFloat(window.ethers.formatEther(juBalance)).toFixed(4);
                    console.log('JU balance loaded:', this.juBalance);
                } catch (error) {
                    console.warn('Failed to get JU balance:', error);
                    this.juBalance = '0.0000';
                }
            } else {
                console.warn('No ethereum provider or account for JU balance');
                this.juBalance = '0.0000';
            }
            
            // Update UI if needed
            this.updateBalanceDisplay();
        } catch (error) {
            console.warn('Failed to update balances:', error);
        }
    }
    
    updateWalletUI() {
        const walletInfo = document.querySelector('.wallet-info');
        if (!walletInfo) {
            console.log('⚠️ Wallet info element not found');
            return;
        }
        
        if (this.isConnected) {
            walletInfo.innerHTML = `
                <div class="wallet-connected">
                    <div class="wallet-dropdown-toggle">
                        <div class="wallet-address">${this.formatAddress(this.account)}</div>
                        <div class="wallet-network">${this.networkName}</div>
                        <span class="dropdown-arrow">▼</span>
                    </div>
                    <div class="wallet-dropdown">
                        <div class="wallet-balances">
                            <div class="balance-item">
                                <span class="balance-label">$Juxie Balance:</span>
                                <span class="balance-value ag-balance">${this.agBalance} $Juxie</span>
                            </div>
                            <div class="balance-item">
                                <span class="balance-label">JU Balance:</span>
                                <span class="balance-value ju-balance">${this.juBalance} JU</span>
                            </div>
                        </div>
                        <div class="wallet-actions">
                            <button class="btn btn-small btn-danger disconnect-wallet-btn">Disconnect</button>
                        </div>
                    </div>
                </div>
            `;
        } else {
            walletInfo.innerHTML = `
                <button class="btn btn-primary connect-wallet-btn">Connect Wallet</button>
            `;
        }
    }
    
    updateBalanceDisplay() {
        const agBalanceElement = document.querySelector('.ag-balance');
        const juBalanceElement = document.querySelector('.ju-balance');
        
        if (agBalanceElement) {
            agBalanceElement.textContent = `${this.agBalance} $Juxie`;
        }
        
        if (juBalanceElement) {
            juBalanceElement.textContent = `${this.juBalance} JU`;
        }
    }
    
    toggleWalletDropdown() {
        const dropdown = document.querySelector('.wallet-dropdown');
        if (dropdown) {
            dropdown.classList.toggle('show');
        }
    }
    
    formatAddress(address) {
        if (!address) return '';
        return `${address.slice(0, 6)}...${address.slice(-4)}`;
    }
    
    startBalanceUpdater() {
        // Update balances every 30 seconds
        setInterval(() => {
            if (this.isConnected) {
                this.updateBalances();
            }
        }, 30000);
    }
    
    showNotification(type, message) {
        // Use existing notification system
        if (window.juxGameWeb3 && window.juxGameWeb3.showNotification) {
            window.juxGameWeb3.showNotification(type, message);
        } else {
            console.log(`${type.toUpperCase()}: ${message}`);
        }
    }
    
    // Public API for other components
    getState() {
        return {
            isConnected: this.isConnected,
            account: this.account,
            agBalance: this.agBalance,
            juBalance: this.juBalance
        };
    }
}

// Global wallet component instance
window.unifiedWallet = null;

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.unifiedWallet = new UnifiedWallet();
    console.log('🔗 Wallet component initialized on DOMContentLoaded');
});

// Also initialize on window load as backup
window.addEventListener('load', () => {
    if (!window.unifiedWallet) {
        console.log('🔄 Backup wallet initialization');
        window.unifiedWallet = new UnifiedWallet();
    }
});

// Force initialization after 2 seconds if still not initialized
setTimeout(() => {
    if (!window.unifiedWallet || !window.unifiedWallet.isConnected) {
        console.log('🚨 Force wallet initialization check');
        if (!window.unifiedWallet) {
            window.unifiedWallet = new UnifiedWallet();
        }
        // Force UI update
        if (window.unifiedWallet && typeof window.unifiedWallet.updateWalletUI === 'function') {
            window.unifiedWallet.updateWalletUI();
        }
    }
}, 3000);

// Make available globally
window.UnifiedWallet = UnifiedWallet;