/**
 * P2E (Play-to-Earn) functionality for JuxGame
 * Handles staking, unstaking, and reward claiming
 */

class JuxGameP2E {
    constructor() {
        this.stakedNFTs = [];
        this.unstakedNFTs = [];
        this.totalPendingRewards = '0';
        this.isProcessing = false;
        this.selectedNFTs = [];
        
        this.init();
    }
    
    async init() {
        this.setupEventListeners();
        this.startRewardTimer();
        
        // Wait for wallet connection
        this.waitForWallet();
    }
    
    waitForWallet() {
        const checkWallet = () => {
            if (window.walletManager && window.walletManager.isConnected()) {
                this.onWalletConnected();
            } else {
                setTimeout(checkWallet, 1000);
            }
        };
        checkWallet();
        
        // Listen for wallet connection events
        window.addEventListener('walletConnected', () => {
            this.onWalletConnected();
        });
    }
    
    async onWalletConnected() {
        await this.loadStakedNFTs();
        await this.loadUnstakedNFTs();
        await this.updateTotalRewards();
        this.updateUI();
        // Force update stats display
        setTimeout(() => {
            this.updateStatsDisplay();
        }, 1000);
    }
    
    setupEventListeners() {
        // Unstake buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('unstake-btn')) {
                e.preventDefault();
                const tokenId = e.target.dataset.tokenId;
                this.handleUnstake(tokenId);
            }
        });
        
        // Claim rewards buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('claim-btn')) {
                e.preventDefault();
                const tokenId = e.target.dataset.tokenId;
                this.handleClaimSingle(tokenId);
            }
        });
        
        // Claim all rewards button
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('claim-all-btn')) {
                e.preventDefault();
                this.handleClaimAll();
            }
        });
        
        // NFT selection for batch operations
        document.addEventListener('change', (e) => {
            if (e.target.classList.contains('nft-select')) {
                this.handleNFTSelection(e.target);
            }
        });
        
        // Select all checkbox
        document.addEventListener('change', (e) => {
            if (e.target.id === 'select-all-nfts') {
                this.handleSelectAll(e.target.checked);
            }
        });
        
        // Refresh data periodically
        setInterval(() => {
            if (window.walletManager && window.walletManager.isConnected()) {
                this.loadStakedNFTs();
                this.updateTotalRewards();
            }
        }, 30000); // Every 30 seconds
    }
    
    startRewardTimer() {
        // Update rewards display every 30 seconds to avoid excessive API calls
        setInterval(() => {
            this.updateRewardDisplays();
        }, 30000);
    }
    
    async loadStakedNFTs() {
        if (!window.walletManager || !window.walletManager.isConnected()) return;
        
        try {
            const web3 = window.walletManager.getJuxGameWeb3();
            const stakedTokensInfo = await web3.getUserStakedTokensInfo(window.walletManager.getAccount());
            
            // Ensure we have an array
            if (!Array.isArray(stakedTokensInfo)) {
                this.stakedNFTs = [];
            } else {
                this.stakedNFTs = stakedTokensInfo.map(info => ({
                    tokenId: info.tokenId.toString(),
                    owner: info.owner,
                    stakedAt: parseInt(info.stakedAt.toString()),
                    lastClaimAt: parseInt(info.lastClaimAt.toString()),
                    petType: parseInt(info.petType.toString()),
                    dailyReward: info.dailyReward // Already formatted by ethers.formatEther in web3-final.js
                }));
            }
            
            this.updateStakedNFTsUI();
            
        } catch (error) {
            console.error('Failed to load staked NFTs:', error);
            this.stakedNFTs = []; // Set to empty array on error
            this.updateStakedNFTsUI();
        }
    }
    
    async loadUnstakedNFTs() {
        if (!window.walletManager || !window.walletManager.isConnected()) return;
        
        try {
            const web3 = window.walletManager.getJuxGameWeb3();
            const contracts = web3.getContracts();
            
            if (!contracts.juxNFT) {
                console.warn('NFT contract not available');
                return;
            }
            
            // Get all user's NFTs
            const tokenIds = await contracts.juxNFT.getTokensByOwner(window.walletManager.getAccount());
            
            // Get staked token IDs
            const stakedTokenIds = await contracts.staking.getUserStakedTokens(window.walletManager.getAccount());
            const stakedSet = new Set(stakedTokenIds.map(id => id.toString()));
            
            // Filter to get only unstaked NFTs
            const unstakedTokenIds = tokenIds.filter(tokenId => !stakedSet.has(tokenId.toString()));
            
            this.unstakedNFTs = [];
            
            for (const tokenId of unstakedTokenIds) {
                const tokenInfo = await contracts.juxNFT.getTokenInfo(tokenId);
                this.unstakedNFTs.push({
                    tokenId: tokenId.toString(),
                    petType: Number(tokenInfo[0].toString()),
                    config: {
                        name: tokenInfo[1].name,
                        rarity: Number(tokenInfo[1].rarity.toString()),
                        dailyReward: ethers.formatEther(tokenInfo[1].dailyReward)
                    }
                });
            }
            
            this.updateUnstakedNFTsUI();
            
        } catch (error) {
            console.error('Failed to load unstaked NFTs:', error);
            this.unstakedNFTs = [];
            this.updateUnstakedNFTsUI();
        }
    }
    
    async updateTotalRewards() {
        if (!window.walletManager || !window.walletManager.isConnected()) return;
        
        try {
            const web3 = window.walletManager.getJuxGameWeb3();
            this.totalPendingRewards = await web3.getUserTotalPendingRewards(window.walletManager.getAccount());
            
            this.updateTotalRewardsUI();
            
        } catch (error) {
            console.error('Failed to update total rewards:', error);
        }
    }
    
    async handleUnstake(tokenId) {
        if (this.isProcessing) return;
        
        if (!window.walletManager || !window.walletManager.isConnected()) {
            this.showNotification('error', 'Please connect your wallet first');
            return;
        }
        
        try {
            this.isProcessing = true;
            
            const web3 = window.walletManager.getJuxGameWeb3();
            
            this.showNotification('info', 'Recalling pet from adventure... Please confirm the transaction. Fee: 0.1 JU');
            
            const tx = await web3.unstakeToken(tokenId);
            
            this.showNotification('success', `Recall transaction submitted! ${tx.hash ? tx.hash.slice(0, 10) : 'Hash'}...`);
            
            await tx.wait();
            
            this.showNotification('success', 'Pet recalled successfully! Treasures have been harvested.');
            
            // Reload data
            setTimeout(() => {
                this.loadStakedNFTs();
                this.updateTotalRewards();
            }, 3000);
            
        } catch (error) {
            console.error('Unstake failed:', error);
            let errorMessage = 'Unstake failed';
            
            if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            } else if (error.message.includes('Insufficient claim fee')) {
                errorMessage = 'Insufficient JU for claim fee (0.1 JU required)';
            }
            
            this.showNotification('error', errorMessage);
        } finally {
            this.isProcessing = false;
        }
    }
    
    async handleClaimSingle(tokenId) {
        if (this.isProcessing) return;
        
        if (!window.walletManager || !window.walletManager.isConnected()) {
            this.showNotification('error', 'Please connect your wallet first');
            return;
        }
        
        try {
            this.isProcessing = true;
            
            const web3 = window.walletManager.getJuxGameWeb3();
            
            this.showNotification('info', 'Harvesting treasures... Please confirm the transaction. Fee: 0.1 JU');
            
            const tx = await web3.claimRewards([tokenId]);
            
            this.showNotification('success', `Harvest transaction submitted! ${tx.hash ? tx.hash.slice(0, 10) : 'Hash'}...`);
            
            await tx.wait();
            
            this.showNotification('success', 'Treasures harvested successfully!');
            
            // Reload data
            setTimeout(() => {
                this.loadStakedNFTs();
                this.updateTotalRewards();
            }, 3000);
            
        } catch (error) {
            console.error('Claim failed:', error);
            let errorMessage = 'Claim failed';
            
            if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            } else if (error.message.includes('No rewards to claim')) {
                errorMessage = 'No treasures available to harvest';
            } else if (error.message.includes('Insufficient claim fee')) {
                errorMessage = 'Insufficient JU for claim fee (0.1 JU required)';
            }
            
            this.showNotification('error', errorMessage);
        } finally {
            this.isProcessing = false;
        }
    }
    
    async handleClaimAll() {
        if (this.isProcessing) return;
        
        const selectedTokenIds = this.selectedNFTs.length > 0 ? 
            this.selectedNFTs : 
            this.stakedNFTs.map(nft => nft.tokenId);
            
        if (selectedTokenIds.length === 0) {
            this.showNotification('error', 'No pets selected for harvesting');
            return;
        }
        
        if (!window.walletManager || !window.walletManager.isConnected()) {
            this.showNotification('error', 'Please connect your wallet first');
            return;
        }
        
        try {
            this.isProcessing = true;
            
            const web3 = window.walletManager.getJuxGameWeb3();
            
            this.showNotification('info', `Harvesting treasures from ${selectedTokenIds.length} pets... Please confirm the transaction. Fee: 0.1 JU`);
            
            const tx = await web3.claimRewards(selectedTokenIds);
            
            this.showNotification('success', `Batch harvest transaction submitted! ${tx.hash ? tx.hash.slice(0, 10) : 'Hash'}...`);
            
            await tx.wait();
            
            this.showNotification('success', `Treasures harvested successfully from ${selectedTokenIds.length} pets!`);
            
            // Clear selections
            this.selectedNFTs = [];
            
            // Reload data
            setTimeout(() => {
                this.loadStakedNFTs();
                this.updateTotalRewards();
            }, 3000);
            
        } catch (error) {
            console.error('Batch claim failed:', error);
            let errorMessage = 'Batch claim failed';
            
            if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            } else if (error.message.includes('No rewards to claim')) {
                errorMessage = 'No treasures available to harvest';
            } else if (error.message.includes('Insufficient claim fee')) {
                errorMessage = 'Insufficient JU for claim fee (0.1 JU required)';
            }
            
            this.showNotification('error', errorMessage);
        } finally {
            this.isProcessing = false;
        }
    }
    
    handleNFTSelection(checkbox) {
        const tokenId = checkbox.value;
        
        if (checkbox.checked) {
            if (!this.selectedNFTs.includes(tokenId)) {
                this.selectedNFTs.push(tokenId);
            }
        } else {
            this.selectedNFTs = this.selectedNFTs.filter(id => id !== tokenId);
        }
        
        this.updateSelectionUI();
    }
    
    handleSelectAll(checked) {
        this.selectedNFTs = checked ? this.stakedNFTs.map(nft => nft.tokenId) : [];
        
        // Update all checkboxes
        document.querySelectorAll('.nft-select').forEach(checkbox => {
            checkbox.checked = checked;
        });
        
        this.updateSelectionUI();
    }
    
    updateSelectionUI() {
        const selectedCount = this.selectedNFTs.length;
        const claimAllBtn = document.querySelector('.claim-all-btn');
        
        if (claimAllBtn) {
            if (selectedCount > 0) {
                claimAllBtn.textContent = `Harvest Selected (${selectedCount})`;
                claimAllBtn.disabled = false;
            } else {
                claimAllBtn.textContent = 'Harvest All';
                claimAllBtn.disabled = this.stakedNFTs.length === 0;
            }
        }
    }
    
    updateStakedNFTsUI() {
        const container = document.querySelector('.adventure-nfts-grid');
        if (!container) return;
        
        if (this.stakedNFTs.length === 0) {
            container.innerHTML = `
                <div class="no-staked-message">
                    <h3>No Pets on Adventure</h3>
                    <p>You don't have any pets on adventure yet.</p>
                    <p>Go to the <a href="game.html">Mint page</a> to get some pets and then send them on adventures to start earning treasures!</p>
                </div>
            `;
            return;
        }
        
        container.innerHTML = this.stakedNFTs.map(nft => {
            const petConfig = this.getPetConfig(nft.petType);
            // Fix dailyReward calculation - it's already formatted by ethers.formatEther
            const dailyReward = parseFloat(nft.dailyReward).toLocaleString();
            const stakedDays = Math.floor((Date.now() / 1000 - nft.stakedAt) / 86400);
            
            return `
                <div class="adventure-nft-card">
                    <div class="nft-select-wrapper">
                        <input type="checkbox" class="nft-select" value="${nft.tokenId}" />
                    </div>
                    <div class="nft-image">
                        <img src="https://www.juxgame.site/assets/p${nft.petType}.png" 
                             alt="${petConfig.name} #${nft.tokenId}" 
                             data-pet-type="${nft.petType}"
                             class="adventure-nft-image"
                             style="opacity: 1; transition: opacity 0.3s ease-in-out;">
                    </div>
                    <div class="nft-info">
                        <h4>${petConfig.name} #${nft.tokenId}</h4>
                        <span class="nft-rarity ${petConfig.rarityClass}">${petConfig.rarity}</span>
                        <div class="nft-stats">
                            <div class="stat">
                                <span class="stat-label">Daily Reward:</span>
                                <span class="stat-value">${dailyReward} $Juxie</span>
                            </div>
                            <div class="stat">
                                <span class="stat-label">Staked:</span>
                                <span class="stat-value">${stakedDays} days</span>
                            </div>
                            <div class="stat">
                                <span class="stat-label">Pending Treasures:</span>
                                <span class="stat-value pending-reward" data-token-id="${nft.tokenId}">Loading...</span>
                            </div>
                        </div>
                        <div class="nft-actions">
                            <button class="btn btn-secondary claim-btn" data-token-id="${nft.tokenId}">Harvest</button>
                            <button class="btn btn-danger unstake-btn" data-token-id="${nft.tokenId}">Recall</button>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        this.updateSelectionUI();
        this.updateRewardDisplays();
        // Update stats immediately after loading staked NFTs
        this.updateStatsDisplay();
    }
    
    updateUnstakedNFTsUI() {
        const container = document.querySelector('.my-pets-grid');
        if (!container) return;
        
        if (this.unstakedNFTs.length === 0) {
            container.innerHTML = `
                <div class="no-staked-message">
                    <h3>No Unstaked Pets</h3>
                    <p>All your pets are currently on adventures, or you don't have any pets yet.</p>
                    <p>Go to the <a href="game.html">Mint page</a> to get more pets!</p>
                </div>
            `;
            return;
        }
        
        container.innerHTML = this.unstakedNFTs.map(nft => {
            const rarityClass = this.getRarityClass(nft.config.rarity);
            const dailyReward = parseFloat(nft.config.dailyReward).toLocaleString();
            
            return `
                <div class="pet-card">
                    <div class="pet-image-container">
                        <img src="https://www.juxgame.site/assets/p${nft.petType}.png" 
                             alt="${nft.config.name} #${nft.tokenId}" 
                             data-pet-type="${nft.petType}"
                             class="pet-image"
                             style="opacity: 1; transition: opacity 0.3s ease-in-out;">
                    </div>
                    <div class="pet-info">
                        <h4>${nft.config.name} #${nft.tokenId}</h4>
                        <span class="pet-rarity ${rarityClass}">${this.getRarityName(nft.config.rarity)}</span>
                        <div class="pet-stats">
                            <span>Earns: ${dailyReward} $Juxie/day</span>
                        </div>
                        <div class="nft-actions">
                            <button class="btn ${window.FEATURES?.STAKING_ENABLED ? 'btn-primary' : 'btn-disabled'} stake-btn" data-token-id="${nft.tokenId}">
                                ${window.FEATURES?.STAKING_ENABLED ? 'Send to Adventure' : 'Coming Soon'}
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        // Add click handlers for stake buttons
        container.querySelectorAll('.stake-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                e.preventDefault();
                e.stopPropagation();
                
                // Check if P2E features are enabled
                if (!window.FEATURES?.STAKING_ENABLED) {
                    this.showNotification('info', '🎮 P2E Features Coming Soon! Stay tuned for the official launch.');
                    return;
                }
                
                const tokenId = parseInt(btn.dataset.tokenId);
                await this.handleStake(tokenId);
            });
        });
    }
    
    updateTotalRewardsUI() {
        const totalRewardsElement = document.querySelector('#total-pending-rewards');
        if (totalRewardsElement) {
            const totalRewards = parseFloat(this.totalPendingRewards).toLocaleString();
            totalRewardsElement.textContent = `${totalRewards} $Juxie`;
        }
    }
    
    async updateRewardDisplays() {
        if (!window.walletManager || !window.walletManager.isConnected()) return;
        
        const pendingElements = document.querySelectorAll('.pending-reward');
        let totalDailyEarnings = 0;
        
        for (const element of pendingElements) {
            const tokenId = element.dataset.tokenId;
            try {
                const web3 = window.walletManager.getJuxGameWeb3();
                const pending = await web3.calculatePendingReward(tokenId);
                const pendingFormatted = parseFloat(pending).toLocaleString();
                element.textContent = `${pendingFormatted} $Juxie`;
                
                // Find corresponding NFT for daily reward calculation
                const nft = this.stakedNFTs.find(n => n.tokenId === tokenId);
                if (nft) {
                    totalDailyEarnings += parseFloat(nft.dailyReward);
                }
            } catch (error) {
                element.textContent = 'Error';
            }
        }
        
        // Update stats display (but not total claimed to avoid spam)
        this.updateBasicStatsDisplay(totalDailyEarnings);
    }
    
    updateBasicStatsDisplay(dailyEarnings = null) {
        // Update staked count
        const stakedCountElement = document.getElementById('staked-count');
        if (stakedCountElement) {
            stakedCountElement.textContent = this.stakedNFTs.length;
        }
        
        // Update daily earnings
        const dailyEarningsElement = document.getElementById('daily-earnings');
        if (dailyEarningsElement && dailyEarnings !== null) {
            dailyEarningsElement.textContent = `${dailyEarnings.toLocaleString()} $Juxie`;
        } else if (dailyEarningsElement) {
            // Calculate from staked NFTs
            const total = this.stakedNFTs.reduce((sum, nft) => sum + parseFloat(nft.dailyReward), 0);
            dailyEarningsElement.textContent = `${total.toLocaleString()} $Juxie`;
        }
    }

    async updateStatsDisplay(dailyEarnings = null) {
        // Update basic stats first
        this.updateBasicStatsDisplay(dailyEarnings);
        
        // Get historical total earned from contract (only when explicitly called)
        const totalEarnedElement = document.getElementById('total-earned');
        if (totalEarnedElement && window.walletManager && window.walletManager.isConnected()) {
            try {
                const web3 = window.walletManager.getJuxGameWeb3();
                const totalClaimed = await web3.getUserTotalClaimed(window.walletManager.getAccount());
                const totalClaimedFormatted = parseFloat(totalClaimed).toLocaleString();
                totalEarnedElement.textContent = `${totalClaimedFormatted} $Juxie`;
            } catch (error) {
                console.error('Failed to get total claimed:', error);
                totalEarnedElement.textContent = '0 $Juxie';
            }
        }
    }
    
    getPetConfig(petType) {
        const configs = {
            1: { name: 'Aqua Striker', rarity: 'Common', rarityClass: 'common' },
            2: { name: 'Terra Guardian', rarity: 'Common', rarityClass: 'common' },
            3: { name: 'Aero Blitzer', rarity: 'Rare', rarityClass: 'rare' },
            4: { name: 'Ember Drake', rarity: 'Rare', rarityClass: 'rare' },
            5: { name: 'Chrono Weaver', rarity: 'Legendary', rarityClass: 'legendary' },
            6: { name: 'Void Lurker', rarity: 'Legendary', rarityClass: 'legendary' }
        };
        return configs[petType] || { name: 'Unknown', rarity: 'Common', rarityClass: 'common' };
    }
    
    async updateUI() {
        this.updateStakedNFTsUI();
        this.updateUnstakedNFTsUI();
        this.updateTotalRewardsUI();
        await this.updateStatsDisplay();
    }
    
    async handleStake(tokenId) {
        if (!window.walletManager || !window.walletManager.isConnected()) {
            this.showNotification('error', 'Please connect your wallet first');
            return;
        }
        
        try {
            this.showNotification('info', 'Sending pet on adventure... Please confirm the transaction.');
            
            const web3 = window.walletManager.getJuxGameWeb3();
            const contracts = web3.getContracts();
            
            // First approve the NFT if needed
            const approvedAddress = await contracts.juxNFT.getApproved(tokenId);
            if (approvedAddress.toLowerCase() !== contracts.staking.target.toLowerCase()) {
                this.showNotification('info', 'Approving NFT for staking... Please confirm the transaction.');
                const approveTx = await contracts.juxNFT.approve(contracts.staking.target, tokenId);
                await approveTx.wait();
                this.showNotification('success', 'NFT approved for staking!');
            }
            
            const tx = await contracts.staking.stakeToken(tokenId);
            
            this.showNotification('success', `Adventure transaction submitted! ${tx.hash.slice(0, 10)}...`);
            
            await tx.wait();
            
            this.showNotification('success', 'Pet sent on adventure successfully! It will start earning treasures.');
            
            // Reload data
            setTimeout(() => {
                this.loadStakedNFTs();
                this.loadUnstakedNFTs();
            }, 3000);
            
        } catch (error) {
            console.error('Stake failed:', error);
            let errorMessage = 'Stake failed';
            
            if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            }
            
            this.showNotification('error', errorMessage);
        }
    }
    
    getRarityClass(rarity) {
        const rarityMap = {
            0: 'common',
            1: 'rare', 
            2: 'legendary'
        };
        return rarityMap[rarity] || 'common';
    }
    
    getRarityName(rarity) {
        const rarityMap = {
            0: 'Common',
            1: 'Rare',
            2: 'Legendary'
        };
        return rarityMap[rarity] || 'Common';
    }
    
    showNotification(type, message, duration = 5000) {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.style.cssText = `
            position: fixed;
            top: 120px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 10000;
            padding: 15px 20px;
            border-radius: 8px;
            max-width: 400px;
            font-weight: 500;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        `;
        
        const colors = {
            success: { bg: '#d4edda', border: '#c3e6cb', color: '#155724' },
            error: { bg: '#f8d7da', border: '#f5c6cb', color: '#721c24' },
            info: { bg: '#d1ecf1', border: '#bee5eb', color: '#0c5460' }
        };
        
        const color = colors[type] || colors.info;
        notification.style.backgroundColor = color.bg;
        notification.style.border = `2px solid ${color.border}`;
        notification.style.color = color.color;
        
        notification.textContent = message;
        
        document.body.appendChild(notification);
        
        // Auto remove
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, duration);
    }
}

// Initialize P2E functionality when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.juxGameP2E = new JuxGameP2E();
});

// Make available globally
window.JuxGameP2E = JuxGameP2E;