import Phaser from 'phaser';
import { GAME_CONFIG, COLORS } from '../config/GameConfig';

export type TrapType = 'spike' | 'laser' | 'poison' | 'slow' | 'stun';

interface TrapEffect {
    activate(trap: Trap, player: any): void;
    update(trap: Trap, time: number, delta: number): void;
}

// 尖刺陷阱 - 造成直接伤害
class SpikeTrapEffect implements TrapEffect {
    private damage: number = 1;
    private cooldown: number = 1000;
    private lastActivation: number = 0;
    
    activate(trap: Trap, player: any): void {
        const now = trap.scene.time.now;
        if (now - this.lastActivation < this.cooldown) return;
        
        this.lastActivation = now;
        
        // 造成伤害
        if (player.takeDamage) {
            player.takeDamage(this.damage);
        }
        
        // 播放尖刺动画
        this.playSpikeAnimation(trap);
        
        // 播放音效
        // trap.scene.sound.play('spike_trap', { volume: 0.6 });
    }
    
    update(trap: Trap, time: number, delta: number): void {
        // 尖刺陷阱可以持续更新视觉效果
    }
    
    private playSpikeAnimation(trap: Trap): void {
        // 创建尖刺动画
        const spikes = [];
        for (let i = 0; i < 4; i++) {
            const angle = (i / 4) * Math.PI * 2;
            const x = trap.x + Math.cos(angle) * 20;
            const y = trap.y + Math.sin(angle) * 20;
            
            const spike = trap.scene.add.triangle(x, y, 0, -10, -5, 5, 5, 5, 0xff0000, 0.8);
            spikes.push(spike);
            
            // 尖刺动画
            trap.scene.tweens.add({
                targets: spike,
                scaleX: 0,
                scaleY: 0,
                alpha: 0,
                duration: 500,
                ease: 'Power2',
                onComplete: () => {
                    spike.destroy();
                }
            });
        }
    }
}

// 激光陷阱 - 持续伤害
class LaserTrapEffect implements TrapEffect {
    private damage: number = 0.5;
    private tickRate: number = 500; // 每500ms造成一次伤害
    private lastTick: number = 0;
    private laserBeam: Phaser.GameObjects.Graphics | null = null;
    private isActive: boolean = false;
    
    activate(trap: Trap, player: any): void {
        this.isActive = true;
        this.createLaserBeam(trap);
        
        // 播放激光音效
        // trap.scene.sound.play('laser_trap', { volume: 0.4, loop: true });
    }
    
    update(trap: Trap, time: number, delta: number): void {
        if (!this.isActive) return;
        
        // 更新激光束
        this.updateLaserBeam(trap);
        
        // 造成伤害
        const now = trap.scene.time.now;
        if (now - this.lastTick >= this.tickRate) {
            this.lastTick = now;
            
            // 检查玩家是否还在陷阱范围内
            if (trap.playerInRange) {
                if (trap.player && trap.player.takeDamage) {
                    trap.player.takeDamage(this.damage);
                }
            }
        }
    }
    
    private createLaserBeam(trap: Trap): void {
        if (this.laserBeam) return;
        
        this.laserBeam = trap.scene.add.graphics();
        this.updateLaserBeam(trap);
    }
    
    private updateLaserBeam(trap: Trap): void {
        if (!this.laserBeam) return;
        
        this.laserBeam.clear();
        
        // 绘制激光束
        const beamLength = 100;
        const beamWidth = 3;
        
        this.laserBeam.lineStyle(beamWidth, 0xff0000, 0.8);
        this.laserBeam.lineBetween(
            trap.x - beamLength / 2,
            trap.y,
            trap.x + beamLength / 2,
            trap.y
        );
        
        // 添加激光脉冲效果
        const pulse = Math.sin(trap.scene.time.now * 0.01) * 0.3 + 0.7;
        this.laserBeam.setAlpha(pulse);
    }
    
    deactivate(trap: Trap): void {
        this.isActive = false;
        
        if (this.laserBeam) {
            this.laserBeam.destroy();
            this.laserBeam = null;
        }
    }
}

// 毒雾陷阱 - 持续伤害和减速
class PoisonTrapEffect implements TrapEffect {
    private damage: number = 0.3;
    private slowFactor: number = 0.7;
    private duration: number = 3000;
    private tickRate: number = 1000;
    private lastTick: number = 0;
    private poisonCloud: Phaser.GameObjects.Particles.ParticleEmitter | null = null;
    
    activate(trap: Trap, player: any): void {
        this.createPoisonCloud(trap);
        this.applyPoisonEffect(player);
        
        // 播放毒雾音效
        // trap.scene.sound.play('poison_trap', { volume: 0.5 });
    }
    
    update(trap: Trap, time: number, delta: number): void {
        if (!this.poisonCloud) return;
        
        const now = trap.scene.time.now;
        if (now - this.lastTick >= this.tickRate) {
            this.lastTick = now;
            
            if (trap.playerInRange && trap.player) {
                if (trap.player.takeDamage) {
                    trap.player.takeDamage(this.damage);
                }
            }
        }
    }
    
    private createPoisonCloud(trap: Trap): void {
        if (this.poisonCloud) return;
        
        this.poisonCloud = trap.scene.add.particles(trap.x, trap.y, 'collectible', {
            scale: { start: 0.5, end: 1.5 },
            lifespan: 2000,
            quantity: 2,
            frequency: 100,
            tint: 0x00ff00,
            alpha: { start: 0.6, end: 0 },
            speed: { min: 10, max: 30 },
            rotate: { start: 0, end: 360 },
            blendMode: 'ADD'
        });
        
        // 设置粒子跟随陷阱
        this.poisonCloud.startFollow(trap);
    }
    
    private applyPoisonEffect(player: any): void {
        // 应用减速效果
        if (player.applyStatusEffect) {
            player.applyStatusEffect('slow', this.slowFactor, this.duration);
        }
    }
    
    deactivate(trap: Trap): void {
        if (this.poisonCloud) {
            this.poisonCloud.stop();
            this.poisonCloud.destroy();
            this.poisonCloud = null;
        }
    }
}

// 减速陷阱 - 降低移动速度
class SlowTrapEffect implements TrapEffect {
    private slowFactor: number = 0.5;
    private duration: number = 2000;
    private slowField: Phaser.GameObjects.Graphics | null = null;
    
    activate(trap: Trap, player: any): void {
        this.createSlowField(trap);
        this.applySlowEffect(player);
        
        // 播放减速音效
        // trap.scene.sound.play('slow_trap', { volume: 0.4 });
    }
    
    update(trap: Trap, time: number, delta: number): void {
        this.updateSlowField(trap);
    }
    
    private createSlowField(trap: Trap): void {
        if (this.slowField) return;
        
        this.slowField = trap.scene.add.graphics();
        this.updateSlowField(trap);
    }
    
    private updateSlowField(trap: Trap): void {
        if (!this.slowField) return;
        
        this.slowField.clear();
        
        // 绘制减速场
        const radius = 40;
        const alpha = 0.3 + Math.sin(trap.scene.time.now * 0.005) * 0.2;
        
        this.slowField.fillStyle(0x0000ff, alpha);
        this.slowField.fillCircle(trap.x, trap.y, radius);
        
        // 添加波纹效果
        for (let i = 1; i <= 3; i++) {
            const ringRadius = radius * i / 3;
            const ringAlpha = alpha * (1 - i / 4) * 0.5;
            
            this.slowField.lineStyle(2, 0x0000ff, ringAlpha);
            this.slowField.strokeCircle(trap.x, trap.y, ringRadius);
        }
    }
    
    private applySlowEffect(player: any): void {
        if (player.applyStatusEffect) {
            player.applyStatusEffect('slow', this.slowFactor, this.duration);
        }
    }
    
    deactivate(trap: Trap): void {
        if (this.slowField) {
            this.slowField.destroy();
            this.slowField = null;
        }
    }
}

// 眩晕陷阱 - 使玩家短暂无法移动
class StunTrapEffect implements TrapEffect {
    private stunDuration: number = 1500;
    private stunEffect: Phaser.GameObjects.Graphics | null = null;
    
    activate(trap: Trap, player: any): void {
        this.createStunEffect(trap);
        this.applyStunEffect(player);
        
        // 播放眩晕音效
        // trap.scene.sound.play('stun_trap', { volume: 0.6 });
    }
    
    update(trap: Trap, time: number, delta: number): void {
        this.updateStunEffect(trap);
    }
    
    private createStunEffect(trap: Trap): void {
        if (this.stunEffect) return;
        
        this.stunEffect = trap.scene.add.graphics();
        this.updateStunEffect(trap);
    }
    
    private updateStunEffect(trap: Trap): void {
        if (!this.stunEffect) return;
        
        this.stunEffect.clear();
        
        // 绘制眩晕效果
        const radius = 35;
        const alpha = 0.4 + Math.sin(trap.scene.time.now * 0.01) * 0.3;
        
        this.stunEffect.fillStyle(0xffff00, alpha);
        this.stunEffect.fillCircle(trap.x, trap.y, radius);
        
        // 添加眩晕星形效果
        this.drawStarEffect(trap);
    }
    
    private drawStarEffect(trap: Trap): void {
        const starSize = 15;
        const rotation = trap.scene.time.now * 0.005;
        
        for (let i = 0; i < 5; i++) {
            const angle = (i / 5) * Math.PI * 2 + rotation;
            const x = trap.x + Math.cos(angle) * 25;
            const y = trap.y + Math.sin(angle) * 25;
            
            this.stunEffect!.fillStyle(0xffffff, 0.8);
            this.stunEffect!.fillCircle(x, y, 3);
        }
    }
    
    private applyStunEffect(player: any): void {
        if (player.applyStatusEffect) {
            player.applyStatusEffect('stun', 1, this.stunDuration);
        }
    }
    
    deactivate(trap: Trap): void {
        if (this.stunEffect) {
            this.stunEffect.destroy();
            this.stunEffect = null;
        }
    }
}

export class Trap extends Phaser.Physics.Arcade.Sprite {
    public type: TrapType;
    public onPlayerHit: (() => void) | null = null;
    public effect: TrapEffect;
    public playerInRange: boolean = false;
    public player: any = null;
    
    private isActive: boolean = true;
    private cooldownTimer: number = 0;
    private cooldownDuration: number = 2000;
    private visualEffect: Phaser.GameObjects.Graphics | null = null;
    private activationAnimation: Phaser.Tweens.Tween | null = null;

    constructor(scene: Phaser.Scene, x: number, y: number, type: TrapType = 'spike') {
        super(scene, x, y, 'trap');
        
        this.type = type;
        this.effect = this.createEffect(type);
        
        scene.add.existing(this);
        scene.physics.add.existing(this);
        
        this.setupTrap();
        this.setupVisuals();
    }

    private setupTrap(): void {
        this.setOrigin(0.5, 0.5);
        this.setScale(GAME_CONFIG.PIXEL_SCALE);
        this.body!.setSize(16, 16);
        this.body!.setOffset(8, 8);
        
        // 设置物理属性
        this.body!.setImmovable(true);
        this.setCollideWorldBounds(true);
        
        // 设置陷阱外观
        this.setTint(this.getColorForType());
        
        // 初始状态为激活
        this.setActiveState(true);
    }

    private getColorForType(): number {
        switch (this.type) {
            case 'spike': return Phaser.Display.Color.HexStringToColor(COLORS.TRAP).color;
            case 'laser': return Phaser.Display.Color.HexStringToColor('#ff0000').color;
            case 'poison': return Phaser.Display.Color.HexStringToColor('#00ff00').color;
            case 'slow': return Phaser.Display.Color.HexStringToColor('#0000ff').color;
            case 'stun': return Phaser.Display.Color.HexStringToColor('#ffff00').color;
            default: return Phaser.Display.Color.HexStringToColor(COLORS.TRAP).color;
        }
    }

    private createEffect(type: TrapType): TrapEffect {
        switch (type) {
            case 'spike': return new SpikeTrapEffect();
            case 'laser': return new LaserTrapEffect();
            case 'poison': return new PoisonTrapEffect();
            case 'slow': return new SlowTrapEffect();
            case 'stun': return new StunTrapEffect();
            default: return new SpikeTrapEffect();
        }
    }

    private setupVisuals(): void {
        this.createBaseVisual();
        this.setupAnimations();
    }

    private createBaseVisual(): void {
        this.visualEffect = this.scene.add.graphics();
        this.updateVisuals();
        
        // 定期更新视觉效果
        this.scene.time.addEvent({
            delay: 100,
            callback: this.updateVisuals,
            callbackScope: this,
            loop: true
        });
    }

    private updateVisuals(): void {
        if (!this.visualEffect) return;
        
        this.visualEffect.clear();
        
        if (!this.isActive) return;
        
        const baseRadius = 20;
        const pulseRadius = baseRadius + Math.sin(this.scene.time.now * 0.008) * 5;
        const alpha = 0.3 + Math.sin(this.scene.time.now * 0.005) * 0.2;
        
        // 绘制危险区域
        this.visualEffect.fillStyle(this.getColorForType(), alpha);
        this.visualEffect.fillCircle(this.x, this.y, pulseRadius);
        
        // 绘制警告边框
        this.visualEffect.lineStyle(2, this.getColorForType(), alpha * 2);
        this.visualEffect.strokeCircle(this.x, this.y, pulseRadius);
        
        // 特殊类型的额外视觉效果
        this.addSpecialVisuals();
        
        // 将视觉效果放在陷阱后面
        this.visualEffect.setDepth(this.depth - 1);
    }

    private addSpecialVisuals(): void {
        if (!this.visualEffect) return;
        
        switch (this.type) {
            case 'spike':
                // 尖刺图案
                for (let i = 0; i < 6; i++) {
                    const angle = (i / 6) * Math.PI * 2;
                    const x = this.x + Math.cos(angle) * 15;
                    const y = this.y + Math.sin(angle) * 15;
                    
                    this.visualEffect.fillStyle(0xffffff, 0.6);
                    this.visualEffect.fillTriangle(x, y - 5, x - 3, y + 3, x + 3, y + 3);
                }
                break;
                
            case 'laser':
                // 激光线
                this.visualEffect.lineStyle(3, 0xff0000, 0.7);
                this.visualEffect.lineBetween(this.x - 25, this.y, this.x + 25, this.y);
                break;
                
            case 'poison':
                // 毒雾效果
                this.visualEffect.fillStyle(0x00ff00, 0.2);
                for (let i = 0; i < 3; i++) {
                    const offsetX = (Math.random() - 0.5) * 20;
                    const offsetY = (Math.random() - 0.5) * 20;
                    const size = 5 + Math.random() * 10;
                    
                    this.visualEffect.fillCircle(
                        this.x + offsetX,
                        this.y + offsetY,
                        size
                    );
                }
                break;
                
            case 'slow':
                // 减速波纹
                for (let i = 1; i <= 3; i++) {
                    const radius = 15 * i;
                    const alpha = 0.3 * (1 - i / 4);
                    
                    this.visualEffect.lineStyle(2, 0x0000ff, alpha);
                    this.visualEffect.strokeCircle(this.x, this.y, radius);
                }
                break;
                
            case 'stun':
                // 眩晕星星
                for (let i = 0; i < 5; i++) {
                    const angle = (i / 5) * Math.PI * 2 + this.scene.time.now * 0.01;
                    const x = this.x + Math.cos(angle) * 20;
                    const y = this.y + Math.sin(angle) * 20;
                    
                    this.visualEffect.fillStyle(0xffff00, 0.8);
                    this.visualEffect.fillCircle(x, y, 2);
                }
                break;
        }
    }

    private setupAnimations(): void {
        // 基础呼吸动画
        this.activationAnimation = this.scene.tweens.add({
            targets: this,
            scaleX: GAME_CONFIG.PIXEL_SCALE * 1.1,
            scaleY: GAME_CONFIG.PIXEL_SCALE * 1.1,
            duration: 1000 + Math.random() * 500,
            yoyo: true,
            repeat: -1,
            ease: 'Sine.easeInOut'
        });
        
        // 旋转动画（不同类型不同速度）
        const rotationSpeed = this.getRotationSpeed();
        this.scene.tweens.add({
            targets: this,
            rotation: Math.PI * 2,
            duration: rotationSpeed,
            repeat: -1,
            ease: 'Linear'
        });
    }

    private getRotationSpeed(): number {
        switch (this.type) {
            case 'laser': return 4000;
            case 'stun': return 2000;
            case 'poison': return 6000;
            case 'slow': return 8000;
            case 'spike':
            default: return 5000;
        }
    }

    update(time: number, delta: number): void {
        this.effect.update(this, time, delta);
        this.updateCooldown(delta);
        this.updateAnimations();
    }

    private updateCooldown(delta: number): void {
        if (this.cooldownTimer > 0) {
            this.cooldownTimer -= delta;
            
            // 冷却期间的视觉效果
            if (this.cooldownTimer > 0) {
                const cooldownAlpha = 0.3 + (this.cooldownTimer / this.cooldownDuration) * 0.7;
                this.setAlpha(cooldownAlpha);
            } else {
                this.setAlpha(1);
            }
        }
    }

    private updateAnimations(): void {
        // 根据状态调整动画
        if (this.playerInRange) {
            // 玩家靠近时的警告动画
            this.setTint(Phaser.Display.Color.HexStringToColor('#ffffff').color);
        } else {
            this.setTint(this.getColorForType());
        }
        
        // 冷却期间的动画
        if (this.cooldownTimer > 0) {
            this.setTint(Phaser.Display.Color.HexStringToColor('#666666').color);
        }
    }

    public activate(player: any): void {
        if (!this.isActive || this.cooldownTimer > 0) return;
        
        this.player = player;
        this.effect.activate(this, player);
        this.cooldownTimer = this.cooldownDuration;
        
        // 触发陷阱触发回调
        if (this.onPlayerHit) {
            this.onPlayerHit();
        }
        
        // 播放陷阱触发音效
        // this.scene.sound.play('trap_trigger', { volume: 0.5 });
    }

    public setActiveState(active: boolean): void {
        this.isActive = active;
        
        if (!active && this.visualEffect) {
            this.visualEffect.clear();
        }
        
        // 重置冷却时间
        if (active) {
            this.cooldownTimer = 0;
        }
    }

    public setPlayerInRange(inRange: boolean): void {
        this.playerInRange = inRange;
    }

    // 公共方法
    public getTrapType(): TrapType {
        return this.type;
    }

    public isTrapActive(): boolean {
        return this.isActive;
    }

    public getPosition(): { x: number, y: number } {
        return { x: this.x, y: this.y };
    }

    public getCooldownRemaining(): number {
        return Math.max(0, this.cooldownTimer);
    }

    // 静态工厂方法
    public static createSpikeTrap(scene: Phaser.Scene, x: number, y: number): Trap {
        return new Trap(scene, x, y, 'spike');
    }

    public static createLaserTrap(scene: Phaser.Scene, x: number, y: number): Trap {
        return new Trap(scene, x, y, 'laser');
    }

    public static createPoisonTrap(scene: Phaser.Scene, x: number, y: number): Trap {
        return new Trap(scene, x, y, 'poison');
    }

    public static createSlowTrap(scene: Phaser.Scene, x: number, y: number): Trap {
        return new Trap(scene, x, y, 'slow');
    }

    public static createStunTrap(scene: Phaser.Scene, x: number, y: number): Trap {
        return new Trap(scene, x, y, 'stun');
    }

    // 清理资源
    public destroy(): void {
        if (this.visualEffect) {
            this.visualEffect.destroy();
        }
        if (this.activationAnimation) {
            this.activationAnimation.stop();
        }
        
        // 确保停用效果
        if (this.effect && 'deactivate' in this.effect) {
            (this.effect as any).deactivate(this);
        }
        
        super.destroy();
    }
}