import Phaser from 'phaser';
import { GAME_CONFIG, COLORS } from '../config/GameConfig';

export class Player extends Phaser.Physics.Arcade.Sprite {
    private cursors!: Phaser.Types.Input.Keyboard.CursorKeys;
    private wasdKeys!: any;
    private speed: number = GAME_CONFIG.PLAYER_SPEED;
    private isInvincible: boolean = false;
    private invincibilityTimer: number = 0;
    private isAttacking: boolean = false;
    private attackCooldown: number = 0;
    private touchControls: boolean = false;
    private touchStartX: number = 0;
    private touchStartY: number = 0;
    private touchDirection: Phaser.Math.Vector2;

    constructor(scene: Phaser.Scene, x: number, y: number) {
        super(scene, x, y, 'player');
        
        scene.add.existing(this);
        scene.physics.add.existing(this);
        
        this.setupPlayer();
        this.setupInput();
        this.touchDirection = new Phaser.Math.Vector2(0, 0);
    }

    private setupPlayer(): void {
        // 设置玩家属性
        this.setOrigin(0.5, 0.5);
        this.setScale(GAME_CONFIG.PIXEL_SCALE);
        this.setCollideWorldBounds(true);
        
        // 设置物理属性
        this.body!.setSize(16, 16);
        this.body!.setOffset(8, 8);
        
        // 添加发光效果
        this.setTint(0xffffff);
    }

    private setupInput(): void {
        // 键盘输入
        this.cursors = this.scene.input.keyboard!.createCursorKeys();
        this.wasdKeys = this.scene.input.keyboard!.addKeys('W,S,A,D');
        
        // 攻击键
        this.scene.input.keyboard!.on('keydown-SPACE', () => {
            this.attack();
        });
        
        // 移动端触摸控制
        if (this.scene.sys.game.device.input.touch) {
            this.setupTouchControls();
        }
    }

    private setupTouchControls(): void {
        this.touchControls = true;
        
        // 简化的触摸控制 - 屏幕左半部分控制移动，右半部分控制攻击
        this.scene.input.on('pointerdown', (pointer: Phaser.Input.Pointer) => {
            const centerX = GAME_CONFIG.WIDTH / 2;
            
            if (pointer.x < centerX) {
                // 左半部分 - 移动控制
                this.startTouchMovement(pointer.x, pointer.y);
            } else {
                // 右半部分 - 攻击
                this.attack();
            }
        });
        
        this.scene.input.on('pointermove', (pointer: Phaser.Input.Pointer) => {
            if (pointer.isDown && pointer.x < GAME_CONFIG.WIDTH / 2) {
                this.updateTouchMovement(pointer.x, pointer.y);
            }
        });
        
        this.scene.input.on('pointerup', () => {
            this.stopTouchMovement();
        });
    }

    public startTouchMovement(x: number, y: number): void {
        this.touchStartX = x;
        this.touchStartY = y;
        this.touchControls = true;
    }

    public updateTouchMovement(x: number, y: number): void {
        if (!this.touchControls) return;
        
        const deltaX = x - this.touchStartX;
        const deltaY = y - this.touchStartY;
        
        // 计算方向向量
        this.touchDirection.set(deltaX, deltaY).normalize();
        
        // 设置移动速度
        const distance = Math.sqrt(deltaX * deltaX + deltaY * deltaY);
        const speedMultiplier = Math.min(distance / 50, 1); // 根据距离调整速度
        
        this.touchDirection.scale(speedMultiplier);
    }

    public stopTouchMovement(): void {
        this.touchControls = false;
        this.touchDirection.set(0, 0);
    }

    update(time: number, delta: number): void {
        this.handleMovement();
        this.updateInvincibility(delta);
        this.updateAttackCooldown(delta);
        this.updateAnimations();
    }

    private handleMovement(): void {
        let velocityX = 0;
        let velocityY = 0;
        
        if (this.touchControls) {
            // 触摸控制
            velocityX = this.touchDirection.x * this.speed;
            velocityY = this.touchDirection.y * this.speed;
        } else {
            // 键盘控制
            if (this.cursors.left.isDown || this.wasdKeys.A.isDown) {
                velocityX = -this.speed;
            } else if (this.cursors.right.isDown || this.wasdKeys.D.isDown) {
                velocityX = this.speed;
            }
            
            if (this.cursors.up.isDown || this.wasdKeys.W.isDown) {
                velocityY = -this.speed;
            } else if (this.cursors.down.isDown || this.wasdKeys.S.isDown) {
                velocityY = this.speed;
            }
        }
        
        // 归一化对角线移动
        if (velocityX !== 0 && velocityY !== 0) {
            velocityX *= 0.707; // 1/√2
            velocityY *= 0.707;
        }
        
        this.setVelocity(velocityX, velocityY);
    }

    private updateInvincibility(delta: number): void {
        if (this.isInvincible) {
            this.invincibilityTimer -= delta;
            
            // 无敌闪烁效果
            const flashDuration = 100; // 闪烁间隔
            const flashTime = this.invincibilityTimer % (flashDuration * 2);
            this.setAlpha(flashTime < flashDuration ? 0.5 : 1);
            
            if (this.invincibilityTimer <= 0) {
                this.isInvincible = false;
                this.setAlpha(1);
            }
        }
    }

    private updateAttackCooldown(delta: number): void {
        if (this.attackCooldown > 0) {
            this.attackCooldown -= delta;
        }
    }

    private updateAnimations(): void {
        const velocity = this.body!.velocity;
        
        // 根据移动方向设置旋转（可选）
        if (velocity.length() > 0) {
            const angle = Math.atan2(velocity.y, velocity.x) * 180 / Math.PI;
            // this.setRotation(angle * Math.PI / 180); // 如果需要角色朝向移动方向
        }
        
        // 无敌状态下的视觉效果
        if (this.isInvincible) {
            this.setTint(0x00ffff); // 青色光环
        } else {
            this.setTint(0xffffff); // 正常颜色
        }
    }

    public attack(): void {
        if (this.isAttacking || this.attackCooldown > 0) return;
        
        this.isAttacking = true;
        this.attackCooldown = 500; // 500ms冷却时间
        
        // 创建攻击效果
        this.createAttackEffect();
        
        // 播放攻击音效
        // this.scene.sound.play('attack');
        
        // 重置攻击状态
        this.scene.time.delayedCall(200, () => {
            this.isAttacking = false;
        });
    }

    private createAttackEffect(): void {
        const attackRange = 50;
        const playerPos = new Phaser.Math.Vector2(this.x, this.y);
        
        // 获取附近的敌人（需要在游戏场景中实现）
        // 这里创建视觉效果
        const attackEffect = this.scene.add.circle(this.x, this.y, 20, 0xffffff, 0.8);
        
        this.scene.tweens.add({
            targets: attackEffect,
            scaleX: 2.5,
            scaleY: 2.5,
            alpha: 0,
            duration: 300,
            ease: 'Power2',
            onComplete: () => {
                attackEffect.destroy();
            }
        });
        
        // 检查攻击范围内的敌人
        // 这需要在游戏场景中实现，通过事件或直接的敌人管理
        this.scene.events.emit('playerAttack', {
            x: this.x,
            y: this.y,
            range: attackRange
        });
    }

    public setInvincible(duration: number = GAME_CONFIG.INVINCIBILITY_TIME): void {
        this.isInvincible = true;
        this.invincibilityTimer = duration;
    }

    public isInvincibleCheck(): boolean {
        return this.isInvincible;
    }

    public setWallCollision(walls: Phaser.GameObjects.Group): void {
        this.scene.physics.add.collider(this, walls);
    }

    public getPosition(): { x: number, y: number } {
        return { x: this.x, y: this.y };
    }

    public getDirection(): Phaser.Math.Vector2 {
        const velocity = this.body!.velocity;
        if (velocity.length() === 0) {
            return new Phaser.Math.Vector2(0, -1); // 默认向上
        }
        return new Phaser.Math.Vector2(velocity.x, velocity.y).normalize();
    }



    // 重置玩家状态
    public reset(): void {
        this.isInvincible = false;
        this.invincibilityTimer = 0;
        this.isAttacking = false;
        this.attackCooldown = 0;
        this.setAlpha(1);
        this.setTint(0xffffff);
        this.setVelocity(0, 0);
    }

    // 获取玩家边界
    public getBounds(): Phaser.Geom.Rectangle {
        return this.getBounds();
    }

    // 检查玩家是否在指定区域内
    public isInBounds(x: number, y: number, width: number, height: number): boolean {
        const playerBounds = this.getBounds();
        const area = new Phaser.Geom.Rectangle(x, y, width, height);
        return Phaser.Geom.Rectangle.Overlaps(playerBounds, area);
    }
}