import Phaser from 'phaser';
import { GAME_CONFIG, COLORS } from '../config/GameConfig';

export type EnemyType = 'chaser' | 'ambusher' | 'random' | 'boss';

interface EnemyBehavior {
    update(enemy: Enemy, time: number, delta: number): void;
    getTargetPosition(enemy: Enemy): { x: number, y: number };
}

// 追踪型敌人 - 直接追击玩家
class ChaserBehavior implements EnemyBehavior {
    update(enemy: Enemy, time: number, delta: number): void {
        if (!enemy.target) return;
        
        const targetPos = this.getTargetPosition(enemy);
        const enemyPos = enemy.getPosition();
        
        const direction = new Phaser.Math.Vector2(
            targetPos.x - enemyPos.x,
            targetPos.y - enemyPos.y
        ).normalize();
        
        enemy.setVelocity(direction.x * enemy.speed, direction.y * enemy.speed);
    }
    
    getTargetPosition(enemy: Enemy): { x: number, y: number } {
        return enemy.target!.getPosition();
    }
}

// 伏击型敌人 - 预测玩家路径
class AmbusherBehavior implements EnemyBehavior {
    private predictionDistance: number = 80;
    
    update(enemy: Enemy, time: number, delta: number): void {
        if (!enemy.target) return;
        
        const targetPos = this.getTargetPosition(enemy);
        const enemyPos = enemy.getPosition();
        
        const direction = new Phaser.Math.Vector2(
            targetPos.x - enemyPos.x,
            targetPos.y - enemyPos.y
        ).normalize();
        
        enemy.setVelocity(direction.x * enemy.speed, direction.y * enemy.speed);
    }
    
    getTargetPosition(enemy: Enemy): { x: number, y: number } {
        if (!enemy.target) return enemy.getPosition();
        
        const playerPos = enemy.target.getPosition();
        const playerDirection = enemy.target.getDirection();
        
        // 预测玩家未来位置
        return {
            x: playerPos.x + playerDirection.x * this.predictionDistance,
            y: playerPos.y + playerDirection.y * this.predictionDistance
        };
    }
}

// 随机型敌人 - 随机移动，偶尔追击
class RandomBehavior implements EnemyBehavior {
    private changeDirectionTimer: number = 0;
    private currentDirection: Phaser.Math.Vector2;
    private chaseChance: number = 0.3;
    private isChasing: boolean = false;
    
    constructor() {
        this.currentDirection = new Phaser.Math.Vector2(
            Math.random() * 2 - 1,
            Math.random() * 2 - 1
        ).normalize();
    }
    
    update(enemy: Enemy, time: number, delta: number): void {
        this.changeDirectionTimer -= delta;
        
        if (this.changeDirectionTimer <= 0) {
            this.changeDirection(enemy);
            this.changeDirectionTimer = 1000 + Math.random() * 2000; // 1-3秒改变方向
        }
        
        if (this.isChasing && enemy.target) {
            const targetPos = this.getTargetPosition(enemy);
            const enemyPos = enemy.getPosition();
            
            const direction = new Phaser.Math.Vector2(
                targetPos.x - enemyPos.x,
                targetPos.y - enemyPos.y
            ).normalize();
            
            enemy.setVelocity(direction.x * enemy.speed, direction.y * enemy.speed);
        } else {
            enemy.setVelocity(
                this.currentDirection.x * enemy.speed,
                this.currentDirection.y * enemy.speed
            );
        }
    }
    
    changeDirection(enemy: Enemy): void {
        // 随机决定是否追击玩家
        if (enemy.target && Math.random() < this.chaseChance) {
            this.isChasing = true;
        } else {
            this.isChasing = false;
            this.currentDirection.set(
                Math.random() * 2 - 1,
                Math.random() * 2 - 1
            ).normalize();
        }
    }
    
    getTargetPosition(enemy: Enemy): { x: number, y: number } {
        return enemy.target!.getPosition();
    }
}

// Boss型敌人 - 结合多种行为模式
class BossBehavior implements EnemyBehavior {
    private behaviors: EnemyBehavior[];
    private currentBehaviorIndex: number = 0;
    private behaviorTimer: number = 0;
    private behaviorDuration: number = 3000; // 3秒切换一次行为
    
    constructor() {
        this.behaviors = [
            new ChaserBehavior(),
            new AmbusherBehavior(),
            new RandomBehavior()
        ];
    }
    
    update(enemy: Enemy, time: number, delta: number): void {
        this.behaviorTimer -= delta;
        
        if (this.behaviorTimer <= 0) {
            this.switchBehavior();
            this.behaviorTimer = this.behaviorDuration;
        }
        
        this.behaviors[this.currentBehaviorIndex].update(enemy, time, delta);
    }
    
    switchBehavior(): void {
        this.currentBehaviorIndex = (this.currentBehaviorIndex + 1) % this.behaviors.length;
    }
    
    getTargetPosition(enemy: Enemy): { x: number, y: number } {
        return this.behaviors[this.currentBehaviorIndex].getTargetPosition(enemy);
    }
}

export class Enemy extends Phaser.Physics.Arcade.Sprite {
    public type: EnemyType;
    public target: Player | null = null;
    public speed: number;
    private behavior: EnemyBehavior;
    private walls: Phaser.GameObjects.Group | null = null;
    private isStunned: boolean = false;
    private stunTimer: number = 0;
    private health: number = 1;
    private maxHealth: number = 1;
    private pathfindingTimer: number = 0;
    private lastKnownTargetPos: { x: number, y: number } | null = null;

    constructor(scene: Phaser.Scene, x: number, y: number, type: EnemyType) {
        super(scene, x, y, 'enemy');
        
        this.type = type;
        this.speed = this.getSpeedForType(type);
        this.behavior = this.createBehavior(type);
        
        scene.add.existing(this);
        scene.physics.add.existing(this);
        
        this.setupEnemy();
    }

    private setupEnemy(): void {
        this.setOrigin(0.5, 0.5);
        this.setScale(GAME_CONFIG.PIXEL_SCALE);
        this.setCollideWorldBounds(true);
        
        // 设置物理属性
        this.body!.setSize(16, 16);
        this.body!.setOffset(8, 8);
        
        // 根据类型设置外观
        this.setTint(this.getColorForType(this.type));
        
        // 设置生命值
        this.health = this.maxHealth = this.getHealthForType(this.type);
    }

    private getSpeedForType(type: EnemyType): number {
        const baseSpeed = GAME_CONFIG.ENEMY_SPEED;
        switch (type) {
            case 'chaser': return baseSpeed;
            case 'ambusher': return baseSpeed * 0.9;
            case 'random': return baseSpeed * 0.8;
            case 'boss': return baseSpeed * 1.2;
            default: return baseSpeed;
        }
    }

    private getHealthForType(type: EnemyType): number {
        switch (type) {
            case 'boss': return 3;
            case 'chaser':
            case 'ambusher':
            case 'random':
            default: return 1;
        }
    }

    private getColorForType(type: EnemyType): number {
        switch (type) {
            case 'chaser': return Phaser.Display.Color.HexStringToColor(COLORS.ENEMY).color;
            case 'ambusher': return Phaser.Display.Color.HexStringToColor('#9b59b6').color;
            case 'random': return Phaser.Display.Color.HexStringToColor('#95a5a6').color;
            case 'boss': return Phaser.Display.Color.HexStringToColor('#8e44ad').color;
            default: return Phaser.Display.Color.HexStringToColor(COLORS.ENEMY).color;
        }
    }

    private createBehavior(type: EnemyType): EnemyBehavior {
        switch (type) {
            case 'chaser': return new ChaserBehavior();
            case 'ambusher': return new AmbusherBehavior();
            case 'random': return new RandomBehavior();
            case 'boss': return new BossBehavior();
            default: return new ChaserBehavior();
        }
    }

    public setTarget(target: Player): void {
        this.target = target;
    }

    public setWallCollision(walls: Phaser.GameObjects.Group): void {
        this.walls = walls;
        this.scene.physics.add.collider(this, walls);
    }

    update(time: number, delta: number): void {
        if (this.isStunned) {
            this.updateStun(delta);
            return;
        }
        
        this.pathfindingTimer -= delta;
        
        // 更新行为
        this.behavior.update(this, time, delta);
        
        // 简单的路径寻找 - 避免卡在角落
        if (this.pathfindingTimer <= 0) {
            this.avoidObstacles();
            this.pathfindingTimer = 500; // 每500ms检查一次
        }
        
        this.updateAnimations();
        this.updateHealthDisplay();
    }

    private avoidObstacles(): void {
        const velocity = this.body!.velocity;
        
        // 如果移动速度很慢，可能是被卡住了
        if (velocity.length() < this.speed * 0.3) {
            // 尝试随机改变方向
            const randomAngle = Math.random() * Math.PI * 2;
            const newDirection = new Phaser.Math.Vector2(
                Math.cos(randomAngle),
                Math.sin(randomAngle)
            );
            
            this.setVelocity(newDirection.x * this.speed, newDirection.y * this.speed);
        }
    }

    private updateAnimations(): void {
        const velocity = this.body!.velocity;
        
        // 根据移动方向设置旋转
        if (velocity.length() > 0) {
            const angle = Math.atan2(velocity.y, velocity.x) * 180 / Math.PI;
            // this.setRotation(angle * Math.PI / 180); // 如果需要敌人朝向移动方向
        }
        
        // 眩晕效果
        if (this.isStunned) {
            this.setTint(0xffff00); // 黄色表示眩晕
            this.setAlpha(0.7);
        } else {
            this.setTint(this.getColorForType(this.type));
            this.setAlpha(1);
        }
        
        // Boss特殊效果
        if (this.type === 'boss') {
            this.setScale(GAME_CONFIG.PIXEL_SCALE * (1 + Math.sin(this.scene.time.now * 0.005) * 0.1));
        }
    }

    private updateHealthDisplay(): void {
        // 如果需要显示生命值条，可以在这里实现
        if (this.health < this.maxHealth) {
            // 显示伤害效果
            if (this.scene.time.now % 500 < 250) {
                this.setTint(0xff0000);
            } else {
                this.setTint(this.getColorForType(this.type));
            }
        }
    }

    private updateStun(delta: number): void {
        this.stunTimer -= delta;
        this.setVelocity(0, 0);
        
        if (this.stunTimer <= 0) {
            this.isStunned = false;
        }
    }

    public takeDamage(damage: number = 1): void {
        this.health -= damage;
        
        if (this.health <= 0) {
            this.die();
        } else {
            // 短暂眩晕
            this.stun(500);
            // 播放受伤音效
            // this.scene.sound.play('enemy_hurt');
        }
    }

    public stun(duration: number): void {
        this.isStunned = true;
        this.stunTimer = duration;
    }

    private die(): void {
        // 播放死亡动画
        this.playDeathAnimation();
        
        // 播放死亡音效
        // this.scene.sound.play('enemy_die');
        
        // 奖励分数
        const scoreValue = this.getScoreValue();
        this.scene.events.emit('enemyKilled', scoreValue);
        
        // 延迟销毁
        this.scene.time.delayedCall(500, () => {
            this.destroy();
        });
    }

    private playDeathAnimation(): void {
        // 缩放和淡出动画
        this.scene.tweens.add({
            targets: this,
            scaleX: 0,
            scaleY: 0,
            alpha: 0,
            duration: 500,
            ease: 'Power2'
        });
        
        // 爆炸效果
        const explosion = this.scene.add.circle(this.x, this.y, 10, 0xffffff, 0.8);
        
        this.scene.tweens.add({
            targets: explosion,
            scaleX: 3,
            scaleY: 3,
            alpha: 0,
            duration: 300,
            ease: 'Power2',
            onComplete: () => {
                explosion.destroy();
            }
        });
    }

    private getScoreValue(): number {
        switch (this.type) {
            case 'boss': return 500;
            case 'chaser': return 100;
            case 'ambusher': return 150;
            case 'random': return 75;
            default: return 100;
        }
    }

    // 公共方法
    public getPosition(): { x: number, y: number } {
        return { x: this.x, y: this.y };
    }

    public getDistanceToTarget(): number {
        if (!this.target) return Infinity;
        
        const targetPos = this.target.getPosition();
        const enemyPos = this.getPosition();
        
        return Phaser.Math.Distance.Between(
            targetPos.x, targetPos.y,
            enemyPos.x, enemyPos.y
        );
    }

    public isAlive(): boolean {
        return this.health > 0;
    }

    public getType(): EnemyType {
        return this.type;
    }

    // 重置敌人状态
    public reset(): void {
        this.health = this.maxHealth;
        this.isStunned = false;
        this.stunTimer = 0;
        this.setAlpha(1);
        this.setTint(this.getColorForType(this.type));
    }
}