import Phaser from 'phaser';
import { GAME_CONFIG, DIFFICULTY_SETTINGS } from '../config/GameConfig';

export interface LevelData {
    level: number;
    name: string;
    width: number;
    height: number;
    walls: Array<{ x: number, y: number }>;
    playerStart: { x: number, y: number };
    enemyPositions: Array<{ x: number, y: number }>;
    collectiblePositions: Array<{ x: number, y: number }>;
    trapPositions: Array<{ x: number, y: number }>;
    exitPosition: { x: number, y: number };
    difficulty: keyof typeof DIFFICULTY_SETTINGS;
    timeLimit?: number;
    specialRules?: string[];
}

export class LevelManager {
    private scene: Phaser.Scene;
    private levels: Map<number, LevelData>;
    private currentLevel: number = 1;

    constructor(scene: Phaser.Scene) {
        this.scene = scene;
        this.levels = new Map();
        this.initializeLevels();
    }

    private initializeLevels(): void {
        // 关卡1 - 超简单迷宫，新手教学
        this.levels.set(1, {
            level: 1,
            name: '新手村',
            width: 16,  // 减小地图尺寸
            height: 16,
            walls: this.generateMazeWalls(16, 16, 0.2),  // 降低迷宫复杂度
            playerStart: { x: 80, y: 80 },
            enemyPositions: [
                { x: 400, y: 300 }  // 只有1个敌人，位置较远
            ],
            collectiblePositions: [
                { x: 150, y: 120 },   // 第一个在近处，容易找到
                { x: 300, y: 200 },   // 第二个稍远
                { x: 450, y: 150 },   // 第三个在敌人附近
                { x: 250, y: 350 },   // 第四个在安全区域
                { x: 500, y: 400 }    // 第五个在出口附近
            ],
            trapPositions: [],  // 第一关没有陷阱
            exitPosition: { x: 550, y: 450 },  // 出口位置合理
            difficulty: 'EASY',
            timeLimit: 300,  // 增加到5分钟，给新手足够时间
            specialRules: ['tutorial', 'no_boss', 'no_traps']
        });

        // 关卡2 - 中等难度，加入更多机制
        this.levels.set(2, {
            level: 2,
            name: '迷雾森林',
            width: 24,
            height: 24,
            walls: this.generateMazeWalls(24, 24, 0.4),
            playerStart: { x: 80, y: 80 },
            enemyPositions: [
                { x: 250, y: 180 },
                { x: 450, y: 320 },
                { x: 650, y: 480 },
                { x: 350, y: 580 }
            ],
            collectiblePositions: [
                { x: 180, y: 130 },
                { x: 380, y: 280 },
                { x: 580, y: 380 },
                { x: 280, y: 480 },
                { x: 480, y: 580 },
                { x: 680, y: 180 },
                { x: 130, y: 380 },
                { x: 580, y: 130 },
                { x: 180, y: 580 },
                { x: 730, y: 480 },
                { x: 330, y: 230 },
                { x: 530, y: 430 },
                { x: 230, y: 530 },
                { x: 630, y: 280 },
                { x: 130, y: 480 }
            ],
            trapPositions: [
                { x: 200, y: 200 },
                { x: 400, y: 300 },
                { x: 600, y: 200 },
                { x: 300, y: 500 },
                { x: 500, y: 500 }
            ],
            exitPosition: { x: 720, y: 720 },
            difficulty: 'MEDIUM',
            timeLimit: 180, // 3分钟
            specialRules: ['fog_of_war', 'random_events']
        });

        // 关卡3 - 高难度，Boss战
        this.levels.set(3, {
            level: 3,
            name: '暗影城堡',
            width: 28,
            height: 28,
            walls: this.generateCastleWalls(),
            playerStart: { x: 60, y: 60 },
            enemyPositions: [
                { x: 200, y: 150 },
                { x: 400, y: 250 },
                { x: 600, y: 350 },
                { x: 300, y: 450 },
                { x: 500, y: 550 },
                { x: 700, y: 150 },
                { x: 150, y: 350 },
                { x: 650, y: 250 }
            ],
            collectiblePositions: [
                { x: 160, y: 110 },
                { x: 360, y: 210 },
                { x: 560, y: 310 },
                { x: 260, y: 410 },
                { x: 460, y: 510 },
                { x: 660, y: 110 },
                { x: 110, y: 310 },
                { x: 610, y: 210 },
                { x: 210, y: 510 },
                { x: 710, y: 410 },
                { x: 310, y: 160 },
                { x: 510, y: 260 },
                { x: 410, y: 460 },
                { x: 610, y: 360 },
                { x: 110, y: 460 },
                { x: 760, y: 210 },
                { x: 160, y: 560 },
                { x: 560, y: 110 },
                { x: 360, y: 560 },
                { x: 760, y: 510 }
            ],
            trapPositions: [
                { x: 180, y: 180 },
                { x: 380, y: 280 },
                { x: 580, y: 180 },
                { x: 280, y: 480 },
                { x: 480, y: 480 },
                { x: 680, y: 380 },
                { x: 180, y: 380 },
                { x: 580, y: 580 }
            ],
            exitPosition: { x: 740, y: 740 },
            difficulty: 'HARD',
            timeLimit: 240, // 4分钟
            specialRules: ['boss_battle', 'time_pressure', 'limited_lives']
        });
    }

    private generateMazeWalls(width: number, height: number, density: number): Array<{ x: number, y: number }> {
        const walls: Array<{ x: number, y: number }> = [];
        const tileSize = GAME_CONFIG.TILE_SIZE;
        
        // 外围墙壁
        for (let x = 0; x < width; x++) {
            walls.push({ x: x * tileSize, y: 0 });
            walls.push({ x: x * tileSize, y: (height - 1) * tileSize });
        }
        
        for (let y = 1; y < height - 1; y++) {
            walls.push({ x: 0, y: y * tileSize });
            walls.push({ x: (width - 1) * tileSize, y: y * tileSize });
        }

        // 内部迷宫墙壁
        for (let x = 2; x < width - 2; x += 2) {
            for (let y = 2; y < height - 2; y += 2) {
                if (Math.random() < density) {
                    walls.push({ x: x * tileSize, y: y * tileSize });
                    
                    // 随机添加相邻墙壁
                    if (Math.random() < 0.7) {
                        const direction = Math.floor(Math.random() * 4);
                        let newX = x;
                        let newY = y;
                        
                        switch (direction) {
                            case 0: newX++; break;
                            case 1: newX--; break;
                            case 2: newY++; break;
                            case 3: newY--; break;
                        }
                        
                        if (newX > 1 && newX < width - 2 && newY > 1 && newY < height - 2) {
                            walls.push({ x: newX * tileSize, y: newY * tileSize });
                        }
                    }
                }
            }
        }

        return walls;
    }

    private generateCastleWalls(): Array<{ x: number, y: number }> {
        const walls: Array<{ x: number, y: number }> = [];
        const tileSize = GAME_CONFIG.TILE_SIZE;
        
        // 城堡外墙
        for (let x = 0; x < 28; x++) {
            walls.push({ x: x * tileSize, y: 0 });
            walls.push({ x: x * tileSize, y: 27 * tileSize });
        }
        
        for (let y = 1; y < 27; y++) {
            walls.push({ x: 0, y: y * tileSize });
            walls.push({ x: 27 * tileSize, y: y * tileSize });
        }

        // 内部城堡结构
        // 内墙
        for (let x = 8; x < 20; x++) {
            walls.push({ x: x * tileSize, y: 8 * tileSize });
            walls.push({ x: x * tileSize, y: 19 * tileSize });
        }
        
        for (let y = 9; y < 19; y++) {
            walls.push({ x: 8 * tileSize, y: y * tileSize });
            walls.push({ x: 19 * tileSize, y: y * tileSize });
        }

        // 房间分隔
        for (let x = 10; x < 18; x++) {
            if (x % 2 === 0) {
                walls.push({ x: x * tileSize, y: 12 * tileSize });
                walls.push({ x: x * tileSize, y: 15 * tileSize });
            }
        }

        for (let y = 10; y < 18; y++) {
            if (y % 2 === 0) {
                walls.push({ x: 12 * tileSize, y: y * tileSize });
                walls.push({ x: 15 * tileSize, y: y * tileSize });
            }
        }

        return walls;
    }

    public getLevelData(level: number): LevelData {
        const levelData = this.levels.get(level);
        if (!levelData) {
            throw new Error(`Level ${level} not found`);
        }
        return levelData;
    }

    public getCurrentLevel(): number {
        return this.currentLevel;
    }

    public setCurrentLevel(level: number): void {
        this.currentLevel = level;
    }

    public getTotalLevels(): number {
        return this.levels.size;
    }

    public hasNextLevel(): boolean {
        return this.currentLevel < this.levels.size;
    }

    public getNextLevel(): number {
        return this.currentLevel + 1;
    }

    public generateRandomCollectibles(count: number, bounds: { minX: number, minY: number, maxX: number, maxY: number }): Array<{ x: number, y: number }> {
        const positions: Array<{ x: number, y: number }> = [];
        const tileSize = GAME_CONFIG.TILE_SIZE;
        
        for (let i = 0; i < count; i++) {
            let x, y;
            let attempts = 0;
            
            do {
                x = Math.random() * (bounds.maxX - bounds.minX) + bounds.minX;
                y = Math.random() * (bounds.maxY - bounds.minY) + bounds.minY;
                
                // 对齐到网格
                x = Math.floor(x / tileSize) * tileSize + tileSize / 2;
                y = Math.floor(y / tileSize) * tileSize + tileSize / 2;
                
                attempts++;
            } while (attempts < 10 && this.isTooClose(x, y, positions, tileSize * 3));
            
            positions.push({ x, y });
        }
        
        return positions;
    }

    private isTooClose(x: number, y: number, positions: Array<{ x: number, y: number }>, minDistance: number): boolean {
        for (const pos of positions) {
            const distance = Phaser.Math.Distance.Between(x, y, pos.x, pos.y);
            if (distance < minDistance) {
                return true;
            }
        }
        return false;
    }

    public isValidPosition(x: number, y: number, walls: Array<{ x: number, y: number }>): boolean {
        // 检查是否与墙壁重叠
        for (const wall of walls) {
            if (Math.abs(x - wall.x) < GAME_CONFIG.TILE_SIZE && Math.abs(y - wall.y) < GAME_CONFIG.TILE_SIZE) {
                return false;
            }
        }
        
        // 检查边界
        return x >= GAME_CONFIG.TILE_SIZE && 
               x <= GAME_CONFIG.WIDTH - GAME_CONFIG.TILE_SIZE && 
               y >= GAME_CONFIG.TILE_SIZE && 
               y <= GAME_CONFIG.HEIGHT - GAME_CONFIG.TILE_SIZE;
    }

    public getDifficultyForLevel(level: number): keyof typeof DIFFICULTY_SETTINGS {
        switch (level) {
            case 1: return 'EASY';
            case 2: return 'MEDIUM';
            case 3: return 'HARD';
            default: return 'MEDIUM';
        }
    }

    public getLevelProgress(currentLevel: number): number {
        return (currentLevel - 1) / this.levels.size;
    }

    public getLevelName(level: number): string {
        const levelData = this.levels.get(level);
        return levelData ? levelData.name : `Level ${level}`;
    }

    public getLevelTimeLimit(level: number): number | undefined {
        const levelData = this.levels.get(level);
        return levelData ? levelData.timeLimit : undefined;
    }

    public hasSpecialRule(level: number, rule: string): boolean {
        const levelData = this.levels.get(level);
        return levelData ? levelData.specialRules?.includes(rule) || false : false;
    }
}