import Phaser from 'phaser';
import { GameConfig } from './config/GameConfig';
import { MenuScene } from './scenes/MenuScene';
import { GameScene } from './scenes/GameScene';
import { GameOverScene } from './scenes/GameOverScene';

class Game {
    private phaserGame: Phaser.Game;

    constructor() {
        this.phaserGame = new Phaser.Game(GameConfig);
        this.setupEventListeners();
    }

    private setupEventListeners(): void {
        // 窗口大小改变时重新调整游戏大小
        window.addEventListener('resize', () => {
            this.resizeGame();
        });

        // 页面可见性改变时暂停/恢复游戏
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                this.phaserGame.scene.scenes.forEach(scene => {
                    if (scene.scene.isActive()) {
                        scene.scene.pause();
                    }
                });
            } else {
                this.phaserGame.scene.scenes.forEach(scene => {
                    if (scene.scene.isPaused()) {
                        scene.scene.resume();
                    }
                });
            }
        });
    }

    private resizeGame(): void {
        const parent = this.phaserGame.canvas.parentElement;
        if (!parent) return;

        const maxWidth = window.innerWidth;
        const maxHeight = window.innerHeight;
        const scale = Math.min(maxWidth / GameConfig.width, maxHeight / GameConfig.height);

        const newWidth = GameConfig.width * scale;
        const newHeight = GameConfig.height * scale;

        this.phaserGame.scale.resize(newWidth, newHeight);
        this.phaserGame.scale.setGameSize(newWidth, newHeight);
    }
}

// 游戏配置
export const GameConfig: Phaser.Types.Core.GameConfig = {
    type: Phaser.AUTO,
    parent: 'phaser-game',
    width: 800,
    height: 800,
    backgroundColor: '#1a1a2e',
    pixelArt: true,
    antialias: false,
    roundPixels: true,
    scale: {
        mode: Phaser.Scale.FIT,
        autoCenter: Phaser.Scale.CENTER_BOTH,
        width: 800,
        height: 800
    },
    physics: {
        default: 'arcade',
        arcade: {
            gravity: { y: 0 },
            debug: false
        }
    },
    scene: [MenuScene, GameScene, GameOverScene],
    audio: {
        disableWebAudio: false
    }
};

// 启动游戏
new Game();