from __future__ import annotations

from datetime import datetime, date

from enum import Enum
from typing import Optional

from sqlalchemy import Date, DateTime, Enum as SAEnum, Float, String
from sqlalchemy.orm import DeclarativeBase, Mapped, mapped_column


class Base(DeclarativeBase):
    pass


class Account(Base):
    __tablename__ = "accounts"

    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    name: Mapped[str] = mapped_column(String(64), unique=True, nullable=False)
    currency: Mapped[str] = mapped_column(String(8), default="CNY")
    total_equity: Mapped[float] = mapped_column(Float, default=0.0)
    cash_available: Mapped[float] = mapped_column(Float, default=0.0)
    cash_frozen: Mapped[float] = mapped_column(Float, default=0.0)
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=datetime.utcnow)
    updated_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True), default=datetime.utcnow, onupdate=datetime.utcnow
    )


class StrategyAccount(Base):
    __tablename__ = "strategy_accounts"

    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    account_id: Mapped[int] = mapped_column(index=True, nullable=False)
    strategy_name: Mapped[str] = mapped_column(String(64), nullable=False)
    total_equity: Mapped[float] = mapped_column(Float, default=0.0)
    cash_available: Mapped[float] = mapped_column(Float, default=0.0)
    cash_frozen: Mapped[float] = mapped_column(Float, default=0.0)
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=datetime.utcnow)
    updated_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True), default=datetime.utcnow, onupdate=datetime.utcnow
    )


class Position(Base):
    __tablename__ = "positions"

    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    strategy_account_id: Mapped[int] = mapped_column(index=True, nullable=False)
    ts_code: Mapped[str] = mapped_column(String(12), index=True, nullable=False)
    quantity: Mapped[float] = mapped_column(Float, default=0.0)
    frozen_quantity: Mapped[float] = mapped_column(Float, default=0.0)
    avg_cost: Mapped[float] = mapped_column(Float, default=0.0)
    market_value: Mapped[float] = mapped_column(Float, default=0.0)
    unrealized_pnl: Mapped[float] = mapped_column(Float, default=0.0)
    realized_pnl: Mapped[float] = mapped_column(Float, default=0.0)
    updated_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True), default=datetime.utcnow, onupdate=datetime.utcnow
    )


class OrderStatus(str, Enum):
    NEW = "NEW"
    PARTIALLY_FILLED = "PARTIALLY_FILLED"
    FILLED = "FILLED"
    CANCELLED = "CANCELLED"
    REJECTED = "REJECTED"


class OrderSide(str, Enum):
    BUY = "BUY"
    SELL = "SELL"


class OrderType(str, Enum):
    MARKET = "MARKET"
    LIMIT = "LIMIT"


class Order(Base):
    __tablename__ = "orders"

    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    strategy_account_id: Mapped[int] = mapped_column(index=True, nullable=False)
    ts_code: Mapped[str] = mapped_column(String(12), nullable=False)
    side: Mapped[OrderSide] = mapped_column(SAEnum(OrderSide), nullable=False)
    order_type: Mapped[OrderType] = mapped_column(SAEnum(OrderType), nullable=False)
    quantity: Mapped[float] = mapped_column(Float, nullable=False)
    price: Mapped[Optional[float]] = mapped_column(Float)
    status: Mapped[OrderStatus] = mapped_column(SAEnum(OrderStatus), default=OrderStatus.NEW)
    filled_quantity: Mapped[float] = mapped_column(Float, default=0.0)
    avg_filled_price: Mapped[float] = mapped_column(Float, default=0.0)
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=datetime.utcnow)
    updated_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True), default=datetime.utcnow, onupdate=datetime.utcnow
    )


class Trade(Base):
    __tablename__ = "trades"

    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    order_id: Mapped[int] = mapped_column(index=True, nullable=False)
    ts_code: Mapped[str] = mapped_column(String(12), nullable=False)
    side: Mapped[OrderSide] = mapped_column(SAEnum(OrderSide), nullable=False)
    trade_time: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=datetime.utcnow)
    price: Mapped[float] = mapped_column(Float, nullable=False)
    quantity: Mapped[float] = mapped_column(Float, nullable=False)
    amount: Mapped[float] = mapped_column(Float, nullable=False)


class CashFlowType(str, Enum):
    DEPOSIT = "DEPOSIT"
    WITHDRAW = "WITHDRAW"
    FEES = "FEES"
    TRADE = "TRADE"
    INTEREST = "INTEREST"


class CashFlow(Base):
    __tablename__ = "cash_flows"

    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    strategy_account_id: Mapped[int] = mapped_column(index=True, nullable=False)
    flow_type: Mapped[CashFlowType] = mapped_column(SAEnum(CashFlowType), nullable=False)
    amount: Mapped[float] = mapped_column(Float, nullable=False)
    balance: Mapped[float] = mapped_column(Float, nullable=False)
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=datetime.utcnow)
    remark: Mapped[Optional[str]] = mapped_column(String(256))


class DailyNav(Base):
    __tablename__ = "daily_nav"

    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    strategy_account_id: Mapped[int] = mapped_column(index=True, nullable=False)
    date: Mapped[date] = mapped_column(Date, nullable=False)
    nav: Mapped[float] = mapped_column(Float, nullable=False)
    total_equity: Mapped[float] = mapped_column(Float, nullable=False)
    cash: Mapped[float] = mapped_column(Float, nullable=False)
    pnl_daily: Mapped[float] = mapped_column(Float, default=0.0)
    pnl_total: Mapped[float] = mapped_column(Float, default=0.0)
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=datetime.utcnow)


__all__ = [
    "Base",
    "Account",
    "StrategyAccount",
    "Position",
    "Order",
    "Trade",
    "CashFlow",
    "DailyNav",
    "OrderStatus",
    "OrderSide",
    "OrderType",
    "CashFlowType",
]
