import React, { useEffect, useRef } from 'react';

const ChatArea = ({ messages, onTriggerConversation, connectionStatus }) => {
  const messagesEndRef = useRef(null);
  const chatMessagesRef = useRef(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  const getAgentAvatarColor = (role) => {
    const colors = {
      'DataAnalyst': '#3b82f6',
      'TechnicalAnalyst': '#8b5cf6',
      'RiskManager': '#ef4444',
      'SentimentAnalyst': '#f59e0b',
      'MacroEconomist': '#10b981',
      'TradingStrategist': '#f97316',
      'Coordinator': '#6366f1'
    };
    return colors[role] || '#6b7280';
  };

  const getAgentInitials = (name) => {
    if (!name) return '??';
    return name.split(' ').map(n => n[0]).join('').toUpperCase();
  };

  const formatTimestamp = (timestamp) => {
    if (!timestamp) return '';
    try {
      const date = new Date(timestamp);
      return date.toLocaleTimeString('en-US', {
        hour12: false,
        hour: '2-digit',
        minute: '2-digit'
      });
    } catch (error) {
      return '';
    }
  };

  const getMessageTypeLabel = (type) => {
    const labels = {
      'normal': '',
      'market_volatility': 'Market Alert',
      'important_news': 'News Alert',
      'interruption': 'Interrupting',
      'random': 'Discussion',
      'meeting': 'Meeting',
      'hourly_meeting': 'Hourly Review',
      'manual': 'Manual Trigger',
      'api_trigger': 'Triggered'
    };
    return labels[type] || type;
  };

  const handleTriggerClick = (topics) => {
    if (connectionStatus === 'connected') {
      onTriggerConversation(topics);
    }
  };

  return (
    <div className="chat-area">
      <div className="chat-messages scrollbar-thin" ref={chatMessagesRef}>
        {messages.length === 0 ? (
          <div style={{
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            height: '100%',
            flexDirection: 'column',
            opacity: 0.6,
            textAlign: 'center',
            color: 'white'
          }}>
            <div style={{ fontSize: '1.2rem', marginBottom: '0.5rem' }}>
              🤖 AI Office is Starting Up
            </div>
            <div style={{ fontSize: '0.9rem' }}>
              The AI team will begin discussing market conditions shortly...
            </div>
            {connectionStatus !== 'connected' && (
              <div style={{ fontSize: '0.8rem', marginTop: '0.5rem', opacity: 0.7 }}>
                Waiting for connection...
              </div>
            )}
          </div>
        ) : (
          messages.map((message) => (
            <div key={message.id} className="message">
              <div 
                className="message-avatar"
                style={{ background: getAgentAvatarColor(message.agent_role) }}
              >
                {getAgentInitials(message.agent_name)}
              </div>
              
              <div className="message-content">
                <div className="message-header">
                  <span className="message-name">{message.agent_name}</span>
                  <span className="message-role">
                    {message.agent_role?.replace(/([A-Z])/g, ' $1').trim()}
                  </span>
                  {message.message_type && getMessageTypeLabel(message.message_type) && (
                    <span className="message-type">
                      {getMessageTypeLabel(message.message_type)}
                    </span>
                  )}
                  <span className="message-timestamp">
                    {formatTimestamp(message.timestamp)}
                  </span>
                </div>
                <div className="message-text">
                  {message.content}
                </div>
              </div>
            </div>
          ))
        )}
        <div ref={messagesEndRef} />
      </div>

      <div className="chat-controls">
        <button 
          className="trigger-btn"
          onClick={() => handleTriggerClick(['market_analysis'])}
          disabled={connectionStatus !== 'connected'}
          title="Trigger market analysis discussion"
        >
          📊 Market Analysis
        </button>
        
        <button 
          className="trigger-btn"
          onClick={() => handleTriggerClick(['risk_assessment'])}
          disabled={connectionStatus !== 'connected'}
          title="Trigger risk assessment discussion"
        >
          ⚠️ Risk Review
        </button>
        
        <button 
          className="trigger-btn"
          onClick={() => handleTriggerClick(['trading_strategy'])}
          disabled={connectionStatus !== 'connected'}
          title="Trigger trading strategy discussion"
        >
          💡 Strategy Talk
        </button>
        
        <button 
          className="trigger-btn"
          onClick={() => handleTriggerClick(['general_discussion'])}
          disabled={connectionStatus !== 'connected'}
          title="Trigger general discussion"
        >
          💬 General Chat
        </button>

        <div style={{ marginLeft: 'auto', fontSize: '0.8rem', opacity: 0.7, color: 'white' }}>
          {messages.length} messages
        </div>
      </div>
    </div>
  );
};

export default ChatArea;