import asyncio
import json
from datetime import datetime, timedelta
from typing import Dict, List, Optional
from apscheduler.schedulers.asyncio import AsyncIOScheduler
from apscheduler.triggers.interval import IntervalTrigger
from apscheduler.triggers.cron import CronTrigger
from agents import OfficeManager
from data_sources import AStockDataManager
from database import get_db, Message, MarketData, NewsData, OfficeState
import re
import random

class ConversationScheduler:
    """Conversation Scheduler - manages organic conversation rhythm and triggers"""
    
    def __init__(self, office_manager: OfficeManager, data_manager: AStockDataManager):
        self.office_manager = office_manager
        self.data_manager = data_manager
        self.scheduler = AsyncIOScheduler()
        self.conversation_active = False
        self.last_message_time = None
        self.websocket_manager = None  # Will be set in main.py
        
        # Conversation parameters - more natural rhythm
        self.min_conversation_interval = 30  # Minimum conversation interval (seconds)
        self.max_silence_duration = 180     # Maximum silence duration (seconds)
        self.conversation_burst_probability = 0.6  # Probability of continuous conversation
        self.natural_conversation_rate = 0.25  # Natural conversation trigger rate
        self.market_change_threshold = 3.0   # Market change threshold (%)
        self.last_trigger_time = None        # Last trigger time
        
    def start(self):
        """Start scheduler"""
        # Data refresh tasks
        self.scheduler.add_job(
            self._refresh_market_data,
            IntervalTrigger(minutes=5),  # 每5分钟刷新市场数据
            id="market_data_refresh"
        )
        
        self.scheduler.add_job(
            self._refresh_news_data,
            IntervalTrigger(hours=1),    # 每小时刷新新闻
            id="news_data_refresh"
        )
        
        self.scheduler.add_job(
            self._refresh_fear_greed_data,
            IntervalTrigger(hours=2),    # 每2小时刷新恐慌贪婪指数
            id="fear_greed_refresh"
        )
        
        self.scheduler.add_job(
            self._refresh_twitter_sentiment,
            IntervalTrigger(minutes=45),  # 每45分钟刷新Twitter情绪
            id="twitter_sentiment_refresh"
        )
        
        # 对话触发任务 - 降低检查频率
        self.scheduler.add_job(
            self._check_conversation_triggers,
            IntervalTrigger(seconds=30), # 每30秒检查对话触发条件（降低频率）
            id="conversation_trigger_check"
        )
        
        # 整点会议
        self.scheduler.add_job(
            self._trigger_hourly_meeting,
            CronTrigger(minute=0),       # 每小时整点
            id="hourly_meeting"
        )
        
        self.scheduler.start()
        print("Conversation Scheduler started")
    
    def stop(self):
        """停止调度器"""
        self.scheduler.shutdown()
        print("Conversation Scheduler stopped")
    
    async def _refresh_market_data(self):
        """刷新市场数据"""
        try:
            market_data = await self.data_manager.refresh_market_data()
            
            # 检查是否有重大价格变化
            await self._check_market_volatility(market_data)
            
            # 保存到数据库
            db = next(get_db())
            market_record = MarketData(
                btc_price=market_data.get("sh_index", {}).get("close", 0),  # 使用上证指数收盘价
                eth_price=market_data.get("sz_index", {}).get("close", 0),  # 使用深证成指收盘价
                btc_change_24h=market_data.get("sh_index", {}).get("pct_chg", 0),  # 使用上证涨跌幅
                eth_change_24h=market_data.get("sz_index", {}).get("pct_chg", 0),  # 使用深证涨跌幅
                volume_data=json.dumps(market_data)
            )
            db.add(market_record)
            db.commit()
            db.close()
            
        except Exception as e:
            print(f"Error refreshing market data: {e}")
    
    async def _refresh_news_data(self):
        """刷新新闻数据"""
        try:
            news_data = await self.data_manager.refresh_news()
            
            # 检查是否有重要新闻
            await self._check_important_news(news_data)
            
        except Exception as e:
            print(f"Error refreshing news data: {e}")
    
    async def _refresh_fear_greed_data(self):
        """刷新恐慌贪婪指数数据"""
        try:
            fear_greed_data = await self.data_manager.refresh_fear_greed()
            
            # 检查指数极值，可能触发对话
            index_value = fear_greed_data.get("fear_greed_index", 50)
            if index_value <= 20 or index_value >= 80:  # 极度恐慌或极度贪婪
                await self._trigger_conversation({
                    "trigger_type": "fear_greed_extreme",
                    "urgency_level": 0.7,
                    "topics": ["market_sentiment", "fear_greed_analysis"],
                    "fear_greed_index": index_value,
                    "classification": fear_greed_data.get("classification")
                })
                
        except Exception as e:
            print(f"Error refreshing fear & greed data: {e}")
    
    async def _refresh_twitter_sentiment(self):
        """刷新Twitter情绪数据"""
        try:
            sentiment_data = await self.data_manager.refresh_twitter_sentiment()
            
            # 检查情绪变化和活跃度
            sentiment_label = sentiment_data.get("sentiment_label")
            mention_volume = sentiment_data.get("mention_volume", 0)
            
            # 高活跃度或极端情绪触发对话
            if mention_volume > 3000 or sentiment_label in ["Very Bullish", "Very Bearish"]:
                await self._trigger_conversation({
                    "trigger_type": "twitter_sentiment",
                    "urgency_level": 0.6,
                    "topics": ["social_sentiment", "twitter_analysis"],
                    "sentiment": sentiment_label,
                    "mention_volume": mention_volume,
                    "trending_topics": sentiment_data.get("trending_topics", [])
                })
                
        except Exception as e:
            print(f"Error refreshing Twitter sentiment: {e}")
    
    async def _check_market_volatility(self, market_data: Dict):
        """检查市场波动性，可能触发对话"""
        sh_change = abs(market_data.get("sh_index", {}).get("pct_chg", 0))  # 上证指数涨跌幅
        sz_change = abs(market_data.get("sz_index", {}).get("pct_chg", 0))  # 深证成指涨跌幅
        cy_change = abs(market_data.get("cy_index", {}).get("pct_chg", 0))  # 创业板指涨跌幅
        
        # 动态阈值检测
        max_change = max(sh_change, sz_change, cy_change)
        if max_change > self.market_change_threshold:
            # 防止过于频繁触发
            if self.last_trigger_time:
                time_since_last = (datetime.utcnow() - self.last_trigger_time).total_seconds()
                if time_since_last < self.min_conversation_interval:
                    return
            
            urgency = min(1.0, max_change / 10.0)  # 10%变化=最高紧急度
            context = {
                "trigger_type": "market_volatility",
                "market_data": market_data,
                "urgency_level": urgency,
                "topics": ["market_data", "volatility", "price_movement"]
            }
            await self._trigger_conversation(context)
            self.last_trigger_time = datetime.utcnow()
    
    async def _check_important_news(self, news_data: List[Dict]):
        """检查重要新闻，可能触发对话"""
        if not news_data:
            return
        
        # 简单的重要性检测（基于A股关键词）
        important_keywords = ["大涨", "大跌", "涨停", "跌停", "监管", "政策", "利好", "利空", "重组", "并购", "业绩", "财报", "IPO", "退市"]
        
        for news in news_data[:3]:  # 检查前3条新闻
            title = news.get("title", "").lower()
            description = news.get("description", "").lower()
            
            if any(keyword in title or keyword in description for keyword in important_keywords):
                context = {
                    "trigger_type": "important_news",
                    "news_data": news_data[:3],
                    "urgency_level": 0.8,
                    "topics": ["news_analysis", "market_sentiment"]
                }
                await self._trigger_conversation(context)
                break
    
    async def _check_trending_topics(self, trends_data: Dict):
        """检查热点话题，可能触发对话"""
        if not trends_data:
            return
        
        market_sentiment = trends_data.get("market_sentiment", {})
        trending_topics = trends_data.get("trending_topics", [])
        
        # 检查恐慌贪婪指数是否极端
        fear_greed = market_sentiment.get("fear_greed_index", 50)
        if fear_greed < 25 or fear_greed > 75:
            context = {
                "trigger_type": "extreme_sentiment",
                "trends_data": trends_data,
                "urgency_level": 0.7,
                "topics": ["market_sentiment", "psychology"]
            }
            await self._trigger_conversation(context)
            return
        
        # 检查是否有特别热门的话题
        hot_topics = ["AI概念", "新能源", "芯片", "医药生物", "房地产", "金融科技", "5G", "新基建"]
        if any(topic in trending_topics for topic in hot_topics):
            context = {
                "trigger_type": "trending_topics",
                "trends_data": trends_data,
                "urgency_level": 0.6,
                "topics": ["trending_analysis", "future_trends"]
            }
            await self._trigger_conversation(context)
    
    async def _check_conversation_triggers(self):
        """检查各种对话触发条件"""
        current_time = datetime.utcnow()
        
        # 0. 防止过于频繁的检查
        if self.last_trigger_time:
            time_since_last = (current_time - self.last_trigger_time).total_seconds()
            if time_since_last < 20:  # 20秒内不重复触发
                return
        
        # 1. 检查静默时长
        if self.last_message_time:
            silence_duration = (current_time - self.last_message_time).total_seconds()
            
            # 如果静默超过最大时长，触发随机对话
            if silence_duration > self.max_silence_duration:
                await self._trigger_random_conversation()
                return
        
        # 2. 检查是否处于对话爆发期
        if self.conversation_active:
            # 在对话爆发期，有一定概率继续对话
            if random.random() < self.conversation_burst_probability:
                await self._continue_conversation()
            else:
                self.conversation_active = False
        
        # 3. 智能随机触发 - 考虑上下文
        elif self._should_trigger_natural_conversation():
            await self._trigger_random_conversation()
    
    def _should_trigger_natural_conversation(self) -> bool:
        """智能判断是否应该触发自然对话"""
        # 基础概率
        base_prob = self.natural_conversation_rate
        
        # 根据静默时长调整
        if self.last_message_time:
            silence_duration = (datetime.utcnow() - self.last_message_time).total_seconds()
            if silence_duration > 60:  # 静默超过1分钟，提高概率
                base_prob *= 1.5
            elif silence_duration < 30:  # 静默不足30秒，降低概率
                base_prob *= 0.3
        
        # 根据办公室状态调整
        office_heat = self.office_manager.office_state.get("discussion_heat", 0)
        if office_heat > 0.7:  # 讨论很热烈，降低概率
            base_prob *= 0.5
        elif office_heat < 0.3:  # 讨论冷淡，提高概率
            base_prob *= 1.3
        
        return random.random() < base_prob
    
    async def _trigger_conversation(self, context: Dict):
        """触发一次对话"""
        try:
            # 获取最新数据
            market_data = await self.data_manager.get_latest_market_data()
            news_data = await self.data_manager.get_latest_news()
            trends_data = await self.data_manager.get_latest_trends()
            
            # 构建完整上下文
            full_context = {
                **context,
                "market_data": market_data,
                "news_data": news_data,
                "trends_data": trends_data,
                "office_state": self.office_manager.office_state,
                "recent_messages": self.office_manager.conversation_history[-5:],
                "silence_duration": self._get_silence_duration()
            }
            
            # 如果是用户输入触发，添加特殊处理
            if context.get("trigger_type") == "user_input":
                full_context["user_input_content"] = context.get("user_content", "")
                full_context["user_input_type"] = context.get("input_type", "general")
            
            # 选择发言者
            speaker = self.office_manager.get_next_speaker(full_context)
            if not speaker:
                return
            
            # 向前端广播"思考中"提示，便于用户感知等待
            if self.websocket_manager:
                await self.websocket_manager.broadcast_message({
                    "type": "system_message",
                    "data": {
                        "content": f"🤔 {speaker.name} is thinking...",
                        "timestamp": datetime.utcnow().isoformat()
                    }
                })
            
            # 生成回应
            response_type = context.get("response_type", "normal")
            rsp = await speaker.generate_response(full_context, response_type)
            raw_content = rsp.get("content", "")

            # 去除<think>内容
            content_no_think = re.sub(r"<think>[\s\S]*?</think>", "", raw_content, flags=re.IGNORECASE)

            # 抓动作 *xxx*
            actions = re.findall(r"\*(.*?)\*", content_no_think)
            display_content = re.sub(r"\*[^\*]+\*", "", content_no_think).strip()

            # 创建消息
            message = {
                "agent_name": speaker.name,
                "agent_role": speaker.role,
                "content": display_content,
                "actions": actions,
                "model": rsp.get("model"),
                "latency_ms": rsp.get("latency_ms"),
                "tokens": rsp.get("tokens"),
                "message_type": context.get("trigger_type", "normal"),
                "interest_score": speaker.calculate_interest_score(full_context),
                "timestamp": datetime.utcnow().isoformat()
            }
            
            # 保存消息
            await self._save_message(message, full_context)
            
            # 更新办公室状态
            self.office_manager.update_office_state(message)
            self.office_manager.conversation_history.append(message)
            
            # 发送到前端
            if self.websocket_manager:
                await self.websocket_manager.broadcast_message({
                    "type": "chat_message",
                    "data": message
                })
            
            # 更新对话状态
            self.last_message_time = datetime.utcnow()
            self.conversation_active = True
            
            # 检查是否有人要打断
            await self._check_interruptions(speaker, display_content, full_context)
            
        except Exception as e:
            print(f"Error in conversation trigger: {e}")
    
    async def _trigger_random_conversation(self):
        """触发随机对话"""
        topics = ["market_overview", "technical_analysis", "risk_assessment", "general_discussion"]
        topic = random.choice(topics)
        
        context = {
            "trigger_type": "random",
            "urgency_level": 0.3,
            "topics": [topic]
        }
        
        await self._trigger_conversation(context)
    
    async def _continue_conversation(self):
        """继续当前对话"""
        if not self.office_manager.conversation_history:
            return
        
        last_message = self.office_manager.conversation_history[-1]
        
        context = {
            "trigger_type": "continuation",
            "urgency_level": 0.5,
            "topics": ["response", "discussion"],
            "response_type": "response"
        }
        
        await self._trigger_conversation(context)
    
    async def _check_interruptions(self, speaking_agent, content: str, context: Dict):
        """检查打断情况"""
        interruption = self.office_manager.check_interruptions(speaking_agent, content, context)
        
        if interruption:
            interrupting_agent, reason = interruption
            
            # 稍微延迟后触发打断
            await asyncio.sleep(2)
            
            interrupt_context = {
                **context,
                "trigger_type": "interruption",
                "response_type": "interruption",
                "urgency_level": 0.8,
                "topics": [reason]
            }
            
            # 强制指定打断者作为下一个发言者
            rsp2 = await interrupting_agent.generate_response(interrupt_context, "interruption")
            raw2 = rsp2.get("content", "")
            content2 = re.sub(r"<think>[\s\S]*?</think>", "", raw2, flags=re.IGNORECASE)
            actions2 = re.findall(r"\*(.*?)\*", content2)
            display2 = re.sub(r"\*[^\*]+\*", "", content2).strip()
            
            message = {
                "agent_name": interrupting_agent.name,
                "agent_role": interrupting_agent.role,
                "content": display2,
                "actions": actions2,
                "model": rsp2.get("model"),
                "latency_ms": rsp2.get("latency_ms"),
                "tokens": rsp2.get("tokens"),
                "message_type": "interruption",
                "interest_score": 1.0,
                "timestamp": datetime.utcnow().isoformat()
            }
            
            await self._save_message(message, interrupt_context)
            self.office_manager.update_office_state(message)
            self.office_manager.conversation_history.append(message)
            
            if self.websocket_manager:
                await self.websocket_manager.broadcast_message({
                    "type": "chat_message",
                    "data": message
                })
    
    async def _trigger_hourly_meeting(self):
        """触发整点会议"""
        print("Starting hourly meeting...")
        
        context = {
            "trigger_type": "hourly_meeting",
            "urgency_level": 0.9,
            "topics": ["meeting", "summary", "decision_making"],
            "response_type": "meeting"
        }
        
        # 会议模式：让协调员先发言
        coordinator = self.office_manager.agents.get("Coordinator")
        if coordinator:
            await self._trigger_conversation({
                **context,
                "forced_speaker": coordinator
            })
        
        # 然后触发其他人的会议发言
        for _ in range(3):  # 最多3轮会议发言
            await asyncio.sleep(10)  # 10秒间隔
            await self._trigger_conversation(context)
    
    async def _save_message(self, message: Dict, context: Dict):
        """保存消息到数据库"""
        try:
            db = next(get_db())
            
            db_message = Message(
                agent_name=message["agent_name"],
                agent_role=message["agent_role"],
                content=message["content"],
                message_type=message["message_type"],
                context_data=json.dumps(context),
                interest_score=message["interest_score"]
            )
            
            db.add(db_message)
            db.commit()
            db.close()
            
        except Exception as e:
            print(f"Error saving message: {e}")
    
    def _get_silence_duration(self) -> float:
        """获取当前静默时长"""
        if not self.last_message_time:
            return 0.0
        return (datetime.utcnow() - self.last_message_time).total_seconds()