import asyncio
import json
import random
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Tuple
from openai import OpenAI
import config as config_module
from config import AGENTS_CONFIG, YUNWU_API_BASE, YUNWU_API_KEY, VOLCES_API_BASE, VOLCES_API_KEY
from database import get_db, Message, AgentState, OfficeState
import math

class Agent:
    """Base Agent class - Core implementation of organic interaction"""
    
    def __init__(self, role: str, config: Dict):
        self.role = role
        self.name = config["name"]
        self.model = config["model"]
        self.personality = config["personality"]
        self.quirks = config.get("quirks", [])
        self.expertise = config["expertise"]
        self.languages = config.get("languages", None)
        self.trigger_sensitivity = config["trigger_sensitivity"]
        self.base_assertiveness = config["assertiveness"]
        self.chattiness = config.get("chattiness", 0.5)
        self.humor_level = config.get("humor_level", 0.3)
        
        # Dynamic states
        self.confidence_level = 0.7 + random.uniform(-0.2, 0.2)  # Personalized starting value
        self.current_assertiveness = self.base_assertiveness
        self.collaboration_mood = 0.8 + random.uniform(-0.3, 0.2)
        self.stress_level = 0.2 + random.uniform(-0.1, 0.3)
        self.energy_level = random.uniform(0.4, 0.9)  # New: energy level
        self.recent_interactions = []
        self.last_spoke_time = None
        self.consecutive_silences = 0  # Consecutive silence count
        
        # AI client - select API based on model
        # 确定实际使用的模型
        self.actual_model = config_module.LOW_COST_MODEL if config_module.LOW_COST_MODE else self.model
        
        if self.actual_model in ["doubao-seed-1-6-thinking-250615", "deepseek-r1-250528", "doubao-seed-1-6-flash", "doubao-seed-1-6-flash-250615"]:
            # Volcano Engine API
            self.client = OpenAI(
                api_key=VOLCES_API_KEY,
                base_url=VOLCES_API_BASE
            )
        else:
            # Yunwu API (OpenAI compatible)
            self.client = OpenAI(
                api_key=YUNWU_API_KEY,
                base_url=YUNWU_API_BASE
            )
    
    def calculate_interest_score(self, context: Dict) -> float:
        """计算对当前话题的兴趣度 - 决定是否主动发言"""
        base_interest = 0.0
        
        # 1. 专业领域相关性 (更强的触发)
        current_topics = context.get("topics", [])
        for topic in current_topics:
            if any(expertise in topic.lower() for expertise in self.expertise):
                base_interest += 0.6  # 增强专业触发
                break
        
        # 2. 数据变化敏感度 (个性化)
        market_data = context.get("market_data", {})
        if market_data:
            btc_change = abs(market_data.get("btc", {}).get("change_24h", 0))
            eth_change = abs(market_data.get("eth", {}).get("change_24h", 0))
            max_change = max(btc_change, eth_change)
            
            if max_change > self.trigger_sensitivity:
                base_interest += 0.4 * (max_change / 5.0)  # 动态调整
        
        # 3. 反驳和补充冲动 (更真实)
        recent_messages = context.get("recent_messages", [])
        if recent_messages and len(recent_messages) > 0:
            last_message = recent_messages[-1]
            if self._wants_to_respond(last_message):
                base_interest += 0.5
        
        # 4. 个性特质影响
        personality_boost = 0.0
        
        # 话痨特质
        if self.chattiness > 0.7:
            personality_boost += 0.3
        
        # 能量水平影响
        personality_boost += (self.energy_level - 0.5) * 0.4
        
        # 压力下的反应
        if self.stress_level > 0.6:
            personality_boost += 0.2  # 压力大时更容易发言
        
        # 5. 静默惩罚/奖励 (防止过度沉默或话痨)
        silence_duration = context.get("silence_duration", 0)
        recent_speech_count = sum(1 for msg in recent_messages[-5:] 
                                if msg.get("agent_name") == self.name)
        
        if silence_duration > 60:  # 1分钟静默，话痨开始想说话
            if self.chattiness > 0.6:
                personality_boost += 0.4
        elif silence_duration > 180:  # 3分钟静默，大家都想说话
            personality_boost += 0.3
        
        if recent_speech_count >= 2:  # 最近说得太多，稍微降低兴趣
            personality_boost -= 0.2
        
        # 6. 随机性和个性波动
        random_factor = random.uniform(-0.2, 0.2)
        mood_factor = (self.collaboration_mood - 0.5) * 0.3
        
        final_interest = base_interest + personality_boost + random_factor + mood_factor
        
        # 应用个性调整
        final_interest *= (self.current_assertiveness * 0.7 + 0.3)
        
        return max(0.0, min(final_interest, 1.0))
    
    def should_interrupt(self, speaking_agent: str, content: str, context: Dict) -> Tuple[bool, str]:
        """判断是否应该打断当前发言者"""
        interrupt_probability = 0.0
        interrupt_reason = ""
        
        # 1. 强烈反对（专业纠错）
        if self._strongly_disagrees_with(content):
            interrupt_probability = 0.7
            interrupt_reason = "disagreement"
        
        # 2. 关键信息补充
        elif self._has_critical_addition(content):
            interrupt_probability = 0.5
            interrupt_reason = "addition"
        
        # 3. 澄清误解
        elif self._needs_clarification(content):
            interrupt_probability = 0.6
            interrupt_reason = "clarification"
        
        # 4. 个性调整
        interrupt_probability *= self.current_assertiveness
        
        # 5. 压力下更容易打断
        if self.stress_level > 0.7:
            interrupt_probability *= 1.3
        
        should_interrupt = random.random() < interrupt_probability
        return should_interrupt, interrupt_reason
    
    async def generate_response(self, context: Dict, response_type: str = "normal") -> dict:
        """生成回应内容"""
        try:
            prompt = self._build_prompt(context, response_type)
            
            start_ts = datetime.utcnow().timestamp()
            response = self.client.chat.completions.create(
                model=self.actual_model,
                messages=[
                    {"role": "system", "content": self._get_system_prompt()},
                    {"role": "user", "content": prompt}
                ],
                max_tokens=120,
                temperature=0.7
            )
            
            latency_ms = int((datetime.utcnow().timestamp() - start_ts) * 1000)
            content = response.choices[0].message.content.strip()
            usage = getattr(response, "usage", None)
            tokens = {
                "prompt": usage.prompt_tokens if usage else None,
                "completion": usage.completion_tokens if usage else None,
                "total": usage.total_tokens if usage else None
            }
            
            # 更新交互历史
            self._update_interaction_history(context, content)
            
            return {
                "content": content,
                "model": self.model,
                "latency_ms": latency_ms,
                "tokens": tokens
            }
            
        except Exception as e:
            print(f"Error generating response for {self.name}: {e}")
            return {
                "content": f"{self.name}: I need some time to think about this...",
                "model": self.model,
                "latency_ms": None,
                "tokens": None
            }
    
    def _get_system_prompt(self) -> str:
        """Get system prompt for A股 analyst"""
        quirks_str = ", ".join(self.quirks) if self.quirks else "无特殊习惯"
        
        return f"""你是{self.name}，A股市场{self.role}分析师。性格特点：{self.personality}

专业领域：{', '.join(self.expertise)} | 个人习惯：{quirks_str}
当前状态：自信度{self.confidence_level:.1f}，活跃度{self.energy_level:.1f}，压力度{self.stress_level:.1f}

请保持简洁(1-2句话)，自然真实地表达观点。专注于你的专业领域。始终用中文回复，体现专业分析师的身份。使用A股市场专业术语。"""
    
    def _build_prompt(self, context: Dict, response_type: str) -> str:
        """Build specific prompt for A股 market"""
        market_data = context.get("market_data", {})
        concept_data = context.get("concept_data", [])
        recent_messages = context.get("recent_messages", [])
        market_stats = context.get("market_stats", {})
        recent_trends = context.get("recent_trends", {})
        
        # Build A股 market information
        prompt = f"""当前A股市场情况："""
        
        # 主要指数
        indices = market_data.get('indices', {})
        for code, data in indices.items():
            name = data.get('name', code)
            close = data.get('close', 0)
            change_pct = data.get('change_pct', 0)
            if close > 0:
                prompt += f"\n{name}: {close:.2f}点 ({change_pct:+.2f}%)"
        
        # 市场情绪指标
        if market_stats:
            up_down_ratio = market_stats.get('up_down_ratio', 1)
            limit_up = market_stats.get('limit_up_count', 0)
            limit_down = market_stats.get('limit_down_count', 0)
            turnover = market_stats.get('turnover_billion', 1000)
            prompt += f"\n涨跌比: {up_down_ratio:.2f}，涨停{limit_up}家，跌停{limit_down}家"
            prompt += f"\n成交额: {turnover:.0f}亿元"
        
        # 热点概念
        if concept_data:
            prompt += f"\n热点概念："
            for i, concept in enumerate(concept_data[:3]):
                name = concept.get('name', '')
                change_pct = concept.get('change_pct', 0)
                prompt += f" {name}({change_pct:+.1f}%)"
                if i < 2:
                    prompt += ","
        
        # 趋势分析
        if recent_trends:
            direction = recent_trends.get('market_direction', '震荡')
            volume_trend = recent_trends.get('volume_trend', '正常')
            risk_level = recent_trends.get('risk_level', '中')
            prompt += f"\n市场趋势: {direction}，成交量{volume_trend}，风险等级{risk_level}"
        
        # 最近讨论
        if recent_messages:
            prompt += "\n\n最近讨论："
            for msg in recent_messages[-2:]:  # 最近2条消息
                content = msg.get('content', '')[:80]  # 截断到80字符
                agent_name = msg.get('agent_name', '')
                prompt += f"\n{agent_name}: {content}"
        
        # 根据响应类型调整提示
        if response_type == "interruption":
            prompt += f"\n\n你觉得有必要打断当前讨论。请简要表达你的观点(1-2句话)："
        elif response_type == "response":
            prompt += f"\n\n请针对以上讨论内容发表你的专业看法(1-2句话)："
        else:
            prompt += f"\n\n基于以上信息，请分享你的专业观点(1-2句话)："
        
        return prompt
    
    def _wants_to_respond(self, last_message: Dict) -> bool:
        """Determine if wants to respond to the last message"""
        if not last_message or last_message.get("agent_name") == self.name:
            return False
        
        content = last_message.get("content", "").lower()
        
        # 1. 被直接@了
        if self.name.lower() in content or f"@{self.name.lower()}" in content:
            return True
        
        # 2. 专业领域被提及
        for expertise in self.expertise:
            if expertise.replace("_", " ") in content:
                return random.random() < 0.7  # 70%概率想回应
        
        # 3. 个性化反应
        # 数据狂Alex对数字敏感
        if self.name == "Alex" and any(char.isdigit() for char in content):
            return random.random() < 0.5
        
        # 风险管理Casey对风险词汇敏感
        if self.name == "Casey" and any(word in content for word in ["risk", "danger", "volatile", "crash", "loss"]):
            return random.random() < 0.8
        
        # 交易员Riley对行动词汇敏感
        if self.name == "Riley" and any(word in content for word in ["buy", "sell", "trade", "profit", "strategy"]):
            return random.random() < 0.6
        
        # 情绪分析师Jordan对情绪词汇敏感
        if self.name == "Jordan" and any(word in content for word in ["feel", "think", "sentiment", "fomo", "fud", "bullish", "bearish"]):
            return random.random() < 0.6
        
        # 4. Chattiness trait
        if self.chattiness > 0.7:
            return random.random() < 0.4
        
        # 5. 能量水平影响
        if self.energy_level > 0.8:
            return random.random() < 0.3
        
        return False
    
    def _has_disagreement(self, opinions: List[str]) -> bool:
        """检测是否与最近的观点存在分歧"""
        if not opinions:
            return False
            
        disagreement_keywords = ["wrong", "disagree", "incorrect", "mistake", "no way", "bullshit"]
        agreement_keywords = ["agree", "correct", "right", "exactly", "true", "yes"]
        
        for opinion in opinions[-2:]:  # 检查最近2个观点
            opinion_lower = opinion.lower()
            if any(keyword in opinion_lower for keyword in agreement_keywords):
                return False
            if any(keyword in opinion_lower for keyword in disagreement_keywords):
                return True
        
        # 基于个性的分歧倾向
        if self.name == "Casey":  # 风险管理员更容易反对
            return random.random() < 0.5
        elif self.name == "Riley":  # 交易员比较直接
            return random.random() < 0.4
        else:
            return random.random() < 0.2
    
    def _strongly_disagrees_with(self, content: str) -> bool:
        """检测是否强烈反对某个观点"""
        # 在专业领域内更容易反对
        content_lower = content.lower()
        expertise_mentioned = any(exp in content_lower for exp in self.expertise)
        
        if expertise_mentioned:
            return random.random() < 0.4
        return random.random() < 0.1
    
    def _has_critical_addition(self, content: str) -> bool:
        """检测是否有关键信息要补充"""
        content_lower = content.lower()
        expertise_mentioned = any(exp in content_lower for exp in self.expertise)
        
        if expertise_mentioned:
            return random.random() < 0.3
        return random.random() < 0.05
    
    def _needs_clarification(self, content: str) -> bool:
        """检测是否需要澄清"""
        return random.random() < 0.1
    
    def _update_interaction_history(self, context: Dict, response: str):
        """更新交互历史"""
        self.recent_interactions.append({
            "timestamp": datetime.utcnow().isoformat(),
            "context": context.get("summary", ""),
            "response": response,
            "market_data": context.get("market_data", {})
        })
        
        # 保持最近10次交互
        if len(self.recent_interactions) > 10:
            self.recent_interactions = self.recent_interactions[-10:]
        
        self.last_spoke_time = datetime.utcnow()
    
    def update_emotional_state(self, interaction_result: Dict):
        """根据交互结果更新情绪状态"""
        # 被采纳或同意 → 提升自信
        if interaction_result.get("was_agreed_with", False):
            self.confidence_level = min(1.0, self.confidence_level * 1.1)
        
        # 被反驳 → 降低自信
        if interaction_result.get("was_disagreed_with", False):
            self.confidence_level = max(0.3, self.confidence_level * 0.9)
        
        # 紧急情况 → 提升主动性
        if interaction_result.get("urgency_level", 0) > 0.8:
            self.current_assertiveness = min(1.0, self.current_assertiveness * 1.2)
        
        # 长时间未发言 → 降低主动性
        if self.last_spoke_time and (datetime.utcnow() - self.last_spoke_time).seconds > 300:
            self.current_assertiveness = max(0.2, self.current_assertiveness * 0.95)

class OfficeManager:
    """办公室管理器 - 协调所有智能体的交互"""
    
    def __init__(self):
        self.agents = {}
        self.office_state = {
            "current_topic": None,
            "discussion_heat": 0.0,
            "consensus_level": 0.0,
            "urgency_level": 0.0,
            "emotional_climate": "neutral",
            "last_speakers": [],
            "silence_start_time": None
        }
        self.conversation_history = []
        self.active_conversation = True
        
        # Initialize all A股 agents (no translator needed)
        for role, config in AGENTS_CONFIG.items():
            self.agents[role] = Agent(role, config)
    
    def get_next_speaker(self, context: Dict) -> Optional[Agent]:
        """确定下一个发言者"""
        if not self.agents:
            return None
        
        speaker_weights = {}
        
        for role, agent in self.agents.items():
            # 计算发言权重
            interest_score = agent.calculate_interest_score(context)
            
            # 避免同一人连续发言
            if agent.name in self.office_state["last_speakers"][-2:]:
                interest_score *= 0.3
            
            # 平衡发言频次
            recent_speech_count = sum(1 for msg in self.conversation_history[-10:] 
                                    if msg.get("agent_name") == agent.name)
            if recent_speech_count > 3:
                interest_score *= 0.5
            
            speaker_weights[agent] = max(0.01, interest_score)
        
        # 加权随机选择
        total_weight = sum(speaker_weights.values())
        if total_weight == 0:
            return random.choice(list(self.agents.values()))
        
        rand_val = random.random() * total_weight
        current_weight = 0
        
        for agent, weight in speaker_weights.items():
            current_weight += weight
            if rand_val <= current_weight:
                return agent
        
        return list(self.agents.values())[0]
    
    def check_interruptions(self, speaking_agent: Agent, content: str, context: Dict) -> Optional[Tuple[Agent, str]]:
        """检查是否有智能体要打断"""
        for role, agent in self.agents.items():
            if agent == speaking_agent:
                continue
            
            should_interrupt, reason = agent.should_interrupt(speaking_agent.name, content, context)
            if should_interrupt:
                return agent, reason
        
        return None
    
    def update_office_state(self, message: Dict):
        """更新办公室整体状态"""
        # 更新最近发言者
        speaker_name = message.get("agent_name")
        if speaker_name:
            if speaker_name in self.office_state["last_speakers"]:
                self.office_state["last_speakers"].remove(speaker_name)
            self.office_state["last_speakers"].insert(0, speaker_name)
            self.office_state["last_speakers"] = self.office_state["last_speakers"][:5]
        
        # 重置静默计时
        self.office_state["silence_start_time"] = None
        
        # 更新讨论热度（基于发言频率）
        recent_messages = len([msg for msg in self.conversation_history[-10:] 
                             if (datetime.utcnow() - datetime.fromisoformat(msg.get("timestamp", datetime.utcnow().isoformat()))).seconds < 120])
        self.office_state["discussion_heat"] = min(1.0, recent_messages / 10.0)
        
        # 分析情绪氛围
        self._analyze_emotional_climate()
    
    def _analyze_emotional_climate(self):
        """分析当前的情绪氛围"""
        if len(self.conversation_history) < 3:
            return
        
        recent_messages = self.conversation_history[-5:]
        
        # 简化的情绪分析
        disagreement_count = sum(1 for msg in recent_messages 
                               if any(word in msg.get("content", "").lower() 
                                    for word in ["disagree", "wrong", "mistake", "no"]))
        
        agreement_count = sum(1 for msg in recent_messages 
                            if any(word in msg.get("content", "").lower() 
                                 for word in ["agree", "right", "exactly", "yes"]))
        
        if disagreement_count > agreement_count + 1:
            self.office_state["emotional_climate"] = "tense"
        elif agreement_count > disagreement_count + 1:
            self.office_state["emotional_climate"] = "harmonious"
        elif self.office_state["discussion_heat"] > 0.7:
            self.office_state["emotional_climate"] = "active"
        else:
            self.office_state["emotional_climate"] = "neutral"


# A股版本不需要翻译功能 - TranslationAgent已移除