import Phaser from 'phaser';
import { GAME_CONFIG, COLORS } from '../config/GameConfig';

export class MenuScene extends Phaser.Scene {
    private titleText!: Phaser.GameObjects.Text;
    private startButton!: Phaser.GameObjects.Text;
    private background!: Phaser.GameObjects.Graphics;

    constructor() {
        super({ key: 'MenuScene' });
    }

    preload(): void {
        // 创建简单的像素艺术资源
        this.createPixelArt();
    }

    create(): void {
        this.createBackground();
        this.createTitle();
        this.createButtons();
        this.createAnimations();
        this.setupInput();
    }

    private createPixelArt(): void {
        // 创建玩家精灵
        const playerGraphics = this.make.graphics({ x: 0, y: 0 });
        playerGraphics.fillStyle(0x4a90e2);
        playerGraphics.fillRect(0, 0, 32, 32);
        playerGraphics.fillStyle(0x2c5aa0);
        playerGraphics.fillRect(8, 8, 16, 16);
        playerGraphics.generateTexture('player', 32, 32);

        // 创建敌人精灵
        const enemyGraphics = this.make.graphics({ x: 0, y: 0 });
        enemyGraphics.fillStyle(0xe74c3c);
        enemyGraphics.fillRect(0, 0, 32, 32);
        enemyGraphics.fillStyle(0xc0392b);
        enemyGraphics.fillRect(8, 8, 16, 16);
        enemyGraphics.generateTexture('enemy', 32, 32);

        // 创建收集品精灵
        const collectibleGraphics = this.make.graphics({ x: 0, y: 0 });
        collectibleGraphics.fillStyle(0xf39c12);
        collectibleGraphics.fillRect(8, 8, 16, 16);
        collectibleGraphics.generateTexture('collectible', 32, 32);
    }

    private createBackground(): void {
        this.background = this.add.graphics();
        
        // 创建像素化的背景图案
        for (let x = 0; x < GAME_CONFIG.WIDTH; x += 40) {
            for (let y = 0; y < GAME_CONFIG.HEIGHT; y += 40) {
                this.background.fillStyle(Phaser.Display.Color.HexStringToColor(COLORS.BACKGROUND).color);
                this.background.fillRect(x, y, 38, 38);
            }
        }
    }

    private createTitle(): void {
        this.titleText = this.add.text(
            GAME_CONFIG.WIDTH / 2,
            150,
            '复古像素冒险',
            {
                fontSize: '48px',
                fontFamily: 'Courier New',
                color: COLORS.TEXT,
                stroke: '#000',
                strokeThickness: 4,
                align: 'center'
            }
        ).setOrigin(0.5);

        // 添加副标题
        this.add.text(
            GAME_CONFIG.WIDTH / 2,
            220,
            '躲避怪物 · 收集宝藏 · 到达终点',
            {
                fontSize: '20px',
                fontFamily: 'Courier New',
                color: COLORS.COLLECTIBLE,
                align: 'center'
            }
        ).setOrigin(0.5);
    }

    private createButtons(): void {
        // 开始游戏按钮
        this.startButton = this.add.text(
            GAME_CONFIG.WIDTH / 2,
            400,
            '开始冒险',
            {
                fontSize: '32px',
                fontFamily: 'Courier New',
                color: COLORS.TEXT,
                backgroundColor: COLORS.PLAYER,
                padding: { x: 30, y: 15 },
                align: 'center'
            }
        ).setOrigin(0.5).setInteractive();

        // 添加按钮悬停效果
        this.startButton.on('pointerover', () => {
            this.startButton.setScale(1.1);
            this.startButton.setBackgroundColor(COLORS.EXIT);
        });

        this.startButton.on('pointerout', () => {
            this.startButton.setScale(1);
            this.startButton.setBackgroundColor(COLORS.PLAYER);
        });

        this.startButton.on('pointerdown', () => {
            this.startGame();
        });

        // 添加操作说明
        this.add.text(
            GAME_CONFIG.WIDTH / 2,
            500,
            '操作说明',
            {
                fontSize: '24px',
                fontFamily: 'Courier New',
                color: COLORS.TEXT,
                align: 'center'
            }
        ).setOrigin(0.5);

        this.add.text(
            GAME_CONFIG.WIDTH / 2,
            540,
            'WASD 或方向键移动\n空格键攻击\n收集所有宝物后到达出口',
            {
                fontSize: '16px',
                fontFamily: 'Courier New',
                color: COLORS.TEXT,
                align: 'center',
                lineSpacing: 8
            }
        ).setOrigin(0.5);
    }

    private createAnimations(): void {
        // 创建标题动画
        this.tweens.add({
            targets: this.titleText,
            scaleX: 1.05,
            scaleY: 1.05,
            duration: 2000,
            yoyo: true,
            repeat: -1,
            ease: 'Sine.easeInOut'
        });

        // 创建按钮脉冲动画
        this.tweens.add({
            targets: this.startButton,
            scaleX: 1.05,
            scaleY: 1.05,
            duration: 1500,
            yoyo: true,
            repeat: -1,
            ease: 'Sine.easeInOut'
        });
    }

    private setupInput(): void {
        // 键盘输入
        this.input.keyboard.on('keydown-SPACE', () => {
            this.startGame();
        });

        this.input.keyboard.on('keydown-ENTER', () => {
            this.startGame();
        });
    }

    private startGame(): void {
        // 添加过渡效果
        this.cameras.main.fadeOut(500, 0, 0, 0, () => {
            this.scene.start('GameScene', { level: 1 });
        });
    }

    update(): void {
        // 可以添加背景动画或其他更新逻辑
    }
}