import Phaser from 'phaser';
import { GAME_CONFIG, COLORS, DIFFICULTY_SETTINGS } from '../config/GameConfig';
import { Player } from '../objects/Player';
import { Enemy } from '../objects/Enemy';
import { Collectible } from '../objects/Collectible';
import { Exit } from '../objects/Exit';
import { Trap } from '../objects/Trap';
import { LevelManager } from '../managers/LevelManager';
import { UIManager } from '../managers/UIManager';
import { AudioManager } from '../managers/AudioManager';
import { AssetGenerator } from '../utils/AssetGenerator';

export class GameScene extends Phaser.Scene {
    private player!: Player;
    private enemies!: Phaser.GameObjects.Group;
    private collectibles!: Phaser.GameObjects.Group;
    private traps!: Phaser.GameObjects.Group;
    private walls!: Phaser.GameObjects.Group;
    private exit!: Exit;
    
    private levelManager!: LevelManager;
    private uiManager!: UIManager;
    private audioManager!: AudioManager;
    
    private currentLevel: number = 1;
    private score: number = 0;
    private lives: number = GAME_CONFIG.PLAYER_LIVES;
    private collectiblesCollected: number = 0;
    private totalCollectibles: number = 0;
    
    private isGameOver: boolean = false;
    private isLevelComplete: boolean = false;
    private isPaused: boolean = false;

    constructor() {
        super({ key: 'GameScene' });
    }

    init(data: { level: number }): void {
        this.currentLevel = data.level || 1;
    }

    preload(): void {
        // 创建游戏资源
        this.createGameAssets();
        
        // 初始化管理器
        this.levelManager = new LevelManager(this);
        this.uiManager = new UIManager(this);
        this.audioManager = new AudioManager(this);
    }

    create(): void {
        this.setupLevel();
        this.setupPhysics();
        this.setupInput();
        this.setupUI();
        this.startGameplay();
    }

    private createGameAssets(): void {
        // 使用新的资源生成器创建像素艺术资源
        const assetGenerator = new AssetGenerator(this);
        assetGenerator.generateAllAssets();
    }

    private setupLevel(): void {
        // 创建游戏对象组
        this.walls = this.add.group();
        this.enemies = this.add.group();
        this.collectibles = this.add.group();
        this.traps = this.add.group();

        // 加载关卡数据
        const levelData = this.levelManager.getLevelData(this.currentLevel);
        
        // 创建关卡布局
        this.createLevelLayout(levelData);
        
        // 创建玩家
        this.createPlayer(levelData.playerStart);
        
        // 创建敌人
        this.createEnemies(levelData.enemyPositions);
        
        // 创建收集品
        this.createCollectibles(levelData.collectiblePositions);
        
        // 创建陷阱
        this.createTraps(levelData.trapPositions);
        
        // 创建出口
        this.createExit(levelData.exitPosition);
    }

    private createLevelLayout(levelData: any): void {
        const { walls } = levelData;
        
        walls.forEach((wallPos: { x: number, y: number }) => {
            const wall = this.add.sprite(wallPos.x, wallPos.y, 'wall-stone');
            wall.setOrigin(0, 0);
            this.walls.add(wall);
        });
    }

    private createPlayer(startPos: { x: number, y: number }): void {
        this.player = new Player(this, startPos.x, startPos.y);
        this.player.setTexture('player');  // 使用新的玩家纹理
        this.player.setWallCollision(this.walls);
    }

    private createEnemies(positions: Array<{ x: number, y: number }>): void {
        positions.forEach((pos, index) => {
            const enemyType = this.getEnemyTypeForLevel(index);
            const enemy = new Enemy(this, pos.x, pos.y, enemyType);
            
            // 使用对应的敌人纹理
            const textureKey = `enemy-${enemyType}`;
            enemy.setTexture(textureKey);
            
            enemy.setWallCollision(this.walls);
            enemy.setTarget(this.player);
            this.enemies.add(enemy);
        });
    }

    private createCollectibles(positions: Array<{ x: number, y: number }>): void {
        this.totalCollectibles = positions.length;
        
        positions.forEach((pos, index) => {
            const collectible = new Collectible(this, pos.x, pos.y);
            
            // 使用不同的收集品纹理
            const textureTypes = ['collectible-coin', 'collectible-gem', 'collectible-key', 'collectible-potion'];
            const textureKey = textureTypes[index % textureTypes.length];
            collectible.setTexture(textureKey);
            
            collectible.onCollected = () => this.onCollectibleCollected();
            this.collectibles.add(collectible);
        });
    }

    private createTraps(positions: Array<{ x: number, y: number }>): void {
        positions.forEach((pos, index) => {
            const trap = new Trap(this, pos.x, pos.y);
            
            // 使用不同的陷阱纹理
            const textureTypes = ['trap-spike', 'trap-fire', 'trap-poison', 'trap-ice'];
            const textureKey = textureTypes[index % textureTypes.length];
            trap.setTexture(textureKey);
            
            trap.onPlayerHit = () => this.onPlayerHitTrap();
            this.traps.add(trap);
        });
    }

    private createExit(position: { x: number, y: number }): void {
        this.exit = new Exit(this, position.x, position.y);
        this.exit.setTexture('exit-portal');  // 使用新的传送门纹理
        this.exit.onPlayerEnter = () => this.onLevelComplete();
    }

    private getEnemyTypeForLevel(index: number): string {
        const difficulty = this.getDifficultyForLevel();
        const settings = DIFFICULTY_SETTINGS[difficulty];
        
        // 根据关卡和索引分配敌人类型
        const types = ['chaser', 'ambusher', 'random', 'boss'];
        return types[index % types.length];
    }

    private getDifficultyForLevel(): keyof typeof DIFFICULTY_SETTINGS {
        switch (this.currentLevel) {
            case 1: return 'EASY';
            case 2: return 'MEDIUM';
            case 3: return 'HARD';
            default: return 'MEDIUM';
        }
    }

    private setupPhysics(): void {
        // 设置碰撞检测
        this.physics.add.overlap(this.player, this.collectibles, 
            (player, collectible) => {
                (collectible as Collectible).collect();
            }
        );

        this.physics.add.overlap(this.player, this.traps,
            (player, trap) => {
                (trap as Trap).activate();
            }
        );

        this.physics.add.overlap(this.player, this.exit,
            (player, exit) => {
                if (this.collectiblesCollected >= this.totalCollectibles) {
                    (exit as Exit).activate();
                }
            }
        );

        this.physics.add.overlap(this.player, this.enemies,
            (player, enemy) => {
                this.onPlayerHitEnemy(enemy as Enemy);
            }
        );

        // 设置攻击事件监听
        this.setupAttackEvents();
    }

    private setupInput(): void {
        // 键盘控制
        this.input.keyboard.on('keydown-P', () => {
            this.togglePause();
        });

        this.input.keyboard.on('keydown-ESC', () => {
            this.togglePause();
        });

        // 移动端触摸控制
        if (this.sys.game.device.input.touch) {
            this.setupTouchControls();
        }
    }

    private setupAttackEvents(): void {
        // 监听玩家攻击事件
        this.events.on('playerAttack', (attackData: { x: number, y: number, range: number }) => {
            this.handlePlayerAttack(attackData);
        });

        // 监听敌人被击杀事件
        this.events.on('enemyKilled', (scoreValue: number) => {
            this.onEnemyKilled(scoreValue);
        });
    }

    private handlePlayerAttack(attackData: { x: number, y: number, range: number }): void {
        const { x, y, range } = attackData;
        
        // 检查攻击范围内的敌人
        this.enemies.children.entries.forEach(enemy => {
            const enemyObj = enemy as Enemy;
            if (enemyObj.isAlive()) {
                const distance = Phaser.Math.Distance.Between(
                    x, y, enemyObj.x, enemyObj.y
                );
                
                if (distance <= range) {
                    // 对敌人造成伤害
                    enemyObj.takeDamage(1);
                    
                    // 播放击中效果
                    this.createHitEffect(enemyObj.x, enemyObj.y);
                    
                    // 播放击中音效
                    this.audioManager.playSound('hit');
                }
            }
        });
    }

    private createHitEffect(x: number, y: number): void {
        // 创建击中特效
        const hitEffect = this.add.circle(x, y, 15, 0xffff00, 0.8);
        
        this.tweens.add({
            targets: hitEffect,
            scaleX: 2,
            scaleY: 2,
            alpha: 0,
            duration: 200,
            ease: 'Power2',
            onComplete: () => {
                hitEffect.destroy();
            }
        });
    }

    private onEnemyKilled(scoreValue: number): void {
        // 增加分数
        this.score += scoreValue;
        
        // 显示得分动画
        this.showScorePopup(scoreValue);
        
        // 更新UI
        this.updateUI();
        
        // 播放击杀音效
        this.audioManager.playSound('enemy_die');
    }

    private showScorePopup(scoreValue: number): void {
        const popup = this.add.text(
            this.player.x,
            this.player.y - 30,
            `+${scoreValue}`,
            {
                fontSize: '20px',
                fontFamily: 'Courier New',
                color: '#f39c12',
                stroke: '#000',
                strokeThickness: 2
            }
        ).setOrigin(0.5);
        
        this.tweens.add({
            targets: popup,
            y: popup.y - 50,
            alpha: 0,
            duration: 1000,
            ease: 'Power2',
            onComplete: () => {
                popup.destroy();
            }
        });
    }

    private setupTouchControls(): void {
        // 创建虚拟摇杆（简化版本）
        this.input.on('pointerdown', (pointer: Phaser.Input.Pointer) => {
            const centerX = GAME_CONFIG.WIDTH / 2;
            const centerY = GAME_CONFIG.HEIGHT / 2;
            
            if (pointer.x < centerX && pointer.y > centerY) {
                // 左下角 - 移动控制
                this.player.startTouchMovement(pointer.x, pointer.y);
            } else if (pointer.x > centerX && pointer.y > centerY) {
                // 右下角 - 攻击按钮
                this.player.attack();
            }
        });

        this.input.on('pointermove', (pointer: Phaser.Input.Pointer) => {
            if (pointer.isDown) {
                this.player.updateTouchMovement(pointer.x, pointer.y);
            }
        });

        this.input.on('pointerup', () => {
            this.player.stopTouchMovement();
        });
    }

    private setupUI(): void {
        this.uiManager.createGameUI();
        this.updateUI();
    }

    private startGameplay(): void {
        // 播放背景音乐
        this.audioManager.playLevelMusic(this.currentLevel);
        
        // 显示关卡开始提示
        this.showLevelStartMessage();
    }

    private showLevelStartMessage(): void {
        const message = this.add.text(
            GAME_CONFIG.WIDTH / 2,
            GAME_CONFIG.HEIGHT / 2,
            `关卡 ${this.currentLevel}\n${this.getLevelObjective()}`,
            {
                fontSize: '32px',
                fontFamily: 'Courier New',
                color: COLORS.TEXT,
                align: 'center',
                backgroundColor: '#000000',
                padding: { x: 20, y: 20 }
            }
        ).setOrigin(0.5);

        this.time.delayedCall(3000, () => {
            message.destroy();
        });
    }

    private getLevelObjective(): string {
        switch (this.currentLevel) {
            case 1: return '收集宝物并到达出口';
            case 2: return '小心陷阱和更多敌人';
            case 3: return '面对最终挑战';
            default: return '完成任务';
        }
    }

    private onCollectibleCollected(): void {
        this.collectiblesCollected++;
        this.score += 100;
        this.audioManager.playSound('collect');
        this.updateUI();

        // 检查是否收集完所有宝物
        if (this.collectiblesCollected >= this.totalCollectibles) {
            this.showMessage('所有宝物已收集！前往出口！');
            this.exit.activate();
        }
    }

    private onPlayerHitTrap(): void {
        if (this.player.isInvincible()) return;
        
        this.lives--;
        this.player.setInvincible(true);
        this.audioManager.playSound('hurt');
        this.updateUI();

        if (this.lives <= 0) {
            this.gameOver();
        }
    }

    private onPlayerHitEnemy(enemy: Enemy): void {
        if (this.player.isInvincible()) return;
        
        this.lives--;
        this.player.setInvincible(true);
        this.audioManager.playSound('hurt');
        this.updateUI();

        if (this.lives <= 0) {
            this.gameOver();
        }
    }

    private onLevelComplete(): void {
        if (this.isLevelComplete) return;
        
        this.isLevelComplete = true;
        this.audioManager.playSound('level_complete');
        
        // 计算奖励分数
        const timeBonus = Math.max(0, 1000 - this.time.now); // 时间奖励
        this.score += timeBonus;
        
        this.showMessage(`关卡完成！奖励分数: ${timeBonus}`);
        
        this.time.delayedCall(2000, () => {
            if (this.currentLevel < GAME_CONFIG.TOTAL_LEVELS) {
                this.nextLevel();
            } else {
                this.gameWin();
            }
        });
    }

    private nextLevel(): void {
        this.scene.start('GameScene', { level: this.currentLevel + 1 });
    }

    private gameOver(): void {
        if (this.isGameOver) return;
        
        this.isGameOver = true;
        this.audioManager.playSound('game_over');
        
        this.time.delayedCall(1000, () => {
            this.scene.start('GameOverScene', { 
                score: this.score, 
                level: this.currentLevel,
                win: false 
            });
        });
    }

    private gameWin(): void {
        this.isGameOver = true;
        this.audioManager.playSound('victory');
        
        this.time.delayedCall(1000, () => {
            this.scene.start('GameOverScene', { 
                score: this.score, 
                level: this.currentLevel,
                win: true 
            });
        });
    }

    private togglePause(): void {
        this.isPaused = !this.isPaused;
        
        if (this.isPaused) {
            this.scene.pause();
            this.showMessage('游戏暂停 - 按P键继续');
        } else {
            this.scene.resume();
        }
    }

    private showMessage(text: string): void {
        const message = this.add.text(
            GAME_CONFIG.WIDTH / 2,
            GAME_CONFIG.HEIGHT / 2,
            text,
            {
                fontSize: '24px',
                fontFamily: 'Courier New',
                color: COLORS.TEXT,
                align: 'center',
                backgroundColor: '#000000',
                padding: { x: 15, y: 10 }
            }
        ).setOrigin(0.5);

        this.time.delayedCall(2000, () => {
            message.destroy();
        });
    }

    private updateUI(): void {
        this.uiManager.updateUI({
            level: this.currentLevel,
            lives: this.lives,
            score: this.score,
            collectibles: this.collectiblesCollected,
            totalCollectibles: this.totalCollectibles
        });
    }

    update(time: number, delta: number): void {
        if (this.isGameOver || this.isPaused) return;

        // 更新玩家
        this.player.update(time, delta);
        
        // 更新敌人
        this.enemies.children.entries.forEach(enemy => {
            (enemy as Enemy).update(time, delta);
        });
        
        // 更新陷阱
        this.traps.children.entries.forEach(trap => {
            (trap as Trap).update(time, delta);
        });
    }
}