"""调度服务骨架。

负责按固定频率推送 `ScheduledTick` 事件到 Orchestrator 队列。
"""

from __future__ import annotations

from typing import Any, Dict, Optional

from apscheduler.schedulers.asyncio import AsyncIOScheduler
import structlog

from lacopro.agent.events import AgentEvent, EventPriority, EventType, EventQueue
from lacopro.agent.trading_state import get_trading_state

logger = structlog.get_logger("scheduler")


class SchedulerService:
    """基于 APScheduler 的异步调度封装。"""

    def __init__(self, queue: EventQueue) -> None:
        self.queue = queue
        self.scheduler = AsyncIOScheduler()
        self._started = False

    def start(self) -> None:
        if self._started:
            logger.warning("scheduler.already_started")
            return
        self.scheduler.start()
        self._started = True
        logger.info("scheduler.start")

    async def shutdown(self) -> None:
        if not self._started:
            return
        logger.info("scheduler.shutdown")
        await self.scheduler.shutdown(wait=False)
        self._started = False

    def schedule_tick(
        self,
        seconds: int,
        *,
        source: str = "scheduler",
        payload: Optional[Dict[str, Any]] = None,
        priority: EventPriority = EventPriority.NORMAL,
    ) -> None:
        """注册一个固定频率的巡检事件。"""

        if seconds <= 0:
            raise ValueError("seconds 必须大于 0")

        async def _emit_tick() -> None:
            # 检查交易状态 - 如果完全暂停，则跳过事件发送
            trading_state = get_trading_state()
            if not trading_state.should_trigger_analysis:
                logger.debug(
                    "scheduler.tick_skipped_paused",
                    mode=trading_state.mode.value,
                    seconds=seconds
                )
                return

            event = AgentEvent.create(
                EventType.SCHEDULED_TICK,
                source=source,
                payload=payload or {"interval": seconds},
                priority=priority,
            )
            await self.queue.put(event)
            logger.debug("scheduler.tick_enqueued", seconds=seconds, event_id=event.event_id)

        self.scheduler.add_job(_emit_tick, "interval", seconds=seconds, id=f"tick_{seconds}")
        logger.info("scheduler.tick_registered", seconds=seconds)


__all__ = ["SchedulerService"]
