"""配置加载模块。

提供一个中心化的 `Settings` 对象，负责读取 `config/.env` 中的所有敏感配置，
并为其他模块提供类型安全的访问方式。
"""

from __future__ import annotations

from functools import lru_cache
from pathlib import Path
from typing import List, Optional

from pydantic import Field
from pydantic_settings import BaseSettings, SettingsConfigDict

BASE_DIR = Path(__file__).resolve().parent.parent
ENV_FILE = BASE_DIR / "config" / ".env"


class Settings(BaseSettings):
    """顶层配置对象。"""

    model_config = SettingsConfigDict(
        env_file=ENV_FILE,
        env_file_encoding="utf-8",
        case_sensitive=False,
        extra="ignore",
    )

    # SSH
    ssh_primary_server: str = Field(..., alias="SSH_PRIMARY_SERVER")
    ssh_lacopro_server: str = Field(..., alias="SSH_LACOPRO_SERVER")

    # Binance
    binance_api_key: str = Field(..., alias="BINANCE_API_KEY")
    binance_secret_key: str = Field(..., alias="BINANCE_SECRET_KEY")

    # Market intelligence
    tokensinsight_api_key: str = Field(..., alias="TOKENSINSIGHT_API_KEY")

    # Tushare / A-share
    tushare_api_token: Optional[str] = Field(default=None, alias="TUSHARE_API_TOKEN")

    # LLM - Yunwu
    yunwu_api_base: str = Field(..., alias="YUNWU_API_BASE")
    yunwu_api_key: str = Field(..., alias="YUNWU_API_KEY")
    yunwu_primary_model: str = Field(..., alias="YUNWU_PRIMARY_MODEL")
    yunwu_secondary_model: str = Field(..., alias="YUNWU_SECONDARY_MODEL")
    yunwu_default_models: str = Field(..., alias="YUNWU_DEFAULT_MODELS")
    yunwu_nano_api_key: str = Field(..., alias="YUNWU_nano_API_KEY")
    yunwu_nano_default_model: str = Field(..., alias="YUNWU_nano_DEFAULT_MODEL")

    # LLM - Doubao
    doubao_api_base: str = Field(..., alias="DOUBAO_API_BASE")
    doubao_api_key: str = Field(..., alias="DOUBAO_API_KEY")
    doubao_default_model: str = Field(..., alias="DOUBAO_DEFAULT_MODEL")

    # Notifications
    telegram_bot_token: str = Field(..., alias="TELEGRAM_BOT_TOKEN")
    telegram_chat_id: str = Field(..., alias="TELEGRAM_CHAT_ID")  # 中文群组（主群组）

    # Multi-language support (optional, for additional language groups)
    telegram_en_chat_id: Optional[str] = Field(default=None, alias="TELEGRAM_EN_CHAT_ID")  # 英文群组（可选）

    @property
    def yunwu_models(self) -> List[str]:
        """将逗号分隔的模型列表转换为 List。"""

        return [m.strip() for m in self.yunwu_default_models.split(",") if m.strip()]


@lru_cache()
def get_settings() -> Settings:
    """使用 LRU 缓存确保配置对象在进程内只初始化一次。"""

    if not ENV_FILE.exists():
        raise FileNotFoundError("config/.env 不存在，请先添加配置文件")
    return Settings()


__all__ = ["Settings", "get_settings"]
