"""Function Calling 架构 - 支持LLM智能工具调用和链式推理。

这个模块实现了类似于OpenAI Function Calling的能力，让LLM能够：
1. 智能选择和调用工具
2. 进行多轮推理和对话
3. 链式执行复杂任务
4. 处理工具调用错误和重试
"""

from __future__ import annotations

import json
import inspect
from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from datetime import datetime, timezone
from enum import Enum
from typing import Any, Dict, List, Optional, Union, Callable, get_type_hints
from uuid import uuid4

import structlog

logger = structlog.get_logger("function_calling")


class ToolCallStatus(str, Enum):
    """工具调用状态。"""
    PENDING = "pending"
    RUNNING = "running" 
    SUCCESS = "success"
    FAILED = "failed"
    TIMEOUT = "timeout"


class ParameterType(str, Enum):
    """参数类型枚举。"""
    STRING = "string"
    INTEGER = "integer"
    NUMBER = "number"
    BOOLEAN = "boolean"
    ARRAY = "array"
    OBJECT = "object"


class PermissionLevel(str):
    """权限级别。"""
    READ_ONLY = "read_only"      # 只读操作
    MARKET_DATA = "market_data"  # 市场数据
    PORTFOLIO = "portfolio"      # 投资组合查询
    TRADING = "trading"          # 交易操作
    RISK = "risk"               # 风险管理
    ADMIN = "admin"             # 管理员


@dataclass
class ToolParameter:
    """工具参数定义。"""
    name: str
    type: ParameterType
    description: str
    required: bool = True
    default: Any = None
    enum: Optional[List[Any]] = None
    minimum: Optional[float] = None
    maximum: Optional[float] = None


@dataclass
class ToolFunction:
    """工具函数定义。"""
    name: str
    description: str
    parameters: List[ToolParameter]
    function: Callable
    category: str = "general"
    risk_level: str = "low"  # low, medium, high
    requires_confirmation: bool = False
    
    def to_openai_format(self) -> Dict[str, Any]:
        """转换为OpenAI Function Calling格式。"""
        properties = {}
        required = []
        
        for param in self.parameters:
            properties[param.name] = {
                "type": param.type.value,
                "description": param.description
            }
            if param.enum:
                properties[param.name]["enum"] = param.enum
            if param.minimum is not None:
                properties[param.name]["minimum"] = param.minimum
            if param.maximum is not None:
                properties[param.name]["maximum"] = param.maximum
                
            if param.required:
                required.append(param.name)
        
        return {
            "type": "function",
            "function": {
                "name": self.name,
                "description": self.description,
                "parameters": {
                    "type": "object",
                    "properties": properties,
                    "required": required
                }
            }
        }


@dataclass
class ToolCall:
    """工具调用记录。"""
    id: str = field(default_factory=lambda: str(uuid4()))
    function_name: str = ""
    arguments: Dict[str, Any] = field(default_factory=dict)
    status: ToolCallStatus = ToolCallStatus.PENDING
    result: Any = None
    error: Optional[str] = None
    started_at: Optional[datetime] = None
    completed_at: Optional[datetime] = None
    execution_time_ms: Optional[int] = None
    
    def start(self) -> None:
        """开始执行。"""
        self.status = ToolCallStatus.RUNNING
        self.started_at = datetime.now(timezone.utc)
    
    def complete(self, result: Any) -> None:
        """完成执行。"""
        self.status = ToolCallStatus.SUCCESS
        self.result = result
        self.completed_at = datetime.now(timezone.utc)
        if self.started_at:
            delta = self.completed_at - self.started_at
            self.execution_time_ms = int(delta.total_seconds() * 1000)
    
    def fail(self, error: str) -> None:
        """执行失败。"""
        self.status = ToolCallStatus.FAILED
        self.error = error
        self.completed_at = datetime.now(timezone.utc)
        if self.started_at:
            delta = self.completed_at - self.started_at
            self.execution_time_ms = int(delta.total_seconds() * 1000)


@dataclass
class ConversationTurn:
    """对话轮次。"""
    id: str = field(default_factory=lambda: str(uuid4()))
    role: str = "assistant"  # user, assistant, tool
    content: str = ""
    tool_calls: List[ToolCall] = field(default_factory=list)
    timestamp: datetime = field(default_factory=lambda: datetime.now(timezone.utc))
    reasoning: Optional[str] = None  # LLM的推理过程


class ToolRegistry:
    """工具注册中心。"""
    
    def __init__(self):
        self._tools: Dict[str, ToolFunction] = {}
        self._categories: Dict[str, List[str]] = {}
    
    def register_tool(self, tool: ToolFunction) -> None:
        """注册工具。"""
        self._tools[tool.name] = tool
        
        if tool.category not in self._categories:
            self._categories[tool.category] = []
        self._categories[tool.category].append(tool.name)
        
        logger.info(
            "tool_registered",
            name=tool.name,
            category=tool.category,
            risk_level=tool.risk_level
        )
    
    def get_tool(self, name: str) -> Optional[ToolFunction]:
        """获取工具。"""
        return self._tools.get(name)
    
    def list_tools(self, category: Optional[str] = None) -> List[ToolFunction]:
        """列出工具。"""
        if category:
            tool_names = self._categories.get(category, [])
            return [self._tools[name] for name in tool_names]
        return list(self._tools.values())
    
    def get_openai_tools(self, category: Optional[str] = None) -> List[Dict[str, Any]]:
        """获取OpenAI格式的工具定义。"""
        tools = self.list_tools(category)
        return [tool.to_openai_format() for tool in tools]


def auto_register_tool(
    name: Optional[str] = None,
    description: Optional[str] = None,
    category: str = "general",
    risk_level: str = "low",
    requires_confirmation: bool = False
):
    """装饰器：自动注册工具函数。"""
    def decorator(func: Callable) -> Callable:
        # 获取函数签名和类型提示
        sig = inspect.signature(func)
        type_hints = get_type_hints(func)
        
        # 解析参数
        parameters = []
        for param_name, param in sig.parameters.items():
            if param_name == 'self':
                continue
                
            param_type = _python_type_to_parameter_type(type_hints.get(param_name, str))
            param_desc = f"Parameter {param_name}"
            
            # 从docstring中提取参数描述（简化版）
            if func.__doc__:
                lines = func.__doc__.split('\n')
                for line in lines:
                    if f"{param_name}:" in line:
                        param_desc = line.split(':', 1)[1].strip()
                        break
            
            parameters.append(ToolParameter(
                name=param_name,
                type=param_type,
                description=param_desc,
                required=param.default == inspect.Parameter.empty
            ))
        
        # 创建工具定义
        tool_name = name or func.__name__
        tool_desc = description or func.__doc__ or f"Function {tool_name}"
        
        tool = ToolFunction(
            name=tool_name,
            description=tool_desc,
            parameters=parameters,
            function=func,
            category=category,
            risk_level=risk_level,
            requires_confirmation=requires_confirmation
        )
        
        # 注册到全局注册中心
        global_registry.register_tool(tool)
        
        return func
    
    return decorator


def _python_type_to_parameter_type(python_type: type) -> ParameterType:
    """将Python类型转换为参数类型。"""
    if python_type == str:
        return ParameterType.STRING
    elif python_type == int:
        return ParameterType.INTEGER
    elif python_type in (float, int):
        return ParameterType.NUMBER
    elif python_type == bool:
        return ParameterType.BOOLEAN
    elif python_type == list:
        return ParameterType.ARRAY
    elif python_type == dict:
        return ParameterType.OBJECT
    else:
        return ParameterType.STRING


# 全局工具注册中心
global_registry = ToolRegistry()


__all__ = [
    "ToolCallStatus",
    "ParameterType",
    "PermissionLevel",
    "ToolParameter",
    "ToolFunction",
    "ToolCall",
    "ConversationTurn",
    "ToolRegistry",
    "auto_register_tool",
    "global_registry"
]