"""A 股交易工具。

提供完整的 A 股交易功能，集成风控规则、T+1 验证、涨跌停检查等。
"""

from __future__ import annotations

from dataclasses import dataclass, field
from datetime import date, datetime
from typing import Any, Dict, List, Optional

import structlog

from laca.adapters.broker import (
    Order,
    OrderSide,
    OrderStatus,
    OrderType,
    SimulatedBrokerClient,
)
from laca.adapters.data.market_data_service import MarketDataService
from laca.risk.ashare_rules import (
    CashUsageRule,
    ConcentrationRule,
    LimitPriceRule,
    SuspensionRule,
    T1SellRule,
    TradingTimeRule,
)
from laca.risk.models import RiskAlert, RiskSeverity
from laca.scheduler.trade_calendar import TradeCalendar
from laca.tools.ashare_portfolio_tool import ASharePortfolioTool

logger = structlog.get_logger(__name__)


class AShareTradingError(Exception):
    """A 股交易工具异常。"""


@dataclass
class TradeResult:
    """交易结果。

    Attributes:
        success: 是否成功
        order: 订单对象
        risk_alerts: 风险告警列表
        message: 结果消息
        details: 详细信息
    """

    success: bool
    order: Optional[Order] = None
    risk_alerts: List[RiskAlert] = field(default_factory=list)
    message: str = ""
    details: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> dict:
        """转换为字典。"""
        return {
            "success": self.success,
            "order": self.order.to_dict() if self.order else None,
            "risk_alerts": [
                {
                    "rule_id": alert.rule_id,
                    "severity": alert.severity.value,
                    "message": alert.message,
                    "details": alert.details,
                }
                for alert in self.risk_alerts
            ],
            "message": self.message,
            "details": self.details,
        }


class AShareTradingTool:
    """A 股交易工具。

    提供完整的交易功能，包括风控检查、订单执行、撮合等。

    Attributes:
        broker_client: 券商客户端
        portfolio_tool: 组合工具
        market_data_service: 行情服务
        trade_calendar: 交易日历
        enable_risk_check: 是否启用风控检查
        auto_match: 是否自动撮合（仅模拟券商）
    """

    def __init__(
        self,
        broker_client: Optional[SimulatedBrokerClient] = None,
        portfolio_tool: Optional[ASharePortfolioTool] = None,
        market_data_service: Optional[MarketDataService] = None,
        trade_calendar: Optional[TradeCalendar] = None,
        enable_risk_check: bool = True,
        auto_match: bool = True,
    ):
        """初始化 A 股交易工具。

        Args:
            broker_client: 券商客户端
            portfolio_tool: 组合工具
            market_data_service: 行情服务
            trade_calendar: 交易日历
            enable_risk_check: 是否启用风控检查
            auto_match: 是否自动撮合
        """
        self.broker_client = broker_client or SimulatedBrokerClient()
        self.portfolio_tool = portfolio_tool or ASharePortfolioTool(
            broker_client=self.broker_client
        )
        self.market_data_service = market_data_service or MarketDataService()
        self.trade_calendar = trade_calendar or TradeCalendar()
        self.enable_risk_check = enable_risk_check
        self.auto_match = auto_match

        # 初始化风控规则
        self._init_risk_rules()

        logger.info(
            "ashare_trading_tool_initialized",
            enable_risk_check=enable_risk_check,
            auto_match=auto_match,
        )

    def _init_risk_rules(self):
        """初始化风控规则。"""
        self.limit_price_rule = LimitPriceRule(
            market_data_service=self.market_data_service
        )
        self.concentration_rule = ConcentrationRule(
            max_single_position_ratio=0.20, warning_ratio=0.15
        )
        self.cash_usage_rule = CashUsageRule(
            min_cash_ratio=0.10, warning_cash_ratio=0.20
        )
        self.trading_time_rule = TradingTimeRule(
            trade_calendar=self.trade_calendar, allow_call_auction=True
        )
        self.suspension_rule = SuspensionRule(
            market_data_service=self.market_data_service
        )
        self.t1_sell_rule = T1SellRule(portfolio_tool=self.portfolio_tool)

    def buy(
        self,
        ts_code: str,
        price: float,
        quantity: int,
        order_type: OrderType = OrderType.LIMIT,
        skip_risk_check: bool = False,
    ) -> TradeResult:
        """买入股票。

        Args:
            ts_code: 股票代码
            price: 买入价格
            quantity: 买入数量
            order_type: 订单类型
            skip_risk_check: 是否跳过风控检查

        Returns:
            交易结果
        """
        risk_alerts: List[RiskAlert] = []

        # 风控检查
        if self.enable_risk_check and not skip_risk_check:
            risk_alerts = self._pre_trade_check(
                ts_code=ts_code,
                side=OrderSide.BUY,
                price=price,
                quantity=quantity,
            )

            # 检查是否有 CRITICAL 级别告警
            critical_alerts = [
                a for a in risk_alerts if a.severity == RiskSeverity.CRITICAL
            ]
            if critical_alerts:
                return TradeResult(
                    success=False,
                    risk_alerts=risk_alerts,
                    message=f"风控检查失败: {critical_alerts[0].message}",
                    details={
                        "ts_code": ts_code,
                        "price": price,
                        "quantity": quantity,
                    },
                )

        # 下单
        try:
            order = self.broker_client.place_order(
                ts_code=ts_code,
                side=OrderSide.BUY,
                price=price,
                quantity=quantity,
                order_type=order_type,
            )

            logger.info(
                "buy_order_placed",
                order_id=order.order_id,
                ts_code=ts_code,
                price=price,
                quantity=quantity,
            )

            # 自动撮合
            if self.auto_match and order.status == OrderStatus.PENDING:
                self._auto_match(order)

            return TradeResult(
                success=True,
                order=order,
                risk_alerts=risk_alerts,
                message=f"买入订单已提交: {order.order_id}",
                details={
                    "order_id": order.order_id,
                    "status": order.status.value,
                    "frozen_cash": order.frozen_cash,
                },
            )

        except Exception as e:
            logger.error("buy_order_failed", ts_code=ts_code, error=str(e))
            return TradeResult(
                success=False,
                risk_alerts=risk_alerts,
                message=f"下单失败: {e}",
                details={"ts_code": ts_code, "price": price, "quantity": quantity},
            )

    def sell(
        self,
        ts_code: str,
        price: float,
        quantity: int,
        order_type: OrderType = OrderType.LIMIT,
        skip_risk_check: bool = False,
    ) -> TradeResult:
        """卖出股票。

        Args:
            ts_code: 股票代码
            price: 卖出价格
            quantity: 卖出数量
            order_type: 订单类型
            skip_risk_check: 是否跳过风控检查

        Returns:
            交易结果
        """
        risk_alerts: List[RiskAlert] = []

        # 风控检查
        if self.enable_risk_check and not skip_risk_check:
            risk_alerts = self._pre_trade_check(
                ts_code=ts_code,
                side=OrderSide.SELL,
                price=price,
                quantity=quantity,
            )

            # 检查是否有 CRITICAL 级别告警
            critical_alerts = [
                a for a in risk_alerts if a.severity == RiskSeverity.CRITICAL
            ]
            if critical_alerts:
                return TradeResult(
                    success=False,
                    risk_alerts=risk_alerts,
                    message=f"风控检查失败: {critical_alerts[0].message}",
                    details={
                        "ts_code": ts_code,
                        "price": price,
                        "quantity": quantity,
                    },
                )

        # 下单
        try:
            order = self.broker_client.place_order(
                ts_code=ts_code,
                side=OrderSide.SELL,
                price=price,
                quantity=quantity,
                order_type=order_type,
            )

            logger.info(
                "sell_order_placed",
                order_id=order.order_id,
                ts_code=ts_code,
                price=price,
                quantity=quantity,
            )

            # 自动撮合
            if self.auto_match and order.status == OrderStatus.PENDING:
                self._auto_match(order)

            return TradeResult(
                success=True,
                order=order,
                risk_alerts=risk_alerts,
                message=f"卖出订单已提交: {order.order_id}",
                details={
                    "order_id": order.order_id,
                    "status": order.status.value,
                    "frozen_quantity": order.frozen_quantity,
                },
            )

        except Exception as e:
            logger.error("sell_order_failed", ts_code=ts_code, error=str(e))
            return TradeResult(
                success=False,
                risk_alerts=risk_alerts,
                message=f"下单失败: {e}",
                details={"ts_code": ts_code, "price": price, "quantity": quantity},
            )

    def cancel_order(self, order_id: str) -> TradeResult:
        """撤单。

        Args:
            order_id: 订单 ID

        Returns:
            交易结果
        """
        try:
            success = self.broker_client.cancel_order(order_id)

            if success:
                order = self.broker_client.orders.get(order_id)
                return TradeResult(
                    success=True,
                    order=order,
                    message=f"订单已撤销: {order_id}",
                    details={"order_id": order_id},
                )
            else:
                return TradeResult(
                    success=False,
                    message=f"撤单失败: {order_id}",
                    details={"order_id": order_id},
                )

        except Exception as e:
            logger.error("cancel_order_failed", order_id=order_id, error=str(e))
            return TradeResult(
                success=False,
                message=f"撤单失败: {e}",
                details={"order_id": order_id},
            )

    def _pre_trade_check(
        self,
        ts_code: str,
        side: OrderSide,
        price: float,
        quantity: int,
        check_date: Optional[date] = None,
        check_time: Optional[datetime] = None,
    ) -> List[RiskAlert]:
        """交易前风控检查。

        Args:
            ts_code: 股票代码
            side: 买卖方向
            price: 价格
            quantity: 数量
            check_date: 检查日期
            check_time: 检查时间

        Returns:
            风险告警列表
        """
        if check_date is None:
            check_date = date.today()

        if check_time is None:
            check_time = datetime.now()

        alerts: List[RiskAlert] = []

        # 1. 交易时间检查
        alerts.extend(self.trading_time_rule.validate_trading_time(check_time))

        # 2. 停牌检查
        alerts.extend(
            self.suspension_rule.validate_suspension(ts_code, check_date)
        )

        # 3. 涨跌停价格检查
        alerts.extend(
            self.limit_price_rule.validate_price(ts_code, side, price, check_date)
        )

        # 4. 买单：资金检查
        if side == OrderSide.BUY:
            validation = self.portfolio_tool.validate_buy_order(
                ts_code, price, quantity
            )
            if not validation["valid"]:
                alerts.append(
                    RiskAlert(
                        rule_id="buy_validation",
                        severity=RiskSeverity.CRITICAL,
                        message=validation["reason"],
                        details=validation,
                    )
                )

        # 5. 卖单：T+1 检查
        elif side == OrderSide.SELL:
            validation = self.portfolio_tool.validate_sell_order(
                ts_code, quantity, check_date
            )
            if not validation["valid"]:
                alerts.append(
                    RiskAlert(
                        rule_id="sell_validation",
                        severity=RiskSeverity.CRITICAL,
                        message=validation["reason"],
                        details=validation,
                    )
                )

        # 6. 账户快照检查（集中度、现金使用率）
        snapshot = self.portfolio_tool.fetch_snapshot()

        # 集中度检查（模拟买入后的情况）
        if side == OrderSide.BUY:
            # 简化：直接检查当前集中度
            concentration_alerts = self.concentration_rule.evaluate(snapshot)
            alerts.extend(concentration_alerts)

        # 现金使用率检查
        cash_alerts = self.cash_usage_rule.evaluate(snapshot)
        alerts.extend(cash_alerts)

        return alerts

    def _auto_match(self, order: Order) -> bool:
        """自动撮合订单（仅模拟券商）。

        Args:
            order: 订单对象

        Returns:
            是否成功撮合
        """
        try:
            # 获取市场价格
            market_price = self.market_data_service.get_latest_price(order.ts_code)
            if market_price is None:
                logger.warning(
                    "auto_match_no_price",
                    order_id=order.order_id,
                    ts_code=order.ts_code,
                )
                return False

            # 撮合
            matched = self.broker_client.match_order(order.order_id, market_price)
            if matched:
                logger.info(
                    "order_auto_matched",
                    order_id=order.order_id,
                    market_price=market_price,
                )
            return matched

        except Exception as e:
            logger.error(
                "auto_match_failed",
                order_id=order.order_id,
                error=str(e),
            )
            return False

    def get_order(self, order_id: str) -> Optional[Order]:
        """查询订单。

        Args:
            order_id: 订单 ID

        Returns:
            订单对象
        """
        return self.broker_client.orders.get(order_id)

    def list_orders(
        self,
        status: Optional[OrderStatus] = None,
        ts_code: Optional[str] = None,
    ) -> List[Order]:
        """查询订单列表。

        Args:
            status: 订单状态过滤
            ts_code: 股票代码过滤

        Returns:
            订单列表
        """
        return self.broker_client.get_orders(status, ts_code)

    def get_account_summary(self) -> dict:
        """获取账户摘要。

        Returns:
            账户摘要字典
        """
        return self.broker_client.get_account_summary()

    async def close(self) -> None:
        """关闭交易工具（清理资源）。

        A股交易工具当前无需特殊清理，保留此方法以兼容 Orchestrator。
        """
        logger.debug("ashare_trading_tool.close")


__all__ = [
    "AShareTradingTool",
    "TradeResult",
    "AShareTradingError",
]
