"""A 股账户与持仓工具。

提供账户快照、持仓查询、可用资金计算等功能，支持 T+1 规则。
"""

from __future__ import annotations

from dataclasses import dataclass, field
from datetime import date, datetime
from typing import Any, Dict, List, Optional

import structlog

from laca.adapters.broker import (
    Account,
    Order,
    OrderSide,
    OrderStatus,
    Position,
    SimulatedBrokerClient,
)
from laca.adapters.data.market_data_service import MarketDataService
from laca.scheduler.trade_calendar import TradeCalendar

logger = structlog.get_logger(__name__)


class ASharePortfolioError(Exception):
    """A 股组合工具异常。"""


@dataclass
class PortfolioSnapshot:
    """账户快照。

    Attributes:
        account: 账户信息
        positions: 持仓列表
        pending_orders: 待成交订单列表
        total_assets: 总资产（现金 + 持仓市值）
        available_cash: 可用资金
        frozen_cash: 冻结资金
        market_value: 持仓市值
        cost_value: 持仓成本
        unrealized_pnl: 未实现盈亏
        unrealized_pnl_pct: 未实现盈亏比例
        snapshot_time: 快照时间
    """

    account: Account
    positions: List[Position]
    pending_orders: List[Order]
    total_assets: float
    available_cash: float
    frozen_cash: float
    market_value: float
    cost_value: float
    unrealized_pnl: float
    unrealized_pnl_pct: float
    snapshot_time: datetime = field(default_factory=datetime.now)

    def to_dict(self) -> dict:
        """转换为字典。"""
        return {
            "account": self.account.to_dict(),
            "positions": [p.to_dict() for p in self.positions],
            "pending_orders": [o.to_dict() for o in self.pending_orders],
            "total_assets": self.total_assets,
            "available_cash": self.available_cash,
            "frozen_cash": self.frozen_cash,
            "market_value": self.market_value,
            "cost_value": self.cost_value,
            "unrealized_pnl": self.unrealized_pnl,
            "unrealized_pnl_pct": self.unrealized_pnl_pct,
            "snapshot_time": self.snapshot_time.isoformat(),
        }


class ASharePortfolioTool:
    """A 股账户与持仓工具。

    提供账户快照、持仓查询、可用资金计算等功能，支持 T+1 规则。

    Attributes:
        broker_client: 券商客户端
        market_data_service: 行情服务
        trade_calendar: 交易日历
    """

    def __init__(
        self,
        broker_client: Optional[SimulatedBrokerClient] = None,
        market_data_service: Optional[MarketDataService] = None,
        trade_calendar: Optional[TradeCalendar] = None,
    ):
        """初始化 A 股组合工具。

        Args:
            broker_client: 券商客户端
            market_data_service: 行情服务
            trade_calendar: 交易日历
        """
        self.broker_client = broker_client or SimulatedBrokerClient()
        self.market_data_service = market_data_service or MarketDataService()
        self.trade_calendar = trade_calendar or TradeCalendar()

        logger.info("ashare_portfolio_tool_initialized")

    def fetch_snapshot(self) -> PortfolioSnapshot:
        """获取账户快照。

        Returns:
            账户快照对象
        """
        # 更新市场价格
        self.broker_client.update_market_prices()

        # 获取账户摘要
        summary = self.broker_client.get_account_summary()

        # 获取持仓
        positions = self.broker_client.get_positions()

        # 获取待成交订单
        pending_orders = self.broker_client.get_orders(status=OrderStatus.PENDING)

        # 计算总资产
        market_value = summary["total_market_value"]
        cost_value = summary["total_cost_value"]
        unrealized_pnl = summary["total_unrealized_pnl"]
        total_assets = summary["total_assets"]

        # 计算盈亏比例
        if cost_value > 0:
            unrealized_pnl_pct = (unrealized_pnl / cost_value) * 100
        else:
            unrealized_pnl_pct = 0.0

        snapshot = PortfolioSnapshot(
            account=self.broker_client.account,
            positions=positions,
            pending_orders=pending_orders,
            total_assets=total_assets,
            available_cash=self.broker_client.account.available_cash,
            frozen_cash=self.broker_client.account.frozen_cash,
            market_value=market_value,
            cost_value=cost_value,
            unrealized_pnl=unrealized_pnl,
            unrealized_pnl_pct=unrealized_pnl_pct,
        )

        logger.info(
            "snapshot_fetched",
            total_assets=total_assets,
            positions=len(positions),
            pending_orders=len(pending_orders),
        )

        return snapshot

    def get_available_cash(self) -> float:
        """获取可用资金（扣除冻结）。

        Returns:
            可用资金金额
        """
        return self.broker_client.account.usable_cash

    def get_position(self, ts_code: str) -> Optional[Position]:
        """获取指定股票持仓。

        Args:
            ts_code: 股票代码

        Returns:
            持仓对象，若无持仓返回 None
        """
        return self.broker_client.get_position(ts_code)

    def get_all_positions(self) -> List[Position]:
        """获取所有持仓。

        Returns:
            持仓列表
        """
        return self.broker_client.get_positions()

    def check_sellable_quantity(
        self,
        ts_code: str,
        check_date: Optional[date] = None,
    ) -> int:
        """检查可卖数量（考虑 T+1 规则）。

        Args:
            ts_code: 股票代码
            check_date: 检查日期（默认今天）

        Returns:
            可卖数量（考虑 T+1 和冻结）
        """
        if check_date is None:
            check_date = date.today()

        position = self.broker_client.get_position(ts_code)
        if not position:
            return 0

        # T+1 检查：如果最后买入日期是今天，则 available_quantity 为 0
        if position.last_buy_date and position.last_buy_date >= check_date:
            logger.debug(
                "t1_restriction",
                ts_code=ts_code,
                last_buy_date=position.last_buy_date,
                check_date=check_date,
            )
            return 0

        return position.sellable_quantity

    def calculate_position_size(
        self,
        ts_code: str,
        current_price: float,
        risk_level: str = "MEDIUM",
        max_position_ratio: float = 0.20,
        max_single_amount: Optional[float] = None,
    ) -> Dict[str, Any]:
        """计算建议仓位大小（A 股规则）。

        Args:
            ts_code: 股票代码
            current_price: 当前价格
            risk_level: 风险等级（HIGH/MEDIUM/LOW）
            max_position_ratio: 单只股票最大仓位比例（默认 20%）
            max_single_amount: 单笔最大买入金额（None 则无限制）

        Returns:
            {
                "suggested_amount": 建议买入金额,
                "suggested_quantity": 建议买入数量,
                "position_ratio": 占总资产比例,
                "usable_cash": 可用资金,
                "reason": 计算理由
            }
        """
        # 获取可用资金
        usable_cash = self.get_available_cash()

        # 获取总资产
        summary = self.broker_client.get_account_summary()
        total_assets = summary["total_assets"]

        # 风险等级对应的仓位比例
        risk_ratios = {
            "LOW": 0.15,  # 15%
            "MEDIUM": 0.10,  # 10%
            "HIGH": 0.05,  # 5%
        }
        base_ratio = risk_ratios.get(risk_level.upper(), 0.10)

        # 检查现有持仓
        position = self.get_position(ts_code)
        current_position_value = position.market_value if position else 0.0

        # 计算可追加金额
        max_position_value = total_assets * max_position_ratio
        available_for_this_stock = max_position_value - current_position_value

        # 计算建议金额
        suggested_amount = min(
            usable_cash * base_ratio,  # 基于风险等级
            available_for_this_stock,  # 不超过单只股票上限
        )

        # 单笔限额
        if max_single_amount and suggested_amount > max_single_amount:
            suggested_amount = max_single_amount

        # 确保非负
        suggested_amount = max(0.0, suggested_amount)

        # 计算建议数量（A 股买入单位：100 股的整数倍）
        if current_price > 0:
            raw_quantity = suggested_amount / current_price
            suggested_quantity = int(raw_quantity / 100) * 100  # 向下取整到100的倍数
            suggested_amount = suggested_quantity * current_price
        else:
            suggested_quantity = 0

        # 计算实际占比
        position_ratio = (
            suggested_amount / total_assets if total_assets > 0 else 0.0
        )

        # 生成理由
        reason_parts = []
        reason_parts.append(f"风险等级 {risk_level}")

        if position:
            current_ratio = (current_position_value / total_assets) * 100
            reason_parts.append(f"现有持仓占比 {current_ratio:.1f}%")

        if suggested_quantity == 0:
            if usable_cash < 100 * current_price:
                reason_parts.append("可用资金不足")
            elif available_for_this_stock <= 0:
                reason_parts.append("已达单只股票仓位上限")
            else:
                reason_parts.append("计算结果不足100股")

        reason = ", ".join(reason_parts)

        result = {
            "suggested_amount": round(suggested_amount, 2),
            "suggested_quantity": suggested_quantity,
            "position_ratio": round(position_ratio, 4),
            "usable_cash": round(usable_cash, 2),
            "reason": reason,
        }

        logger.info(
            "position_size_calculated",
            ts_code=ts_code,
            **result,
        )

        return result

    def get_concentration_risk(self) -> Dict[str, Any]:
        """计算持仓集中度风险。

        Returns:
            {
                "top_positions": [{"ts_code": "000001.SZ", "ratio": 0.25, ...}],
                "concentration_ratio": 0.60,  # 前3大持仓占比
                "is_high_risk": True,  # 集中度是否过高
                "warning": "前3大持仓占比60%，建议分散"
            }
        """
        positions = self.get_all_positions()
        if not positions:
            return {
                "top_positions": [],
                "concentration_ratio": 0.0,
                "is_high_risk": False,
                "warning": "",
            }

        # 获取总资产
        summary = self.broker_client.get_account_summary()
        total_assets = summary["total_assets"]

        # 计算每只股票占比
        position_ratios = []
        for position in positions:
            ratio = position.market_value / total_assets if total_assets > 0 else 0.0
            position_ratios.append(
                {
                    "ts_code": position.ts_code,
                    "market_value": position.market_value,
                    "ratio": ratio,
                    "pnl_pct": position.unrealized_pnl_pct,
                }
            )

        # 按市值降序排序
        position_ratios.sort(key=lambda x: x["market_value"], reverse=True)

        # 计算前3大持仓占比
        top3_ratio = sum(p["ratio"] for p in position_ratios[:3])

        # 风险判断：前3大持仓占比超过 60% 视为高风险
        is_high_risk = top3_ratio > 0.60

        warning = ""
        if is_high_risk:
            warning = f"前3大持仓占比 {top3_ratio*100:.1f}%，建议分散投资"

        return {
            "top_positions": position_ratios[:5],  # 返回前5大持仓
            "concentration_ratio": round(top3_ratio, 4),
            "is_high_risk": is_high_risk,
            "warning": warning,
        }

    def validate_buy_order(
        self,
        ts_code: str,
        price: float,
        quantity: int,
    ) -> Dict[str, Any]:
        """验证买单是否可行（资金、限额等）。

        Args:
            ts_code: 股票代码
            price: 买入价格
            quantity: 买入数量

        Returns:
            {
                "valid": True/False,
                "reason": "验证失败原因",
                "required_cash": 所需资金,
                "available_cash": 可用资金,
            }
        """
        # 计算所需资金
        amount = price * quantity
        commission = self.broker_client.account.calculate_commission(amount)
        required_cash = amount + commission

        # 获取可用资金
        available_cash = self.get_available_cash()

        # 验证资金
        if required_cash > available_cash:
            return {
                "valid": False,
                "reason": f"资金不足：需要 {required_cash:.2f}，可用 {available_cash:.2f}",
                "required_cash": required_cash,
                "available_cash": available_cash,
            }

        # 验证数量（100 股的整数倍）
        if quantity % 100 != 0:
            return {
                "valid": False,
                "reason": f"买入数量必须是 100 的整数倍，当前 {quantity}",
                "required_cash": required_cash,
                "available_cash": available_cash,
            }

        # 验证价格
        if price <= 0:
            return {
                "valid": False,
                "reason": f"价格必须为正数，当前 {price}",
                "required_cash": required_cash,
                "available_cash": available_cash,
            }

        return {
            "valid": True,
            "reason": "验证通过",
            "required_cash": required_cash,
            "available_cash": available_cash,
        }

    def validate_sell_order(
        self,
        ts_code: str,
        quantity: int,
        check_date: Optional[date] = None,
    ) -> Dict[str, Any]:
        """验证卖单是否可行（T+1、可卖数量等）。

        Args:
            ts_code: 股票代码
            quantity: 卖出数量
            check_date: 检查日期

        Returns:
            {
                "valid": True/False,
                "reason": "验证失败原因",
                "sellable_quantity": 可卖数量,
            }
        """
        if check_date is None:
            check_date = date.today()

        # 获取可卖数量
        sellable_quantity = self.check_sellable_quantity(ts_code, check_date)

        # 验证数量
        if quantity > sellable_quantity:
            return {
                "valid": False,
                "reason": f"可卖数量不足：需要 {quantity}，可卖 {sellable_quantity}（T+1 规则）",
                "sellable_quantity": sellable_quantity,
            }

        # 验证数量（100 股的整数倍，或清仓）
        position = self.get_position(ts_code)
        if position and quantity != position.total_quantity and quantity % 100 != 0:
            return {
                "valid": False,
                "reason": f"卖出数量必须是 100 的整数倍或全部卖出，当前 {quantity}",
                "sellable_quantity": sellable_quantity,
            }

        return {
            "valid": True,
            "reason": "验证通过",
            "sellable_quantity": sellable_quantity,
        }


__all__ = [
    "ASharePortfolioTool",
    "PortfolioSnapshot",
    "ASharePortfolioError",
]
