"""调度服务骨架。

负责按固定频率推送 `ScheduledTick` 事件到 Orchestrator 队列。
"""

from __future__ import annotations

from datetime import datetime
from typing import Any, Dict, Optional

from apscheduler.schedulers.asyncio import AsyncIOScheduler
import structlog

from laca.agent.events import AgentEvent, EventPriority, EventType, EventQueue
from laca.agent.trading_state import TradingMode, get_trading_state

try:  # 仅在运行时导入，避免在文档构建中增加依赖
    from laca.scheduler.trade_calendar import TradeCalendar
except Exception:  # pragma: no cover - fallback
    TradeCalendar = Any  # type: ignore

logger = structlog.get_logger("scheduler")


class SchedulerService:
    """基于 APScheduler 的异步调度封装。"""

    def __init__(self, queue: EventQueue) -> None:
        self.queue = queue
        self.scheduler = AsyncIOScheduler()
        self._started = False
        self._adaptive_job_id: Optional[str] = None
        self._adaptive_monitor_job_id: Optional[str] = None
        self._adaptive_interval: Optional[int] = None

    def start(self) -> None:
        if self._started:
            logger.warning("scheduler.already_started")
            return
        self.scheduler.start()
        self._started = True
        logger.info("scheduler.start")

    async def shutdown(self) -> None:
        if not self._started:
            return
        logger.info("scheduler.shutdown")
        await self.scheduler.shutdown(wait=False)
        self._started = False

    def schedule_tick(
        self,
        seconds: int,
        *,
        source: str = "scheduler",
        payload: Optional[Dict[str, Any]] = None,
        priority: EventPriority = EventPriority.NORMAL,
    ) -> None:
        """注册一个固定频率的巡检事件。"""

        if seconds <= 0:
            raise ValueError("seconds 必须大于 0")

        async def _emit_tick() -> None:
            # 检查交易状态 - 如果完全暂停，则跳过事件发送
            trading_state = get_trading_state()
            if not trading_state.should_trigger_analysis:
                logger.debug(
                    "scheduler.tick_skipped_paused",
                    mode=trading_state.mode.value,
                    seconds=seconds
                )
                return

            event = AgentEvent.create(
                EventType.SCHEDULED_TICK,
                source=source,
                payload=payload or {"interval": seconds},
                priority=priority,
            )
            await self.queue.put(event)
            logger.debug("scheduler.tick_enqueued", seconds=seconds, event_id=event.event_id)

        self.scheduler.add_job(_emit_tick, "interval", seconds=seconds, id=f"tick_{seconds}")
        logger.info("scheduler.tick_registered", seconds=seconds)

    def start_adaptive_ticks(
        self,
        trade_calendar: TradeCalendar,
        *,
        trading_interval_seconds: int = 15 * 60,
        off_hours_interval_seconds: int = 2 * 60 * 60,
        non_trading_day_interval_seconds: Optional[int] = None,
        monitor_interval_seconds: int = 60,
        source: str = "scheduler",
        payload: Optional[Dict[str, Any]] = None,
        priority: EventPriority = EventPriority.NORMAL,
    ) -> None:
        """根据交易时段动态调整 Tick 频率并同步更新交易模式。

        Args:
            trade_calendar: 交易日历，用于判断当前时段
            trading_interval_seconds: 交易时段触发频率（默认 15 分钟）
            off_hours_interval_seconds: 非交易时段触发频率（默认 2 小时）
            non_trading_day_interval_seconds: 非交易日触发频率（默认同 off-hours）
            monitor_interval_seconds: 调度检查频率（默认每 60 秒重新评估）
            source: 事件来源标签
            payload: 附加到事件的自定义 payload
            priority: 事件优先级
        """

        if not self._started:
            raise RuntimeError("SchedulerService must be started before registering adaptive ticks")

        non_trading_day_interval_seconds = (
            non_trading_day_interval_seconds or off_hours_interval_seconds
        )

        self._adaptive_job_id = "adaptive_tick"
        self._adaptive_monitor_job_id = "adaptive_tick_monitor"
        self._adaptive_interval = None

        async def _emit_tick() -> None:
            state = get_trading_state()

            if not state.should_trigger_analysis:
                logger.debug(
                    "scheduler.tick_skipped_mode",
                    mode=state.mode.value,
                    interval_seconds=self._adaptive_interval,
                )
                return

            event_payload = dict(payload or {})
            event_payload.setdefault("interval_seconds", self._adaptive_interval)
            event_payload.setdefault("mode", state.mode.value)

            event = AgentEvent.create(
                EventType.SCHEDULED_TICK,
                source=source,
                payload=event_payload,
                priority=priority,
            )
            await self.queue.put(event)
            logger.debug(
                "scheduler.adaptive_tick_enqueued",
                event_id=event.event_id,
                mode=state.mode.value,
                interval_seconds=self._adaptive_interval,
            )

        async def _evaluate_schedule() -> None:
            state = get_trading_state()
            now = datetime.now()

            # 更新交易模式
            new_mode = state.update_from_calendar(trade_calendar, current_time=now)

            # 选择目标频率
            if new_mode is TradingMode.TRADING:
                target_interval = trading_interval_seconds
            else:
                if trade_calendar.is_trading_day(now.date()):
                    target_interval = off_hours_interval_seconds
                else:
                    target_interval = non_trading_day_interval_seconds

            if target_interval != self._adaptive_interval:
                self._adaptive_interval = target_interval
                self.scheduler.add_job(
                    _emit_tick,
                    "interval",
                    seconds=target_interval,
                    id=self._adaptive_job_id,
                    replace_existing=True,
                    next_run_time=now,
                )
                logger.info(
                    "scheduler.adaptive_interval_updated",
                    new_interval_seconds=target_interval,
                    mode=new_mode.value,
                )

        # 先立即评估一次，确保快速落地正确频率
        self.scheduler.add_job(
            _evaluate_schedule,
            "interval",
            seconds=monitor_interval_seconds,
            id=self._adaptive_monitor_job_id,
            replace_existing=True,
            next_run_time=datetime.now(),
        )

        logger.info(
            "scheduler.adaptive_mode_enabled",
            trading_interval_seconds=trading_interval_seconds,
            off_hours_interval_seconds=off_hours_interval_seconds,
            monitor_interval_seconds=monitor_interval_seconds,
        )


__all__ = ["SchedulerService"]
