"""代理编排器最小实现（带决策链占位）。

当前版本串联事件队列、上下文构建、简单 Reasoner 与交易执行，
用于演示 `manual_command` → 决策 → dry-run 交易 → 风控巡检 的基本流程。
"""

from __future__ import annotations

import asyncio
from contextlib import AsyncExitStack
from typing import Any, Optional

import structlog

from laca.agent.context import ContextBuilder, Observation
from laca.agent.decision_summarizer import DecisionSummarizer
from laca.agent.enhanced_llm_reasoner import EnhancedLLMReasoner
from laca.agent.llm_reasoner import LLMReasoner
from laca.agent.memory_system import DecisionContext
from laca.agent.reasoner import Reasoner, ReasonerResult, RuleBasedReasoner, SimpleRuleReasoner
from laca.agent.trading_state import get_trading_state
from laca.config.logging import configure_logging
from laca.config.settings import get_settings
from laca.memory.store import Database
from laca.notifications import TelegramNotifier
from laca.notifications.base import NotificationLevel
from laca.risk import MarginSafetyRule, PositionLossRule, RiskEngine
from laca.services.risk_service import RiskService
from laca.tools.portfolio_tool import PortfolioTool
from laca.tools.trading_tool import TradingResult, TradingTool

from .events import AgentEvent, EventQueue, EventPriority, EventType

logger = structlog.get_logger("orchestrator")


class Orchestrator:
    """事件驱动的编排器骨架。"""

    def __init__(
        self,
        queue: Optional[EventQueue] = None,
        risk_service: Optional[RiskService] = None,
        database: Optional[Database] = None,
        context_builder: Optional[ContextBuilder] = None,
        reasoner: Optional[Reasoner] = None,
        trading_tool: Optional[TradingTool] = None,
        notifier: Optional[Any] = None,
        summarizer: Optional[DecisionSummarizer] = None,
    ) -> None:
        self.queue = queue or EventQueue()
        self._running = False
        self._exit_stack = AsyncExitStack()
        self._risk_service = risk_service
        self._database = database
        self._context_builder = context_builder
        self._reasoner = reasoner
        self._trading_tool = trading_tool
        self._notifier = notifier
        self._summarizer = summarizer or DecisionSummarizer()

    async def start(self) -> None:
        if self._running:
            logger.warning("orchestrator.already_running")
            return

        self._running = True
        logger.info("orchestrator.start")

        if self._risk_service:
            self._exit_stack.push_async_callback(self._risk_service.close)
        if self._trading_tool:
            self._exit_stack.push_async_callback(self._trading_tool.close)

        async with self._exit_stack:
            while self._running:
                event = await self.queue.get()
                logger.info(
                    "event.received",
                    event_type=event.event_type.value,
                    priority=int(event.priority),
                    source=event.source,
                )
                try:
                    await self.handle_event(event)
                except Exception as exc:  # noqa: BLE001
                    logger.exception("event.error", error=str(exc), event_id=event.event_id)
                finally:
                    self.queue.task_done()

    async def stop(self) -> None:
        if not self._running:
            logger.warning("orchestrator.not_running")
            return

        logger.info("orchestrator.stop")
        self._running = False
        await self.queue.put(
            AgentEvent.create(
                EventType.MANUAL_COMMAND,
                source="system",
                payload={"command": "shutdown"},
                priority=EventPriority.CRITICAL,
            )
        )

    async def handle_event(self, event: AgentEvent) -> None:
        if event.event_type is EventType.SCHEDULED_TICK:
            await self._handle_tick(event)
        elif event.event_type is EventType.MANUAL_COMMAND:
            await self._handle_manual(event)
        elif event.event_type is EventType.AUTO_SIGNAL:
            await self._handle_auto_signal(event)
        else:
            logger.warning("event.unsupported", event_type=event.event_type.value)

    async def _handle_tick(self, event: AgentEvent) -> None:
        logger.info("tick.process", payload=event.payload)
        if self._risk_service:
            await self._risk_service.run_health_check()
        # 在tick事件中也运行LLM决策pipeline
        await self._run_pipeline(event)
        await asyncio.sleep(0)

    async def _handle_manual(self, event: AgentEvent) -> None:
        command = event.payload.get("command") if event.payload else None
        if command == "shutdown":
            logger.info("manual.shutdown_ack")
            return
        if command == "alerts":
            await self._handle_show_alerts(event)
            return
        logger.info("manual.process", payload=event.payload)
        await self._run_pipeline(event)

    async def _handle_auto_signal(self, event: AgentEvent) -> None:
        if not self._context_builder or not self._reasoner:
            logger.warning("autosignal.no_pipeline")
            return
        logger.info("autosignal.trigger", payload=event.payload)
        await self._run_pipeline(event)

    async def _handle_show_alerts(self, event: AgentEvent) -> None:
        if not self._database:
            logger.warning("manual.alerts.no_database")
            return
        limit = event.payload.get("limit", 5) if event.payload else 5
        try:
            limit = int(limit)
        except (TypeError, ValueError):
            limit = 5
        records = await self._database.fetch_recent_risk_alerts(limit=limit)
        if not records:
            logger.info("manual.alerts.empty", limit=limit)
            return
        for record in records:
            logger.info(
                "manual.alerts.record",
                rule_id=record.rule_id,
                severity=record.severity,
                message=record.message,
                created_at=record.created_at.isoformat(),
            )

    def _check_preconditions(self, plan, observation: Observation) -> bool:
        metrics = observation.metrics or {}
        for key, expression in plan.preconditions.items():
            value = metrics.get(key)
            if value is None:
                logger.warning("plan.precondition.missing_metric", key=key)
                return False
            if not self._evaluate_condition(value, expression):
                logger.warning(
                    "plan.precondition.failed",
                    key=key,
                    condition=expression,
                    actual=value,
                )
                return False
        return True

    @staticmethod
    def _evaluate_condition(value: Any, expression: str) -> bool:
        expr = (expression or "").strip()
        try:
            if expr.startswith(">="):
                return float(value) >= float(expr[2:].strip())
            if expr.startswith("<="):
                return float(value) <= float(expr[2:].strip())
            if expr.startswith(">"):
                return float(value) > float(expr[1:].strip())
            if expr.startswith("<"):
                return float(value) < float(expr[1:].strip())
            if expr.startswith("=="):
                return float(value) == float(expr[2:].strip())
        except (ValueError, TypeError):
            logger.warning("plan.precondition.parse_failed", expression=expression, value=value)
            return False
        return str(value).lower() == expr.lower()

    @staticmethod
    def _mark_step(plan, name: str, *, status: str, result: Optional[str] = None) -> None:
        for step in plan.steps:
            if step.name == name:
                step.status = status
                step.result = result
                break

    async def _send_trade_execution_notification(
        self, plan, before_observation: Observation, after_observation: Observation, trade_results
    ) -> None:
        """发送交易执行结果通知到Telegram。"""
        if not self._notifier:
            return

        try:
            metadata = plan.metadata or {}
            decision = metadata.get("decision", {})
            action = decision.get("action", "UNKNOWN")
            symbol = decision.get("symbol", "N/A")
            confidence = decision.get("confidence", 0.0)
            risk_level = decision.get("risk_level", "UNKNOWN")
            reason = decision.get("reason", "无详细理由")

            # 交易前后账户对比
            # 兼容 AShare 和 Crypto 两种快照结构
            before_equity = getattr(before_observation.snapshot, 'total_equity', None) or getattr(before_observation.snapshot, 'total_assets', 0)
            after_equity = getattr(after_observation.snapshot, 'total_equity', None) or getattr(after_observation.snapshot, 'total_assets', 0)
            equity_change = after_equity - before_equity
            equity_change_pct = (equity_change / before_equity * 100) if before_equity > 0 else 0

            before_cash = getattr(before_observation.snapshot, 'cash_balance', None) or getattr(before_observation.snapshot, 'available_cash', 0)
            after_cash = getattr(after_observation.snapshot, 'cash_balance', None) or getattr(after_observation.snapshot, 'available_cash', 0)
            cash_change = after_cash - before_cash

            # 交易执行结果
            execution_info = ""
            success_count = 0
            failed_count = 0

            for request, result in trade_results:
                if result and result.success and result.executed:
                    success_count += 1
                    side_emoji = "🟢" if request.side.value == "BUY" else "🔴"
                    execution_info += f"{side_emoji} {request.side.value} {request.symbol}\n"
                    execution_info += f"   数量: {result.filled_quantity}\n"
                    execution_info += f"   成交价: ${result.avg_price:,.2f}\n"
                    execution_info += f"   订单号: {result.order_id}\n"
                    execution_info += f"   ✅ 成交成功\n"
                else:
                    failed_count += 1
                    execution_info += f"❌ {request.side.value} {request.symbol} × {request.quantity}\n"
                    execution_info += f"   ❌ 下单失败\n"

            # 市场信息
            market_info = ""
            # 兼容 Crypto (tickers) 和 AShare (quotes) 两种市场数据结构
            if after_observation.market:
                if hasattr(after_observation.market, 'tickers') and after_observation.market.tickers:
                    # Crypto 风格
                    for sym, ticker in list(after_observation.market.tickers.items())[:3]:
                        market_info += f"• {sym}: ${ticker.last_price:,.2f} ({ticker.price_change_percent:+.2f}%)\n"
                elif hasattr(after_observation.market, 'quotes') and after_observation.market.quotes:
                    # AShare 风格
                    for ts_code, quote in list(after_observation.market.quotes.items())[:3]:
                        price = quote.get('price', 0.0)
                        market_info += f"• {ts_code}: ¥{price:.2f}\n"

            # 如果market_info为空，提供默认提示
            if not market_info:
                market_info = "（市场数据暂时不可用）"

            # 构建消息
            result_emoji = "✅" if failed_count == 0 else ("⚠️" if success_count > 0 else "❌")

            # 使用nano模型生成智能摘要（交易执行报告）
            ai_summary = f"{action} {symbol}: {reason[:100]}{'...' if len(reason) > 100 else ''}"  # 默认降级方案
            try:
                generated_summary = await self._summarizer.generate_summary(
                    metadata=metadata,
                    max_length=180,
                )
                # 如果生成的摘要不为空，使用生成的摘要
                if generated_summary and generated_summary.strip():
                    ai_summary = generated_summary
                else:
                    logger.warning(
                        "trade_execution_summary.empty_result",
                        fallback_used=True,
                        reason="nano model returned empty content"
                    )
            except Exception as exc:
                logger.warning("trade_execution_summary.generation_failed", error=str(exc))

            message = f"""
{result_emoji} **交易执行报告** {result_emoji}

━━━━━━━━━━━━━━━━━━━━
🎯 **AI决策摘要**
{ai_summary}

━━━━━━━━━━━━━━━━━━━━
📊 **执行结果**
{execution_info}
━━━━━━━━━━━━━━━━━━━━
💼 **账户变化**
• 总权益: ${before_equity:,.2f} → ${after_equity:,.2f} ({equity_change:+,.2f} USDT, {equity_change_pct:+.2f}%)
• 可用现金: ${before_cash:,.2f} → ${after_cash:,.2f} ({cash_change:+,.2f} USDT)

━━━━━━━━━━━━━━━━━━━━
📈 **当前市场**
{market_info}
━━━━━━━━━━━━━━━━━━━━
🤖 **系统信息**
• LLM: {metadata.get('llm_provider', 'unknown')}/{metadata.get('llm_model', 'unknown')}
• 交易模式: {"实盘" if self._trading_tool and hasattr(self._trading_tool, 'mode') and self._trading_tool.mode.value == "real" else "模拟"}
• 时间: {after_observation.event.created_at.strftime("%Y-%m-%d %H:%M:%S")}
"""

            await self._notifier.send(
                subject=f"交易执行: {result_emoji} {action} {symbol}",
                content=message,
                level=NotificationLevel.INFO
            )
            logger.info("trade_execution_notification_sent", success=success_count, failed=failed_count)

        except Exception as exc:  # noqa: BLE001
            logger.exception("trade_execution_notification_failed", error=str(exc))

    async def _record_decision_to_memory(self, plan, observation: Observation) -> None:
        """将决策记录到记忆系统。"""
        if not self._context_builder:
            return

        try:
            metadata = plan.metadata or {}
            decision = metadata.get("decision", {})

            # 提取机会币种信息
            opportunities = []
            if metadata.get("opportunity_coins"):
                opportunities = [
                    f"{opp.get('symbol', 'UNKNOWN')} {opp.get('volatility', 0):+.2f}%"
                    if isinstance(opp, dict) else str(opp)
                    for opp in metadata["opportunity_coins"][:3]
                ]

            # 构建决策上下文
            context = DecisionContext(
                timestamp=observation.collected_at,
                decision_type=decision.get("action", "UNKNOWN"),
                symbol=decision.get("symbol", "NO_TRADE"),
                confidence=decision.get("confidence", 0.0),
                risk_level=decision.get("risk_level", "UNKNOWN"),
                reasoning=decision.get("reason", "")[:200],  # 限制长度
                market_view=metadata.get("market_summary", "")[:100] if metadata.get("market_summary") else "",
                opportunity_coins=opportunities,
                rejected_reason=decision.get("reason", "")[:100] if decision.get("action") == "HOLD" else None,
                tools_used=list(set(metadata.get("tools_called", []))),
                reasoning_rounds=metadata.get("reasoning_steps", 0),
            )

            # 添加到记忆系统
            self._context_builder.memory_system.add_decision(context)

            logger.info(
                "memory_system.decision_recorded",
                decision_type=context.decision_type,
                symbol=context.symbol,
                confidence=context.confidence,
            )

        except Exception as exc:
            logger.warning("memory_system.record_failed", error=str(exc))

    async def _send_decision_notification(self, plan, observation: Observation) -> None:
        """发送GPT决策通知到Telegram。"""
        if not self._notifier:
            return

        try:
            # 提取决策信息
            metadata = plan.metadata or {}
            decision = metadata.get("decision", {})
            action = decision.get("action", "UNKNOWN")
            symbol = decision.get("symbol", "N/A")
            confidence = decision.get("confidence", 0.0)
            risk_level = decision.get("risk_level", "UNKNOWN")
            reason = decision.get("reason", "无详细理由")

            # 账户信息
            snapshot = observation.snapshot
            # 兼容 AShare 和 Crypto 两种快照结构
            equity = getattr(snapshot, 'total_equity', None) or getattr(snapshot, 'total_assets', 0)
            cash = getattr(snapshot, 'cash_balance', None) or getattr(snapshot, 'available_cash', 0)
            positions_count = len(snapshot.positions)

            # 市场信息
            market_info = ""
            # 兼容 Crypto (tickers) 和 AShare (quotes) 两种市场数据结构
            if observation.market:
                if hasattr(observation.market, 'tickers') and observation.market.tickers:
                    # Crypto 风格
                    for sym, ticker in list(observation.market.tickers.items())[:3]:
                        market_info += f"• {sym}: ${ticker.last_price:,.2f} ({ticker.price_change_percent:+.2f}%)\n"
                elif hasattr(observation.market, 'quotes') and observation.market.quotes:
                    # AShare 风格
                    for ts_code, quote in list(observation.market.quotes.items())[:3]:
                        price = quote.get('price', 0.0)
                        market_info += f"• {ts_code}: ¥{price:.2f}\n"

            if not market_info:
                market_info = "• 暂无市场数据\n"

            # 持仓信息
            positions_info = ""
            if snapshot.positions:
                for pos in snapshot.positions[:3]:  # 最多3个持仓
                    pnl_pct = ((pos.mark_price - pos.entry_price) / pos.entry_price * 100) if pos.entry_price > 0 else 0
                    positions_info += f"• {pos.symbol}: {pos.quantity} @ ${pos.entry_price:.2f} ({pnl_pct:+.2f}%)\n"
            else:
                positions_info = "• 暂无持仓\n"

            # 构建消息
            has_trades = bool(plan.trades)
            emoji = "🔄" if has_trades else "⏸️"
            action_emoji = {"BUY": "🟢", "SELL": "🔴", "HOLD": "⏸️"}.get(action, "❓")
            action_desc = f"{action} {symbol}" if has_trades else f"{action}"

            # 增强推理信息
            enhanced_info = ""
            if metadata.get("enhanced_reasoning"):
                reasoning_steps = metadata.get("reasoning_steps", 0)
                tool_calls = metadata.get("tool_calls", 0)
                enhanced_info = f"""
━━━━━━━━━━━━━━━━━━━━
📊 **增强AI推理详情**
• 推理轮次: {reasoning_steps} 轮
• 工具调用: {tool_calls} 次
• Function Calling: ✅ 启用
"""

            # 机会币种发现信息
            opportunity_info = ""
            opportunities = metadata.get("opportunity_coins", [])
            if opportunities:
                # 只显示前3个机会币种
                top_opportunities = opportunities[:3]
                opportunity_info = "\n━━━━━━━━━━━━━━━━━━━━\n"
                opportunity_info += "🎯 **机会币种发现**\n"
                for i, opp in enumerate(top_opportunities, 1):
                    # 假设opp是字典格式: {"symbol": "BTCUSDT", "volatility": 5.5, "volume": 1000000}
                    if isinstance(opp, dict):
                        sym = opp.get("symbol", "UNKNOWN")
                        vol = opp.get("volatility", 0.0)
                        opportunity_info += f"{i}. {sym}: 波动{vol:+.2f}%\n"
                    else:
                        # 如果只是字符串
                        opportunity_info += f"{i}. {opp}\n"

            # 智能仓位管理信息
            position_sizing_info = ""
            position_calc = metadata.get("position_calculation")
            if position_calc:
                suggested_usdt = position_calc.get("suggested_usdt", 0.0)
                position_ratio = position_calc.get("position_ratio", 0.0)
                reason = position_calc.get("reason", "")
                max_reached = position_calc.get("max_positions_reached", False)

                position_sizing_info = "\n━━━━━━━━━━━━━━━━━━━━\n"
                position_sizing_info += "💰 **智能仓位管理**\n"
                if max_reached:
                    position_sizing_info += "⚠️ 已达持仓上限（3个），建议HOLD\n"
                else:
                    position_sizing_info += f"• 建议投入: ${suggested_usdt:.2f} USDT\n"
                    position_sizing_info += f"• 仓位比例: {position_ratio*100:.2f}%\n"
                    if reason:
                        position_sizing_info += f"• 计算依据: {reason}\n"

            # 使用nano模型生成智能摘要
            ai_summary = reason[:100] + "..." if len(reason) > 100 else reason  # 默认降级方案
            try:
                generated_summary = await self._summarizer.generate_summary(
                    metadata=metadata,
                    max_length=200,
                )
                # 如果生成的摘要不为空，使用生成的摘要
                if generated_summary and generated_summary.strip():
                    ai_summary = generated_summary
                else:
                    logger.warning(
                        "decision_summary.empty_result",
                        fallback_used=True,
                        reason="nano model returned empty content"
                    )
            except Exception as exc:
                logger.warning("decision_summary.generation_failed", error=str(exc))

            message = f"""
{emoji} **AI交易决策报告** {action_emoji}

━━━━━━━━━━━━━━━━━━━━
🎯 **决策结果**
• 动作: {action_desc}
• 置信度: {confidence:.1%}
• 风险等级: {risk_level}

━━━━━━━━━━━━━━━━━━━━
💭 **AI决策摘要**
{ai_summary}
{enhanced_info}{opportunity_info}{position_sizing_info}
━━━━━━━━━━━━━━━━━━━━
💼 **账户状态**
• 总权益: ${equity:,.2f} USDT
• 可用资金: ${cash:,.2f} USDT
• 持仓数量: {positions_count}

━━━━━━━━━━━━━━━━━━━━
📈 **市场快照**
{market_info}
━━━━━━━━━━━━━━━━━━━━
📋 **当前持仓**
{positions_info}
━━━━━━━━━━━━━━━━━━━━
🤖 **系统信息**
• LLM: {metadata.get('llm_provider', 'unknown')}/{metadata.get('llm_model', 'unknown')}
• 交易模式: {"实盘" if self._trading_tool and hasattr(self._trading_tool, 'mode') and self._trading_tool.mode.value == "real" else "模拟"}
• 时间: {observation.event.created_at.strftime("%Y-%m-%d %H:%M:%S")}
"""

            # 如果有交易，添加交易详情
            if has_trades:
                message += "\n━━━━━━━━━━━━━━━━━━━━\n"
                message += "📝 **交易详情**\n"
                for trade in plan.trades:
                    side_emoji = "🟢" if trade.side.value == "BUY" else "🔴"
                    message += f"{side_emoji} {trade.side.value} {trade.symbol} × {trade.quantity}\n"

            await self._notifier.send(
                subject=f"AI决策: {action_emoji} {action_desc}",
                content=message,
                level=NotificationLevel.INFO
            )
            logger.info("decision.notification_sent", action=action)

        except Exception as exc:  # noqa: BLE001
            logger.exception("decision.notification_failed", error=str(exc))

    async def _run_pipeline(self, event: AgentEvent, max_retries: int = 2) -> None:
        """执行决策-交易流程，支持智能重试机制。

        Args:
            event: 触发事件
            max_retries: 最大重试次数（默认2次，总共3次尝试）
        """
        if not self._context_builder or not self._reasoner:
            logger.warning("pipeline.unavailable")
            return

        # 用于记录失败历史，供AI参考
        failure_history = []

        # 重试循环
        trading_state = get_trading_state()

        for attempt in range(max_retries + 1):
            is_retry = attempt > 0

            if is_retry:
                logger.info(
                    "pipeline.retry_attempt",
                    attempt=attempt + 1,
                    max_attempts=max_retries + 1,
                    previous_failures=len(failure_history)
                )

            # 构建observation（重试时包含失败历史）
            observation = await self._context_builder.build(
                event,
                metadata={"trading_mode": trading_state.mode.value},
            )

            # 如果是重试，将失败信息添加到observation的metadata
            if is_retry and failure_history:
                observation.metadata["previous_failures"] = failure_history
                observation.metadata["retry_attempt"] = attempt + 1
                logger.info(
                    "pipeline.retry_context_added",
                    failures_count=len(failure_history)
                )

            # AI决策
            plan_result: ReasonerResult = await self._reasoner.plan(observation)

            for note in plan_result.notes:
                logger.info("reasoner.note", note=note)

            plan = plan_result.plan
            if not plan:
                logger.warning("reasoner.no_plan_generated")
                return

            # 记录当前交易模式
            plan.metadata.setdefault("trading_mode", trading_state.mode.value)

            # 非交易时段禁止执行实际下单
            if plan.trades and not trading_state.should_execute_trades:
                blocked = len(plan.trades)
                plan.metadata["trade_execution_blocked"] = {
                    "mode": trading_state.mode.value,
                    "blocked_orders": blocked,
                    "reason": "当前处于分析模式，仅输出情报，不执行下单",
                }
                plan.trades.clear()
                logger.info(
                    "pipeline.trading_skipped_off_hours",
                    blocked_orders=blocked,
                    mode=trading_state.mode.value,
                )

            # 记录决策详情（包括HOLD）
            has_trades = bool(plan.trades)
            action_type = "TRADE" if has_trades else "HOLD"

            logger.info(
                "reasoner.decision",
                action_type=action_type,
                summary=plan.summary,
                trades_count=len(plan.trades),
                metadata=plan.metadata,
                is_retry=is_retry,
            )

            # 始终记录到数据库（包括HOLD决策）
            if self._database:
                try:
                    await self._database.log_trading_plan(
                        summary=plan.summary,
                        plan=plan.to_dict(),
                        source_event_id=event.event_id,
                    )
                    logger.info("plan.persisted", action_type=action_type)
                except Exception as exc:  # noqa: BLE001
                    logger.exception("plan.persist_failed", error=str(exc))

            # 记录决策到记忆系统
            await self._record_decision_to_memory(plan, observation)

            # 如果没有实际交易，发送决策通知后流程结束
            if not has_trades:
                await self._send_decision_notification(plan, observation)
                logger.info("reasoner.no_trades_to_execute")
                return

            # 以下是交易执行逻辑
            if plan.preconditions and not self._check_preconditions(plan, observation):
                self._mark_step(plan, "validate_preconditions", status="blocked", result="precondition_failed")
                logger.warning("plan.preconditions_failed")
                return
            if plan.preconditions:
                self._mark_step(plan, "validate_preconditions", status="satisfied")

            logger.info(
                "reasoner.plan.executing",
                summary=plan.summary,
                steps=len(plan.steps),
                is_retry=is_retry,
            )

            if not self._trading_tool:
                logger.warning("pipeline.no_trading_tool")
                return

            # 存储交易执行结果
            trade_results = []
            current_failures = []

            for request in plan.trades:
                try:
                    result: TradingResult = await self._trading_tool.place_order(request)
                    trade_results.append((request, result))

                    # 检查交易是否真正成功
                    if not result.success or not result.executed:
                        error_msg = result.raw_response.get("error", "Unknown error")
                        current_failures.append({
                            "symbol": request.symbol,
                            "side": request.side.value,
                            "quantity": request.quantity,
                            "error": error_msg,
                            "attempt": attempt + 1
                        })
                        logger.warning(
                            "trading.order_failed",
                            symbol=request.symbol,
                            error=error_msg,
                            attempt=attempt + 1
                        )

                except Exception as exc:  # noqa: BLE001
                    logger.exception(
                        "trading.execution_failed",
                        symbol=request.symbol,
                        error=str(exc),
                    )
                    self._mark_step(plan, "submit_order", status="failed", result=str(exc))
                    # 记录失败的交易
                    trade_results.append((request, None))
                    current_failures.append({
                        "symbol": request.symbol,
                        "side": request.side.value,
                        "quantity": request.quantity,
                        "error": str(exc),
                        "attempt": attempt + 1
                    })
                    continue

                self._mark_step(
                    plan,
                    "submit_order",
                    status="completed",
                    result=result.order_id or "dry-run",
                )

                logger.info(
                    "trading.order_result",
                    symbol=request.symbol,
                    side=request.side.value,
                    quantity=request.quantity,
                    mode=self._trading_tool.mode.value,
                    order_id=result.order_id,
                    executed=result.executed,
                )

            # 如果所有交易都成功，结束重试循环
            if not current_failures:
                logger.info("pipeline.all_trades_successful", attempt=attempt + 1)
                break

            # 如果还有失败且还能重试，记录失败并继续
            if current_failures and attempt < max_retries:
                failure_history.extend(current_failures)
                logger.info(
                    "pipeline.will_retry",
                    current_failures=len(current_failures),
                    total_failures=len(failure_history),
                    next_attempt=attempt + 2
                )
                # 等待一小段时间再重试，给系统一些时间更新状态
                await asyncio.sleep(1)
                continue

            # 已达最大重试次数且仍有失败
            if current_failures and attempt >= max_retries:
                failure_history.extend(current_failures)
                logger.warning(
                    "pipeline.max_retries_reached",
                    total_attempts=attempt + 1,
                    total_failures=len(failure_history)
                )

        # 交易执行后，获取新的账户状态并发送通知
        if self._context_builder:
            try:
                # 重新构建观察对象获取最新状态
                new_observation = await self._context_builder.build(
                    event,
                    metadata={"trading_mode": trading_state.mode.value},
                )
                await self._send_trade_execution_notification(
                    plan, observation, new_observation, trade_results
                )
            except Exception as exc:  # noqa: BLE001
                logger.exception("trade_notification_failed", error=str(exc))

async def _demo_bootstrap() -> None:
    """当模块直接运行时，启动一个简单的演示循环。"""
    from laca.services.scheduler import SchedulerService  # Import locally to avoid circular dependency

    configure_logging("INFO")

    settings = get_settings()
    database = Database()
    await database.init_db()

    portfolio_tool = PortfolioTool(
        settings=settings,
        database=database,
    )
    trading_tool = TradingTool(
        settings=settings,
        database=database,
    )
    notifier = TelegramNotifier(
        token=settings.telegram_bot_token,
        chat_id=settings.telegram_chat_id,
    )
    risk_engine = RiskEngine(
        [
            PositionLossRule(),
            MarginSafetyRule(),
        ],
        notifier=notifier,
    )

    queue = EventQueue()
    risk_service = RiskService(
        portfolio_tool=portfolio_tool,
        risk_engine=risk_engine,
        queue=queue,
        database=database,
    )

    context_builder = ContextBuilder(portfolio_tool)
    reasoner = EnhancedLLMReasoner(settings=settings, provider="yunwu")

    orchestrator = Orchestrator(
        queue=queue,
        risk_service=risk_service,
        database=database,
        context_builder=context_builder,
        reasoner=reasoner,
        trading_tool=trading_tool,
        notifier=notifier,
    )
    scheduler = SchedulerService(orchestrator.queue)
    scheduler.schedule_tick(1, payload={"note": "demo tick every 1s"})
    scheduler.start()

    async def produce_events() -> None:
        await asyncio.sleep(0.2)
        await orchestrator.queue.put(
            AgentEvent.create(
                EventType.MANUAL_COMMAND,
                source="cli",
                payload={"command": "status"},
                priority=EventPriority.HIGH,
            )
        )
        await asyncio.sleep(0.3)
        await orchestrator.queue.put(
            AgentEvent.create(
                EventType.MANUAL_COMMAND,
                source="cli",
                payload={
                    "command": "demo_trade",
                    "symbol": "BTCUSDT",
                    "quantity": 0.001,
                    "side": "BUY",
                },
                priority=EventPriority.HIGH,
            )
        )
        await asyncio.sleep(2.5)
        await orchestrator.stop()
        await scheduler.shutdown()

    await asyncio.gather(orchestrator.start(), produce_events())


if __name__ == "__main__":
    asyncio.run(_demo_bootstrap())
