/**
 * JuxGame Refund System
 * Handles refunds for expired mint requests
 */

class JuxGameRefund {
    constructor() {
        this.refundContract = null;
        this.refundContractAddress = '0xFE0640C79Bf5Db5ed083c151C8373C345a9C12ce';
        this.userRefunds = [];
        this.selectedRefunds = new Set();
        this.isProcessing = false;
        this.contractStats = null;
        
        this.init();
    }
    
    async init() {
        this.setupEventListeners();
        this.updateUI();
        
        // Wait for wallet connection
        this.waitForWallet();
        
        // Load contract stats periodically
        setInterval(() => {
            if (this.refundContract) {
                this.loadContractStats();
            }
        }, 30000);
    }
    
    waitForWallet() {
        const checkWallet = () => {
            if (window.refundWeb3 && window.refundWeb3.isConnected()) {
                this.onWalletConnected();
            } else {
                setTimeout(checkWallet, 1000);
            }
        };
        checkWallet();
        
        // Listen for wallet connection events
        document.addEventListener('click', (e) => {
            if (e.target.id === 'connect-wallet-btn') {
                this.handleConnectWallet();
            }
        });
        
        window.addEventListener('walletDisconnected', () => {
            this.onWalletDisconnected();
        });
    }
    
    async handleConnectWallet() {
        try {
            await window.refundWeb3.connectWallet();
            this.onWalletConnected();
        } catch (error) {
            console.error('Wallet connection failed:', error);
            this.showNotification('error', error.message || 'Failed to connect wallet');
        }
    }
    
    async onWalletConnected() {
        try {
            // Initialize refund contract
            await this.initRefundContract();
            
            // Update wallet UI
            this.updateWalletUI();
            
            // Load contract stats
            await this.loadContractStats();
            
            // Auto-scan for refunds
            await this.scanForRefunds();
            
        } catch (error) {
            console.error('Failed to initialize refund system:', error);
            this.showNotification('error', 'Failed to initialize refund system');
        }
    }
    
    onWalletDisconnected() {
        this.refundContract = null;
        this.userRefunds = [];
        this.selectedRefunds.clear();
        this.contractStats = null;
        
        this.updateWalletUI();
        this.updateRefundsList();
        this.updateStats();
    }
    
    async initRefundContract() {
        if (!this.refundContractAddress) {
            // TODO: Set this after contract deployment
            throw new Error('Refund contract not deployed yet');
        }
        
        const signer = window.refundWeb3.getSigner();
        
        const refundABI = [
            // View functions
            'function getUserRefundRequests(address user) view returns (bytes32[])',
            'function getUserRefundInfo(address user) view returns (tuple(address minter, uint256 quantity, uint256 refundAmount, bool claimed, uint256 requestBlockNumber, uint256 addedTimestamp)[])',
            'function getUserTotalRefundable(address user) view returns (uint256)',
            'function hasRefund(bytes32 requestId) view returns (bool)',
            'function getRefundInfo(bytes32 requestId) view returns (tuple(address minter, uint256 quantity, uint256 refundAmount, bool claimed, uint256 requestBlockNumber, uint256 addedTimestamp))',
            'function getStats() view returns (uint256 totalRefunds, uint256 totalClaimed, uint256 pendingRefunds, uint256 contractBalance, uint256 refundsCount, uint256 claimsCount)',
            
            // User functions
            'function claimRefund(bytes32 requestId) external',
            'function claimRefundsBatch(bytes32[] requestIds) external',
            
            // Events
            'event RefundClaimed(bytes32 indexed requestId, address indexed minter, uint256 refundAmount)'
        ];
        
        this.refundContract = new ethers.Contract(
            this.refundContractAddress,
            refundABI,
            signer
        );
    }
    
    setupEventListeners() {
        // Connect wallet button
        document.getElementById('connect-wallet-btn').addEventListener('click', async () => {
            if (window.refundWeb3 && window.refundWeb3.isConnected()) {
                // Disconnect - just refresh the page for simplicity
                location.reload();
            } else {
                await this.handleConnectWallet();
            }
        });
        
        // Scan for refunds
        document.getElementById('scan-refunds-btn').addEventListener('click', async () => {
            await this.scanForRefunds();
        });
        
        // Refresh button
        document.getElementById('refresh-btn').addEventListener('click', async () => {
            await this.refreshData();
        });
        
        // Claim all selected
        document.getElementById('claim-all-btn').addEventListener('click', async () => {
            await this.claimSelectedRefunds();
        });
        
        // Delegated event listeners for dynamic content
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('claim-single-btn')) {
                const requestId = e.target.dataset.requestId;
                this.claimSingleRefund(requestId);
            }
            
            if (e.target.classList.contains('refund-checkbox')) {
                const requestId = e.target.dataset.requestId;
                this.toggleRefundSelection(requestId);
            }
        });
    }
    
    async scanForRefunds() {
        if (!window.refundWeb3 || !window.refundWeb3.isConnected()) {
            this.showNotification('error', 'Please connect your wallet first');
            return;
        }
        
        if (!this.refundContract) {
            this.showNotification('error', 'Refund contract not available');
            return;
        }
        
        try {
            this.setScanning(true);
            
            const userAddress = window.refundWeb3.getAccount();
            
            // Load user's refund information
            const refundInfo = await this.refundContract.getUserRefundInfo(userAddress);
            const requestIds = await this.refundContract.getUserRefundRequests(userAddress);
            
            this.userRefunds = [];
            
            for (let i = 0; i < refundInfo.length; i++) {
                const info = refundInfo[i];
                const requestId = requestIds[i];
                
                this.userRefunds.push({
                    requestId: requestId,
                    minter: info.minter,
                    quantity: Number(info.quantity.toString()),
                    refundAmount: ethers.utils.formatEther(info.refundAmount),
                    claimed: info.claimed,
                    requestBlockNumber: Number(info.requestBlockNumber.toString()),
                    addedTimestamp: Number(info.addedTimestamp.toString())
                });
            }
            
            this.updateRefundsList();
            this.updateUserStats();
            
            if (this.userRefunds.length === 0) {
                this.showNotification('info', 'No refunds found for your account');
            } else {
                const unclaimedCount = this.userRefunds.filter(r => !r.claimed).length;
                this.showNotification('success', `Found ${unclaimedCount} unclaimed refunds`);
            }
            
        } catch (error) {
            console.error('Failed to scan for refunds:', error);
            this.showNotification('error', 'Failed to scan for refunds');
        } finally {
            this.setScanning(false);
        }
    }
    
    async refreshData() {
        await this.loadContractStats();
        await this.scanForRefunds();
    }
    
    async loadContractStats() {
        if (!this.refundContract) return;
        
        try {
            const stats = await this.refundContract.getStats();
            
            this.contractStats = {
                totalRefunds: ethers.utils.formatEther(stats.totalRefunds),
                totalClaimed: ethers.utils.formatEther(stats.totalClaimed),
                pendingRefunds: ethers.utils.formatEther(stats.pendingRefunds),
                contractBalance: ethers.utils.formatEther(stats.contractBalance),
                refundsCount: Number(stats.refundsCount.toString()),
                claimsCount: Number(stats.claimsCount.toString())
            };
            
            this.updateStats();
            
        } catch (error) {
            console.error('Failed to load contract stats:', error);
        }
    }
    
    async claimSingleRefund(requestId) {
        if (this.isProcessing) return;
        
        try {
            this.isProcessing = true;
            
            const refund = this.userRefunds.find(r => r.requestId === requestId);
            if (!refund) {
                throw new Error('Refund not found');
            }
            
            if (refund.claimed) {
                this.showNotification('info', 'This refund has already been claimed');
                return;
            }
            
            this.showNotification('info', 'Processing refund claim...');
            
            const tx = await this.refundContract.claimRefund(requestId);
            
            this.showNotification('success', `Claim transaction submitted! ${tx.hash.slice(0, 10)}...`);
            
            // Wait for confirmation
            await tx.wait();
            
            this.showNotification('success', `Successfully claimed ${refund.refundAmount} $Juxie!`);
            
            // Refresh data
            setTimeout(() => {
                this.refreshData();
            }, 3000);
            
        } catch (error) {
            console.error('Claim failed:', error);
            let errorMessage = 'Claim failed';
            
            if (error.message.includes('Already claimed')) {
                errorMessage = 'This refund has already been claimed';
            } else if (error.message.includes('Not the original minter')) {
                errorMessage = 'You are not authorized to claim this refund';
            } else if (error.message.includes('Insufficient contract balance')) {
                errorMessage = 'Contract has insufficient balance. Please contact support.';
            } else if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            }
            
            this.showNotification('error', errorMessage);
        } finally {
            this.isProcessing = false;
        }
    }
    
    async claimSelectedRefunds() {
        if (this.isProcessing || this.selectedRefunds.size === 0) return;
        
        try {
            this.isProcessing = true;
            
            const requestIds = Array.from(this.selectedRefunds);
            const refunds = this.userRefunds.filter(r => 
                requestIds.includes(r.requestId) && !r.claimed
            );
            
            if (refunds.length === 0) {
                this.showNotification('info', 'No unclaimed refunds selected');
                return;
            }
            
            const totalAmount = refunds.reduce((sum, r) => sum + parseFloat(r.refundAmount), 0);
            
            this.showNotification('info', `Processing ${refunds.length} refund claims...`);
            
            const tx = await this.refundContract.claimRefundsBatch(requestIds);
            
            this.showNotification('success', `Batch claim transaction submitted! ${tx.hash.slice(0, 10)}...`);
            
            // Wait for confirmation
            await tx.wait();
            
            this.showNotification('success', `Successfully claimed ${totalAmount.toLocaleString()} $Juxie from ${refunds.length} refunds!`);
            
            // Clear selection
            this.selectedRefunds.clear();
            
            // Refresh data
            setTimeout(() => {
                this.refreshData();
            }, 3000);
            
        } catch (error) {
            console.error('Batch claim failed:', error);
            let errorMessage = 'Batch claim failed';
            
            if (error.message.includes('Too many requests')) {
                errorMessage = 'Too many refunds selected. Please claim in smaller batches.';
            } else if (error.message.includes('Insufficient contract balance')) {
                errorMessage = 'Contract has insufficient balance. Please contact support.';
            } else if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            }
            
            this.showNotification('error', errorMessage);
        } finally {
            this.isProcessing = false;
        }
    }
    
    toggleRefundSelection(requestId) {
        if (this.selectedRefunds.has(requestId)) {
            this.selectedRefunds.delete(requestId);
        } else {
            this.selectedRefunds.add(requestId);
        }
        
        this.updateBatchActions();
        this.updateRefundsList(); // Update checkboxes
    }
    
    updateWalletUI() {
        const addressEl = document.getElementById('wallet-address');
        const balanceEl = document.getElementById('wallet-balance');
        const connectBtn = document.getElementById('connect-wallet-btn');
        
        if (window.refundWeb3 && window.refundWeb3.isConnected()) {
            const account = window.refundWeb3.getAccount();
            const balance = window.refundWeb3.getBalance();
            if (addressEl) addressEl.textContent = `${account.slice(0, 6)}...${account.slice(-4)}`;
            if (balanceEl) balanceEl.textContent = `${balance} $Juxie`;
            if (connectBtn) connectBtn.textContent = 'Disconnect';
        } else {
            if (addressEl) addressEl.textContent = 'Not Connected';
            if (balanceEl) balanceEl.textContent = 'Connect your wallet to view refunds';
            if (connectBtn) connectBtn.textContent = 'Connect Wallet';
        }
    }
    
    updateStats() {
        // Only update user-specific stats, not total stats
        // Contract stats are loaded but not displayed
    }
    
    updateUserStats() {
        const userRefundCount = this.userRefunds.filter(r => !r.claimed).length;
        const userRefundAmount = this.userRefunds
            .filter(r => !r.claimed)
            .reduce((sum, r) => sum + parseFloat(r.refundAmount), 0);
        
        document.getElementById('your-refunds').textContent = userRefundCount;
        document.getElementById('your-amount').textContent = userRefundAmount.toLocaleString();
    }
    
    updateRefundsList() {
        const container = document.getElementById('refund-list');
        
        if (this.userRefunds.length === 0) {
            container.innerHTML = `
                <div class="no-refunds">
                    <h3>No Refunds Found</h3>
                    <p>Connect your wallet and scan for available refunds</p>
                </div>
            `;
            return;
        }
        
        const refundsHTML = this.userRefunds.map(refund => {
            const isSelected = this.selectedRefunds.has(refund.requestId);
            const addedDate = new Date(refund.addedTimestamp * 1000);
            const formatAmount = parseFloat(refund.refundAmount).toLocaleString();
            
            return `
                <div class="refund-item ${refund.claimed ? 'claimed' : ''}">
                    <div class="refund-header-row">
                        <div>
                            <span class="refund-id">ID: ${refund.requestId.slice(0, 8)}...</span>
                            ${!refund.claimed ? `
                                <input type="checkbox" class="refund-checkbox" 
                                       data-request-id="${refund.requestId}" 
                                       ${isSelected ? 'checked' : ''}>
                            ` : ''}
                        </div>
                        <div class="refund-amount">${formatAmount} $Juxie</div>
                    </div>
                    
                    <div class="refund-details">
                        <div class="detail-item">
                            📦 <span>${refund.quantity} NFT${refund.quantity > 1 ? 's' : ''}</span>
                        </div>
                        <div class="detail-item">
                            🏗️ <span>Block #${refund.requestBlockNumber.toLocaleString()}</span>
                        </div>
                        <div class="detail-item">
                            📅 <span>${addedDate.toLocaleDateString()}</span>
                        </div>
                        <div class="detail-item">
                            ${refund.claimed ? '✅ <span>Claimed</span>' : '⏳ <span>Available</span>'}
                        </div>
                    </div>
                    
                    <div class="refund-actions">
                        ${refund.claimed ? 
                            '<span class="claimed-badge">✅ Claimed</span>' : 
                            `<button class="claim-btn claim-single-btn" 
                                     data-request-id="${refund.requestId}"
                                     ${this.isProcessing ? 'disabled' : ''}>
                                Claim Refund
                             </button>`
                        }
                    </div>
                </div>
            `;
        }).join('');
        
        container.innerHTML = refundsHTML;
    }
    
    updateBatchActions() {
        const batchActions = document.getElementById('batch-actions');
        const selectedCount = document.getElementById('selected-count');
        const batchTotal = document.getElementById('batch-total');
        const claimAllBtn = document.getElementById('claim-all-btn');
        
        const selectedRefunds = this.userRefunds.filter(r => 
            this.selectedRefunds.has(r.requestId) && !r.claimed
        );
        
        if (selectedRefunds.length > 0) {
            const totalAmount = selectedRefunds.reduce((sum, r) => sum + parseFloat(r.refundAmount), 0);
            
            batchActions.classList.add('show');
            selectedCount.textContent = selectedRefunds.length;
            batchTotal.textContent = totalAmount.toLocaleString();
            claimAllBtn.disabled = this.isProcessing;
        } else {
            batchActions.classList.remove('show');
        }
    }
    
    setScanning(isScanning) {
        const scanBtn = document.getElementById('scan-refunds-btn');
        const scanIcon = document.getElementById('scan-icon');
        const scanText = document.getElementById('scan-text');
        
        if (isScanning) {
            scanBtn.disabled = true;
            scanIcon.innerHTML = '<div class="loading-spinner"></div>';
            scanText.textContent = 'Scanning...';
        } else {
            scanBtn.disabled = false;
            scanIcon.textContent = '🔍';
            scanText.textContent = 'Scan for Refunds';
        }
    }
    
    updateUI() {
        this.updateWalletUI();
        this.updateRefundsList();
        this.updateStats();
        this.updateUserStats();
        this.updateBatchActions();
    }
    
    showNotification(type, message, duration = 5000) {
        // Remove existing notifications
        const existing = document.querySelectorAll('.notification');
        existing.forEach(n => n.remove());
        
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.textContent = message;
        
        document.body.appendChild(notification);
        
        // Auto remove
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, duration);
    }
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', () => {
    window.juxRefund = new JuxGameRefund();
});

// Handle mobile menu toggle
const mobileMenu = document.getElementById('mobile-menu');
const navMenu = document.querySelector('.nav-menu');

mobileMenu.addEventListener('click', () => {
    mobileMenu.classList.toggle('active');
    navMenu.classList.toggle('active');
});