/**
 * Mint functionality for JuxGame NFTs
 * Implements 2-phase minting with anti-MEV protection
 */

// Feature Control - Mint functionality enabled for production
const MINT_FEATURES = {
    MINT_ENABLED: true,  // Mint functionality is now enabled
};

// Import ethers from global scope
const { ethers } = window;

// Import FEATURES from global scope (defined in web3-final.js)
// Use window.FEATURES directly to avoid redeclaration

class JuxGameMint {
    constructor() {
        this.selectedQuantity = 1;
        this.isProcessing = false;
        this.mintRequests = [];
        this.userNFTs = [];
        
        this.init();
    }
    
    async init() {
        this.setupEventListeners();
        this.updateUI();
        
        // Wait for wallet connection
        this.waitForWallet();
    }
    
    waitForWallet() {
        const checkWallet = () => {
            if (window.walletManager && window.walletManager.isConnected()) {
                // Additional check to ensure contracts are ready
                const web3 = window.walletManager.getJuxGameWeb3();
                if (web3 && web3.getContracts && web3.getContracts().juxNFT) {
                    this.onWalletConnected();
                } else {
                    setTimeout(checkWallet, 1000);
                }
            } else {
                setTimeout(checkWallet, 1000);
            }
        };
        checkWallet();
        
        // Listen for wallet connection events
        window.addEventListener('walletConnected', () => {
            // Add a small delay to ensure contracts are fully initialized
            setTimeout(() => {
                this.onWalletConnected();
            }, 1000);
        });
    }
    
    async onWalletConnected() {
        await this.loadMintRequests();
        await this.loadUserNFTs();
        this.updateUI();
    }
    
    setupEventListeners() {
        // Quantity selection
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('quantity-btn')) {
                this.selectQuantity(e.target);
            }
        });
        
        // Mint button
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('summon-btn')) {
                e.preventDefault();
                this.handleMint();
            }
        });
        
        // Reveal buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('reveal-btn')) {
                e.preventDefault();
                const requestId = e.target.dataset.requestId;
                this.handleReveal(requestId);
            }
        });
        
        
        // Refresh data periodically
        setInterval(() => {
            if (window.walletManager && window.walletManager.isConnected()) {
                this.loadMintRequests();
                this.loadUserNFTs();
            }
        }, 30000); // Every 30 seconds
    }
    
    selectQuantity(button) {
        // Remove active class from all buttons
        document.querySelectorAll('.quantity-btn').forEach(btn => {
            btn.classList.remove('active');
        });
        
        // Add active class to selected button
        button.classList.add('active');
        
        // Update selected quantity
        this.selectedQuantity = parseInt(button.dataset.quantity);
        
        // Update UI
        this.updateMintButton();
    }
    
    updateMintButton() {
        const mintButton = document.querySelector('.summon-btn');
        const buttonText = document.querySelector('#summon-btn-text');
        const buttonCost = document.querySelector('#summon-btn-cost');
        
        if (buttonText && buttonCost) {
            if (this.selectedQuantity === 1) {
                buttonText.textContent = 'Summon Blind Box';
            } else {
                buttonText.textContent = `Summon ${this.selectedQuantity} Blind Boxes`;
            }
            
            const totalCost = this.selectedQuantity * 44444;
            buttonCost.textContent = `Cost: ${totalCost.toLocaleString()} $Juxie`;
        }
        
        if (mintButton) {
            mintButton.disabled = this.isProcessing || !window.walletManager?.isConnected();
        }
    }
    
    async handleMint() {
        if (this.isProcessing) return;
        
        // Check if mint feature is enabled
        if (!MINT_FEATURES.MINT_ENABLED) {
            showMintComingSoonNotification();
            return;
        }
        
        if (!window.walletManager || !window.walletManager.isConnected()) {
            this.showNotification('error', 'Please connect your wallet first');
            return;
        }
        
        try {
            this.isProcessing = true;
            this.updateMintButton();
            
            // Check wallet connection
            if (!window.walletManager.isConnected()) {
                throw new Error('Wallet not connected');
            }
            
            // Check $Juxie balance
            const balance = await window.walletManager.getAGBalance();
            const totalCost = this.selectedQuantity * 44444;
            
            if (parseFloat(balance) < totalCost) {
                throw new Error(`Insufficient $Juxie balance. Need ${totalCost.toLocaleString()} $Juxie, have ${parseFloat(balance).toLocaleString()} $Juxie`);
            }
            
            this.showNotification('info', 'Requesting mint... Please confirm the transaction.');
            
            // Request mint
            const tx = await window.walletManager.requestMint(this.selectedQuantity);
            
            this.showNotification('success', `Mint request submitted! Transaction: ${tx.transactionHash.slice(0, 10)}...`);
            
            // Wait for transaction confirmation
            await tx.wait();
            
            this.showNotification('success', 'Mint request confirmed! You can reveal your NFTs in the next block.');
            
            // Reload mint requests
            setTimeout(() => {
                this.loadMintRequests();
            }, 5000);
            
        } catch (error) {
            console.error('Mint failed:', error);
            let errorMessage = 'Mint failed';
            
            if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            } else if (error.message.includes('Insufficient $Juxie balance')) {
                errorMessage = error.message;
            } else if (error.message.includes('Payment failed')) {
                errorMessage = 'Failed to transfer $Juxie tokens. Please approve the contract first.';
            }
            
            this.showNotification('error', errorMessage);
        } finally {
            this.isProcessing = false;
            this.updateMintButton();
        }
    }
    
    async handleReveal(requestId) {
        if (!window.walletManager || !window.walletManager.isConnected()) {
            this.showNotification('error', 'Please connect your wallet first');
            return;
        }
        
        try {
            // First check if the request is still valid
            const web3 = window.walletManager.getJuxGameWeb3();
            const contracts = web3.getContracts();
            
            // Safety check for web3 and provider
            if (!web3 || !web3.provider || !contracts) {
                throw new Error('Web3 not properly initialized');
            }
            
            const request = await contracts.juxNFT.getMintRequest(requestId);
            const currentBlock = await web3.provider.getBlockNumber();
            const requestBlock = Number(request.blockNumber.toString());
            
            if (request.revealed) {
                this.showNotification('info', 'This request has already been revealed');
                this.loadMintRequests();
                this.loadUserNFTs();
                return;
            }
            
            if (currentBlock <= requestBlock) {
                this.showNotification('error', 'Please wait for the next block before revealing');
                return;
            }
            
            this.showNotification('info', 'Revealing NFTs... Please confirm the transaction.');
            
            const tx = await contracts.juxNFT.revealMint(requestId);
            
            this.showNotification('success', `Reveal transaction submitted! ${tx.hash ? tx.hash.slice(0, 10) : 'Hash'}...`);
            
            await tx.wait();
            
            this.showNotification('success', 'NFTs revealed! Check your collection below.');
            
            // Reload data
            setTimeout(() => {
                this.loadMintRequests();
                this.loadUserNFTs();
            }, 3000);
            
        } catch (error) {
            console.error('Reveal failed:', error);
            let errorMessage = 'Reveal failed';
            
            if (error.message.includes('Too early to reveal')) {
                errorMessage = 'Please wait for the next block before revealing';
            } else if (error.message.includes('Reveal window expired')) {
                errorMessage = 'Reveal window has expired for this request. You may need to request a new mint.';
            } else if (error.message.includes('User rejected')) {
                errorMessage = 'Transaction cancelled by user';
            } else if (error.message.includes('Already revealed')) {
                errorMessage = 'This request has already been revealed';
            }
            
            this.showNotification('error', errorMessage);
            
            // Reload data to refresh UI
            setTimeout(() => {
                this.loadMintRequests();
            }, 2000);
        }
    }
    
    
    async loadMintRequests() {
        if (!window.walletManager || !window.walletManager.isConnected()) return;
        
        try {
            // Use wallet manager methods directly
            const web3 = window.walletManager.getJuxGameWeb3();
            
            // Check if contracts are available
            if (!web3 || !web3.getContracts || !web3.provider) {
                console.warn('Web3 or provider not properly initialized in loadMintRequests');
                return;
            }
            
            const contracts = web3.getContracts();
            if (!contracts.juxNFT) {
                console.warn('NFT contract not available');
                return;
            }
            
            const requestIds = await contracts.juxNFT.getUserMintRequests(window.walletManager.getAccount());
            
            this.mintRequests = [];
            
            for (const requestId of requestIds) {
                const request = await contracts.juxNFT.getMintRequest(requestId);
                // Get actual block timestamp
                const blockNumber = Number(request.blockNumber?.toString() || 0);
                let blockTimestamp = null;
                try {
                    // Get provider from web3 state
                    if (web3.provider && typeof web3.provider.getBlock === 'function') {
                        const block = await web3.provider.getBlock(blockNumber);
                        if (block && block.timestamp) {
                            blockTimestamp = block.timestamp;
                            console.log(`Got real timestamp for block ${blockNumber}: ${new Date(block.timestamp * 1000).toISOString()}`);
                        } else {
                            throw new Error('Block timestamp not available');
                        }
                    } else if (web3.provider && typeof web3.provider.getBlockNumber === 'function') {
                        // Fallback: estimate timestamp based on current time and JuChain block time (~3 seconds)
                        const currentBlock = await web3.provider.getBlockNumber();
                        const blockDiff = currentBlock - blockNumber;
                        const estimatedSecondsAgo = blockDiff * 3; // JuChain ~3 seconds per block
                        blockTimestamp = Math.floor(Date.now() / 1000) - estimatedSecondsAgo;
                        console.log(`Estimated timestamp for block ${blockNumber}: ${new Date(blockTimestamp * 1000).toISOString()}`);
                    } else {
                        throw new Error('Provider not available');
                    }
                } catch (error) {
                    // Final fallback: use reasonable estimate
                    console.warn('Failed to get block timestamp, using recent time estimate:', error.message);
                    // Assume the request was made recently (within last hour)
                    blockTimestamp = Math.floor(Date.now() / 1000) - 300; // 5 minutes ago as fallback
                }
                
                this.mintRequests.push({
                    id: requestId,
                    minter: request.minter,
                    quantity: Number(request.quantity?.toString() || 0),
                    blockNumber: blockNumber,
                    blockTimestamp: blockTimestamp,
                    revealed: request.revealed || false,
                    tokenIds: request.tokenIds || []
                });
            }
            
            this.updateMintRequestsUI();
            
        } catch (error) {
            console.error('Failed to load mint requests:', error);
        }
    }
    
    async loadUserNFTs() {
        if (!window.walletManager || !window.walletManager.isConnected()) return;
        
        try {
            // Use wallet manager methods directly
            const web3 = window.walletManager.getJuxGameWeb3();
            
            // Check if contracts are available
            if (!web3 || !web3.getContracts) {
                return;
            }
            
            const contracts = web3.getContracts();
            if (!contracts.juxNFT) {
                return;
            }
            
            const tokenIds = await contracts.juxNFT.getTokensByOwner(window.walletManager.getAccount());
            
            this.userNFTs = [];
            
            for (const tokenId of tokenIds) {
                const tokenInfo = await contracts.juxNFT.getTokenInfo(tokenId);
                this.userNFTs.push({
                    tokenId: tokenId.toString(),
                    petType: Number(tokenInfo[0].toString()),
                    config: {
                        name: tokenInfo[1].name,
                        rarity: Number(tokenInfo[1].rarity.toString()),
                        dailyReward: ethers.formatEther(tokenInfo[1].dailyReward)
                    }
                });
            }
            
            this.updateUserNFTsUI();
            
        } catch (error) {
            console.error('Failed to load user NFTs:', error);
        }
    }
    
    updateMintRequestsUI() {
        const container = document.querySelector('#mint-requests-container');
        if (!container) return;
        
        if (this.mintRequests.length === 0) {
            container.innerHTML = '<p class="text-muted">No pending mint requests</p>';
            return;
        }
        
        let requestsHTML = '';
        
        for (const request of this.mintRequests) {
            const blockNumber = request.blockNumber;
            const quantity = request.quantity;
            const revealed = request.revealed;
            const cost = quantity * 44444;
            
            // Determine status and actions
            let statusBadge = '';
            let actionButton = '';
            
            if (revealed) {
                statusBadge = '<span class="status-badge revealed">Revealed</span>';
                if (request.tokenIds && request.tokenIds.length > 0) {
                    actionButton = `<button class="btn btn-outline btn-small" onclick="viewNFTs('${request.id}')">View NFTs</button>`;
                }
            } else {
                // Use a more realistic block check
                const currentTime = Math.floor(Date.now() / 1000);
                const blockTime = blockNumber;
                const timeDiff = currentTime - blockTime;
                
                if (timeDiff > 3600) { // 1 hour passed
                    statusBadge = '<span class="status-badge expired">Expired</span>';
                    actionButton = `<button class="btn btn-warning btn-small reveal-btn" data-request-id="${request.id}">Try Reveal</button>`;
                } else if (timeDiff > 30) { // 30 seconds passed
                    statusBadge = '<span class="status-badge ready">Ready</span>';
                    actionButton = `<button class="btn btn-primary btn-small reveal-btn" data-request-id="${request.id}">Reveal NFTs</button>`;
                } else {
                    statusBadge = '<span class="status-badge pending">Pending</span>';
                    actionButton = `<span class="text-muted">Wait ${30 - timeDiff}s</span>`;
                }
            }
            
            // Use actual block timestamp if available, otherwise estimate
            let requestTime;
            if (request.blockTimestamp) {
                const date = new Date(request.blockTimestamp * 1000);
                const now = new Date();
                const diffMs = now.getTime() - date.getTime();
                const diffSeconds = Math.floor(diffMs / 1000);
                const diffMinutes = Math.floor(diffSeconds / 60);
                const diffHours = Math.floor(diffMinutes / 60);
                
                if (diffSeconds < 60) {
                    requestTime = `${diffSeconds}s ago`;
                } else if (diffMinutes < 60) {
                    requestTime = `${diffMinutes}m ago`;
                } else if (diffHours < 24) {
                    requestTime = `${diffHours}h ago`;
                } else {
                    requestTime = date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute: '2-digit'});
                }
            } else {
                // Fallback: use current time since we can't reliably estimate block timestamps
                console.warn('No block timestamp available, using current time as fallback');
                requestTime = 'Recently';
            }
            
            requestsHTML += `
                <div class="mint-request-item ${revealed ? 'revealed' : (!revealed && Date.now()/1000 > blockNumber) ? 'ready' : 'pending'}">
                    <div class="request-info">
                        <div class="request-header">
                            <span class="request-id">Request #${request.id.slice(0, 8)}...</span>
                            <span class="request-time">${requestTime}</span>
                        </div>
                        <div class="request-details">
                            <span class="detail-item">📦 ${quantity} NFT${quantity > 1 ? 's' : ''}</span>
                            <span class="detail-item">💰 ${cost.toLocaleString()} $Juxie</span>
                            ${statusBadge}
                        </div>
                    </div>
                    <div class="request-actions">
                        ${actionButton}
                    </div>
                </div>
            `;
        }
        
        container.innerHTML = requestsHTML;
    }
    
    updateUserNFTsUI() {
        const container = document.querySelector('.my-pets-grid');
        if (!container) return;
        
        if (this.userNFTs.length === 0) {
            container.innerHTML = '<p class="no-pets">No NFTs found. Mint your first Axie Pet!</p>';
            return;
        }
        
        container.innerHTML = this.userNFTs.map(nft => {
            const rarityClass = this.getRarityClass(nft.config.rarity);
            // Fix dailyReward calculation - it's already formatted by ethers.formatEther
            const dailyReward = parseFloat(nft.config.dailyReward).toLocaleString();
            
            return `
                <div class="pet-card">
                    <div class="pet-image-container">
                        <img src="https://www.juxgame.site/assets/p${nft.petType}.png" 
                             alt="${nft.config.name} #${nft.tokenId}" 
                             data-pet-type="${nft.petType}"
                             class="pet-image"
                             style="opacity: 1; transition: opacity 0.3s ease-in-out;">
                    </div>
                    <div class="pet-info">
                        <h4>${nft.config.name} #${nft.tokenId}</h4>
                        <span class="pet-rarity ${rarityClass}">${this.getRarityName(nft.config.rarity)}</span>
                        <div class="pet-stats">
                            <span>Earns: ${dailyReward} $Juxie/day</span>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
    }
    
    getRarityClass(rarity) {
        const rarityMap = {
            0: 'common',
            1: 'rare', 
            2: 'legendary'
        };
        return rarityMap[rarity] || 'common';
    }
    
    getRarityName(rarity) {
        const rarityMap = {
            0: 'Common',
            1: 'Rare',
            2: 'Legendary'
        };
        return rarityMap[rarity] || 'Common';
    }
    
    updateUI() {
        this.updateMintButton();
        this.updateMintRequestsUI();
        this.updateUserNFTsUI();
    }
    
    showNotification(type, message, duration = 5000) {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.style.cssText = `
            position: fixed;
            top: 120px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 10000;
            padding: 15px 20px;
            border-radius: 8px;
            max-width: 400px;
            font-weight: 500;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        `;
        
        const colors = {
            success: { bg: '#d4edda', border: '#c3e6cb', color: '#155724' },
            error: { bg: '#f8d7da', border: '#f5c6cb', color: '#721c24' },
            info: { bg: '#d1ecf1', border: '#bee5eb', color: '#0c5460' }
        };
        
        const color = colors[type] || colors.info;
        notification.style.backgroundColor = color.bg;
        notification.style.border = `2px solid ${color.border}`;
        notification.style.color = color.color;
        
        notification.textContent = message;
        
        document.body.appendChild(notification);
        
        // Auto remove
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, duration);
    }
}

// Show mint coming soon notification
function showMintComingSoonNotification() {
    // Remove existing notifications
    document.querySelectorAll('.mint-coming-soon-notification').forEach(n => n.remove());
    
    // Create notification
    const notification = document.createElement('div');
    notification.className = 'mint-coming-soon-notification';
    notification.style.cssText = `
        position: fixed; 
        top: 120px; 
        left: 50%; 
        transform: translateX(-50%); 
        z-index: 10000;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 20px 30px;
        border-radius: 12px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        text-align: center;
        max-width: 400px;
        animation: slideInDown 0.3s ease-out;
    `;
    
    notification.innerHTML = `
        <h3 style="margin: 0 0 10px 0; font-size: 1.2rem;">🎮 Mint Feature Coming Soon!</h3>
        <p style="margin: 0; opacity: 0.9;">NFT minting will be available in the next update. Stay tuned for the official launch!</p>
    `;
    
    // Add animation if it doesn't exist
    if (!document.querySelector('#mint-notification-animation')) {
        const style = document.createElement('style');
        style.id = 'mint-notification-animation';
        style.textContent = `
            @keyframes slideInDown {
                from { opacity: 0; transform: translateX(-50%) translateY(-20px); }
                to { opacity: 1; transform: translateX(-50%) translateY(0); }
            }
        `;
        document.head.appendChild(style);
    }
    
    document.body.appendChild(notification);
    
    // Auto remove after 4 seconds
    setTimeout(() => {
        notification.style.animation = 'slideInDown 0.3s ease-out reverse';
        setTimeout(() => notification.remove(), 300);
    }, 4000);
}

// Initialize mint functionality when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.juxGameMint = new JuxGameMint();
});

// Make available globally
window.JuxGameMint = JuxGameMint;