// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

import "@openzeppelin/contracts/token/ERC721/IERC721.sol";
import "@openzeppelin/contracts/token/ERC20/IERC20.sol";
import "@openzeppelin/contracts/access/Ownable.sol";
import "@openzeppelin/contracts/security/ReentrancyGuard.sol";
import "@openzeppelin/contracts/security/Pausable.sol";
import "./JuxGameNFT.sol";

/**
 * @title StakingRewards
 * @dev Production-ready P2E staking contract with fixed daily rewards based on rarity
 */
contract StakingRewards is Ownable, ReentrancyGuard, Pausable {
    IERC20 public agToken;
    JuxGameNFT public nftContract;
    
    uint256 public constant CLAIM_FEE = 0.1 ether; // 0.1 JU claim fee
    uint256 public constant SECONDS_PER_HOUR = 3600; // 1 hour reward cycle
    uint256 public constant SECONDS_PER_DAY = 86400; // Still used for daily reward display
    
    struct StakedToken {
        uint256 tokenId;
        address owner;
        uint256 stakedAt;
        uint256 lastClaimAt;
        uint256 petType;
        uint256 dailyReward;
    }
    
    // Track total rewards claimed by each user for historical purposes
    mapping(address => uint256) public userTotalClaimed;
    
    mapping(uint256 => StakedToken) public stakedTokens;
    mapping(address => uint256[]) public userStakedTokens;
    mapping(uint256 => uint256) private stakedTokenIndex; // tokenId => index in userStakedTokens
    
    uint256 public totalStaked;
    uint256 public totalFeesCollected;
    
    event TokenStaked(address indexed user, uint256 indexed tokenId, uint256 petType, uint256 dailyReward);
    event TokenUnstaked(address indexed user, uint256 indexed tokenId, uint256 reward);
    event RewardsClaimed(address indexed user, uint256[] tokenIds, uint256 totalReward, uint256 claimFee);
    event FeesWithdrawn(address indexed owner, uint256 amount);
    
    constructor(address _agToken, address _nftContract) {
        agToken = IERC20(_agToken);
        nftContract = JuxGameNFT(_nftContract);
    }
    
    /**
     * @dev Stake NFT to start earning rewards
     * @param tokenId The NFT token ID to stake
     */
    function stakeToken(uint256 tokenId) external nonReentrant whenNotPaused {
        require(nftContract.ownerOf(tokenId) == msg.sender, "Not token owner");
        require(stakedTokens[tokenId].owner == address(0), "Token already staked");
        
        // Get pet type and daily reward from NFT contract
        (uint256 petType, JuxGameNFT.PetConfig memory config) = nftContract.getTokenInfo(tokenId);
        
        // Transfer NFT to this contract
        nftContract.transferFrom(msg.sender, address(this), tokenId);
        
        // Record staking info
        stakedTokens[tokenId] = StakedToken({
            tokenId: tokenId,
            owner: msg.sender,
            stakedAt: block.timestamp,
            lastClaimAt: block.timestamp,
            petType: petType,
            dailyReward: config.dailyReward
        });
        
        // Add to user's staked tokens array
        userStakedTokens[msg.sender].push(tokenId);
        stakedTokenIndex[tokenId] = userStakedTokens[msg.sender].length - 1;
        
        totalStaked++;
        
        emit TokenStaked(msg.sender, tokenId, petType, config.dailyReward);
    }
    
    /**
     * @dev Stake multiple NFTs at once
     * @param tokenIds Array of NFT token IDs to stake
     */
    function stakeBatch(uint256[] calldata tokenIds) external nonReentrant whenNotPaused {
        require(tokenIds.length > 0, "Empty token list");
        require(tokenIds.length <= 50, "Too many tokens");
        
        for (uint256 i = 0; i < tokenIds.length; i++) {
            uint256 tokenId = tokenIds[i];
            require(nftContract.ownerOf(tokenId) == msg.sender, "Not token owner");
            require(stakedTokens[tokenId].owner == address(0), "Token already staked");
            
            // Get pet type and daily reward from NFT contract
            (uint256 petType, JuxGameNFT.PetConfig memory config) = nftContract.getTokenInfo(tokenId);
            
            // Transfer NFT to this contract
            nftContract.transferFrom(msg.sender, address(this), tokenId);
            
            // Record staking info
            stakedTokens[tokenId] = StakedToken({
                tokenId: tokenId,
                owner: msg.sender,
                stakedAt: block.timestamp,
                lastClaimAt: block.timestamp,
                petType: petType,
                dailyReward: config.dailyReward
            });
            
            // Add to user's staked tokens array
            userStakedTokens[msg.sender].push(tokenId);
            stakedTokenIndex[tokenId] = userStakedTokens[msg.sender].length - 1;
            
            emit TokenStaked(msg.sender, tokenId, petType, config.dailyReward);
        }
        
        totalStaked += tokenIds.length;
    }
    
    /**
     * @dev Unstake NFT and claim all pending rewards
     * @param tokenId The NFT token ID to unstake
     */
    function unstakeToken(uint256 tokenId) external payable nonReentrant {
        StakedToken storage stakedToken = stakedTokens[tokenId];
        require(stakedToken.owner == msg.sender, "Not token owner");
        require(msg.value >= CLAIM_FEE, "Insufficient claim fee");
        
        // Calculate pending rewards
        uint256 pendingReward = calculatePendingReward(tokenId);
        
        // Remove from user's staked tokens array
        _removeFromUserTokens(msg.sender, tokenId);
        
        // Transfer NFT back to owner
        nftContract.transferFrom(address(this), msg.sender, tokenId);
        
        // Transfer rewards if any
        if (pendingReward > 0) {
            require(agToken.transfer(msg.sender, pendingReward), "Reward transfer failed");
            // Update user's total claimed amount
            userTotalClaimed[msg.sender] += pendingReward;
        }
        
        // Collect claim fee
        totalFeesCollected += CLAIM_FEE;
        
        // Refund excess JU
        if (msg.value > CLAIM_FEE) {
            payable(msg.sender).transfer(msg.value - CLAIM_FEE);
        }
        
        emit TokenUnstaked(msg.sender, tokenId, pendingReward);
        
        // Clear staking data
        delete stakedTokens[tokenId];
        delete stakedTokenIndex[tokenId];
        totalStaked--;
    }
    
    /**
     * @dev Claim rewards for staked tokens without unstaking
     * @param tokenIds Array of token IDs to claim rewards for
     */
    function claimRewards(uint256[] calldata tokenIds) external payable nonReentrant {
        require(tokenIds.length > 0, "Empty token list");
        require(msg.value >= CLAIM_FEE, "Insufficient claim fee");
        
        uint256 totalReward = 0;
        
        for (uint256 i = 0; i < tokenIds.length; i++) {
            uint256 tokenId = tokenIds[i];
            StakedToken storage stakedToken = stakedTokens[tokenId];
            require(stakedToken.owner == msg.sender, "Not token owner");
            
            uint256 pendingReward = calculatePendingReward(tokenId);
            totalReward += pendingReward;
            
            // Update last claim time
            stakedToken.lastClaimAt = block.timestamp;
        }
        
        require(totalReward > 0, "No rewards to claim");
        
        // Transfer rewards
        require(agToken.transfer(msg.sender, totalReward), "Reward transfer failed");
        
        // Update user's total claimed amount
        userTotalClaimed[msg.sender] += totalReward;
        
        // Collect claim fee
        totalFeesCollected += CLAIM_FEE;
        
        // Refund excess JU
        if (msg.value > CLAIM_FEE) {
            payable(msg.sender).transfer(msg.value - CLAIM_FEE);
        }
        
        emit RewardsClaimed(msg.sender, tokenIds, totalReward, CLAIM_FEE);
    }
    
    /**
     * @dev Calculate pending rewards for a staked token
     * @param tokenId The NFT token ID
     * @return Pending reward amount
     */
    function calculatePendingReward(uint256 tokenId) public view returns (uint256) {
        StakedToken storage stakedToken = stakedTokens[tokenId];
        if (stakedToken.owner == address(0)) {
            return 0;
        }
        
        uint256 timeStaked = block.timestamp - stakedToken.lastClaimAt;
        uint256 hoursStaked = timeStaked / SECONDS_PER_HOUR;
        
        // Calculate hourly reward (dailyReward / 24 hours)
        uint256 hourlyReward = stakedToken.dailyReward / 24;
        
        // Calculate reward based on full hours only
        return hoursStaked * hourlyReward;
    }
    
    /**
     * @dev Get user's staked tokens
     * @param user The user address
     * @return Array of staked token IDs
     */
    function getUserStakedTokens(address user) external view returns (uint256[] memory) {
        return userStakedTokens[user];
    }
    
    /**
     * @dev Get detailed info for user's staked tokens
     * @param user The user address
     * @return Array of StakedToken structs
     */
    function getUserStakedTokensInfo(address user) external view returns (StakedToken[] memory) {
        uint256[] memory tokenIds = userStakedTokens[user];
        StakedToken[] memory tokens = new StakedToken[](tokenIds.length);
        
        for (uint256 i = 0; i < tokenIds.length; i++) {
            tokens[i] = stakedTokens[tokenIds[i]];
        }
        
        return tokens;
    }
    
    /**
     * @dev Calculate total pending rewards for user
     * @param user The user address
     * @return Total pending reward amount
     */
    function getUserTotalPendingRewards(address user) external view returns (uint256) {
        uint256[] memory tokenIds = userStakedTokens[user];
        uint256 totalReward = 0;
        
        for (uint256 i = 0; i < tokenIds.length; i++) {
            totalReward += calculatePendingReward(tokenIds[i]);
        }
        
        return totalReward;
    }
    
    /**
     * @dev Get user's total claimed rewards (historical)
     * @param user The user address
     * @return Total claimed reward amount
     */
    function getUserTotalClaimed(address user) external view returns (uint256) {
        return userTotalClaimed[user];
    }
    
    /**
     * @dev Remove token from user's staked tokens array
     */
    function _removeFromUserTokens(address user, uint256 tokenId) private {
        uint256 index = stakedTokenIndex[tokenId];
        uint256[] storage userTokens = userStakedTokens[user];
        
        // Move last element to the index of element to remove
        if (index < userTokens.length - 1) {
            uint256 lastTokenId = userTokens[userTokens.length - 1];
            userTokens[index] = lastTokenId;
            stakedTokenIndex[lastTokenId] = index;
        }
        
        // Remove last element
        userTokens.pop();
    }
    
    /**
     * @dev Update AG token address (owner only)
     */
    function updateAGToken(address newAGToken) external onlyOwner {
        agToken = IERC20(newAGToken);
    }
    
    /**
     * @dev Withdraw collected claim fees (owner only)
     */
    function withdrawFees() external onlyOwner {
        require(totalFeesCollected > 0, "No fees to withdraw");
        uint256 amount = totalFeesCollected;
        totalFeesCollected = 0;
        
        payable(owner()).transfer(amount);
        emit FeesWithdrawn(owner(), amount);
    }
    
    /**
     * @dev Emergency withdraw AG tokens (owner only)
     */
    function emergencyWithdrawAGTokens() external onlyOwner {
        uint256 balance = agToken.balanceOf(address(this));
        require(balance > 0, "No tokens to withdraw");
        require(agToken.transfer(owner(), balance), "Transfer failed");
    }
    
    /**
     * @dev Emergency pause (owner only)
     */
    function pause() external onlyOwner {
        _pause();
    }
    
    /**
     * @dev Unpause (owner only)
     */
    function unpause() external onlyOwner {
        _unpause();
    }
    
    /**
     * @dev Allow contract to receive BNB for claim fees
     */
    receive() external payable {}
}