// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

import "@openzeppelin/contracts/token/ERC20/IERC20.sol";
import "@openzeppelin/contracts/access/Ownable.sol";
import "@openzeppelin/contracts/security/ReentrancyGuard.sol";
import "@openzeppelin/contracts/security/Pausable.sol";
/**
 * @title RefundContract
 * @dev Handles refunds for expired NFT mint requests
 */
contract RefundContract is Ownable, ReentrancyGuard, Pausable {
    IERC20 public agToken;
    
    // Refund information structure
    struct RefundInfo {
        address minter;           // Original minter address
        uint256 quantity;         // Number of NFTs
        uint256 refundAmount;     // Refund amount in $Juxie
        bool claimed;             // Whether refund has been claimed
        uint256 requestBlockNumber; // Original request block (for verification)
        uint256 addedTimestamp;   // When this refund was added
    }
    
    // Mapping from requestId to refund information
    mapping(bytes32 => RefundInfo) public refunds;
    
    // Track user's refundable requests
    mapping(address => bytes32[]) public userRefundRequests;
    
    // Statistics
    uint256 public totalRefundAmount;
    uint256 public totalClaimedAmount;
    uint256 public refundCount;
    uint256 public claimedCount;
    
    // Events
    event RefundAdded(bytes32 indexed requestId, address indexed minter, uint256 quantity, uint256 refundAmount);
    event RefundClaimed(bytes32 indexed requestId, address indexed minter, uint256 refundAmount);
    event FundsDeposited(address indexed depositor, uint256 amount);
    event FundsWithdrawn(address indexed owner, uint256 amount);
    
    constructor(address _agToken) {
        agToken = IERC20(_agToken);
    }
    
    /**
     * @dev Add refunds for expired mint requests (batch operation)
     * @param requestIds Array of request IDs
     * @param minters Array of minter addresses
     * @param quantities Array of NFT quantities
     * @param blockNumbers Array of original request block numbers
     */
    function addRefunds(
        bytes32[] calldata requestIds,
        address[] calldata minters,
        uint256[] calldata quantities,
        uint256[] calldata blockNumbers
    ) external onlyOwner whenNotPaused {
        require(requestIds.length == minters.length, "Length mismatch: requestIds != minters");
        require(minters.length == quantities.length, "Length mismatch: minters != quantities");
        require(quantities.length == blockNumbers.length, "Length mismatch: quantities != blockNumbers");
        require(requestIds.length > 0, "Empty arrays not allowed");
        require(requestIds.length <= 100, "Too many refunds at once"); // Gas limit protection
        
        uint256 MINT_PRICE = 44444 * 10**18; // Same as NFT contract
        
        for (uint256 i = 0; i < requestIds.length; i++) {
            bytes32 requestId = requestIds[i];
            address minter = minters[i];
            uint256 quantity = quantities[i];
            uint256 blockNumber = blockNumbers[i];
            
            // Validations
            require(minter != address(0), "Invalid minter address");
            require(quantity > 0 && quantity <= 50, "Invalid quantity");
            require(blockNumber > 0, "Invalid block number");
            require(refunds[requestId].minter == address(0), "Refund already exists");
            
            // Calculate refund amount
            uint256 refundAmount = MINT_PRICE * quantity;
            
            // Store refund information
            refunds[requestId] = RefundInfo({
                minter: minter,
                quantity: quantity,
                refundAmount: refundAmount,
                claimed: false,
                requestBlockNumber: blockNumber,
                addedTimestamp: block.timestamp
            });
            
            // Add to user's refund list
            userRefundRequests[minter].push(requestId);
            
            // Update statistics
            totalRefundAmount += refundAmount;
            refundCount++;
            
            emit RefundAdded(requestId, minter, quantity, refundAmount);
        }
    }
    
    /**
     * @dev Claim refund for a single expired request
     * @param requestId The request ID to claim refund for
     */
    function claimRefund(bytes32 requestId) external nonReentrant whenNotPaused {
        RefundInfo storage refund = refunds[requestId];
        
        require(refund.minter == msg.sender, "Not the original minter");
        require(!refund.claimed, "Refund already claimed");
        require(refund.refundAmount > 0, "No refund available");
        
        // Check contract has sufficient funds
        uint256 contractBalance = agToken.balanceOf(address(this));
        require(contractBalance >= refund.refundAmount, "Insufficient contract balance");
        
        // Mark as claimed
        refund.claimed = true;
        
        // Update statistics
        totalClaimedAmount += refund.refundAmount;
        claimedCount++;
        
        // Transfer refund
        require(agToken.transfer(msg.sender, refund.refundAmount), "Refund transfer failed");
        
        emit RefundClaimed(requestId, msg.sender, refund.refundAmount);
    }
    
    /**
     * @dev Claim refunds for multiple expired requests
     * @param requestIds Array of request IDs to claim refunds for
     */
    function claimRefundsBatch(bytes32[] calldata requestIds) external nonReentrant whenNotPaused {
        require(requestIds.length > 0, "Empty request list");
        require(requestIds.length <= 20, "Too many requests at once");
        
        uint256 totalRefund = 0;
        
        // First pass: validate all requests and calculate total
        for (uint256 i = 0; i < requestIds.length; i++) {
            RefundInfo storage refund = refunds[requestIds[i]];
            
            require(refund.minter == msg.sender, "Not the original minter");
            require(!refund.claimed, "Refund already claimed");
            require(refund.refundAmount > 0, "No refund available");
            
            totalRefund += refund.refundAmount;
        }
        
        // Check contract has sufficient funds
        uint256 contractBalance = agToken.balanceOf(address(this));
        require(contractBalance >= totalRefund, "Insufficient contract balance");
        
        // Second pass: mark as claimed and update statistics
        for (uint256 i = 0; i < requestIds.length; i++) {
            RefundInfo storage refund = refunds[requestIds[i]];
            refund.claimed = true;
            totalClaimedAmount += refund.refundAmount;
            claimedCount++;
            
            emit RefundClaimed(requestIds[i], msg.sender, refund.refundAmount);
        }
        
        // Transfer total refund
        require(agToken.transfer(msg.sender, totalRefund), "Batch refund transfer failed");
    }
    
    /**
     * @dev Get user's refundable requests
     * @param user The user address
     * @return Array of request IDs that have refunds available
     */
    function getUserRefundRequests(address user) external view returns (bytes32[] memory) {
        return userRefundRequests[user];
    }
    
    /**
     * @dev Get detailed refund information for user
     * @param user The user address
     * @return Array of RefundInfo structs
     */
    function getUserRefundInfo(address user) external view returns (RefundInfo[] memory) {
        bytes32[] memory requestIds = userRefundRequests[user];
        RefundInfo[] memory userRefunds = new RefundInfo[](requestIds.length);
        
        for (uint256 i = 0; i < requestIds.length; i++) {
            userRefunds[i] = refunds[requestIds[i]];
        }
        
        return userRefunds;
    }
    
    /**
     * @dev Get user's total refundable amount (unclaimed)
     * @param user The user address
     * @return Total amount available for refund
     */
    function getUserTotalRefundable(address user) external view returns (uint256) {
        bytes32[] memory requestIds = userRefundRequests[user];
        uint256 total = 0;
        
        for (uint256 i = 0; i < requestIds.length; i++) {
            RefundInfo storage refund = refunds[requestIds[i]];
            if (!refund.claimed) {
                total += refund.refundAmount;
            }
        }
        
        return total;
    }
    
    /**
     * @dev Check if a specific request has refund available
     * @param requestId The request ID to check
     * @return Whether refund is available for this request
     */
    function hasRefund(bytes32 requestId) external view returns (bool) {
        RefundInfo storage refund = refunds[requestId];
        return refund.minter != address(0) && !refund.claimed;
    }
    
    /**
     * @dev Get refund information for a specific request
     * @param requestId The request ID
     * @return RefundInfo struct
     */
    function getRefundInfo(bytes32 requestId) external view returns (RefundInfo memory) {
        return refunds[requestId];
    }
    
    /**
     * @dev Owner deposits funds for refunds
     * @param amount Amount of tokens to deposit
     */
    function depositFunds(uint256 amount) external onlyOwner {
        require(amount > 0, "Amount must be greater than 0");
        require(agToken.transferFrom(msg.sender, address(this), amount), "Deposit transfer failed");
        
        emit FundsDeposited(msg.sender, amount);
    }
    
    /**
     * @dev Emergency withdrawal of funds (owner only)
     */
    function emergencyWithdraw() external onlyOwner {
        uint256 balance = agToken.balanceOf(address(this));
        require(balance > 0, "No funds to withdraw");
        require(agToken.transfer(owner(), balance), "Emergency withdrawal failed");
        
        emit FundsWithdrawn(owner(), balance);
    }
    
    /**
     * @dev Check if contract has sufficient funds for all pending refunds
     * @return Whether contract can fulfill all refunds
     */
    function hasSufficientFunds() external view returns (bool) {
        uint256 pendingRefunds = totalRefundAmount - totalClaimedAmount;
        uint256 contractBalance = agToken.balanceOf(address(this));
        return contractBalance >= pendingRefunds;
    }
    
    /**
     * @dev Get contract statistics
     * @return totalRefunds Total refund amount added
     * @return totalClaimed Total amount claimed
     * @return pendingRefunds Pending refund amount
     * @return contractBalance Current contract balance
     * @return refundsCount Total number of refunds
     * @return claimsCount Total number of claims
     */
    function getStats() external view returns (
        uint256 totalRefunds,
        uint256 totalClaimed, 
        uint256 pendingRefunds,
        uint256 contractBalance,
        uint256 refundsCount,
        uint256 claimsCount
    ) {
        totalRefunds = totalRefundAmount;
        totalClaimed = totalClaimedAmount;
        pendingRefunds = totalRefundAmount - totalClaimedAmount;
        contractBalance = agToken.balanceOf(address(this));
        refundsCount = refundCount;
        claimsCount = claimedCount;
    }
    
    /**
     * @dev Pause contract (emergency)
     */
    function pause() external onlyOwner {
        _pause();
    }
    
    /**
     * @dev Unpause contract
     */
    function unpause() external onlyOwner {
        _unpause();
    }
    
    /**
     * @dev Update AG token address (for upgrades)
     */
    function updateAGToken(address newAGToken) external onlyOwner {
        require(newAGToken != address(0), "Invalid token address");
        agToken = IERC20(newAGToken);
    }
}