// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

import "@openzeppelin/contracts/token/ERC721/ERC721.sol";
import "@openzeppelin/contracts/token/ERC721/extensions/ERC721Enumerable.sol";
import "@openzeppelin/contracts/token/ERC20/IERC20.sol";
import "@openzeppelin/contracts/access/Ownable.sol";
import "@openzeppelin/contracts/security/ReentrancyGuard.sol";
import "@openzeppelin/contracts/security/Pausable.sol";

/**
 * @title JuxGameNFT
 * @dev Production-ready NFT contract with 2-phase minting mechanism and anti-MEV protection
 */
contract JuxGameNFT is ERC721, ERC721Enumerable, Ownable, ReentrancyGuard, Pausable {
    IERC20 public agToken;
    
    uint256 public constant MINT_PRICE = 44444 * 10**18; // 44,444 AG tokens
    uint256 public constant MAX_BATCH_SIZE = 50;
    
    string private _baseTokenURI;
    uint256 private _tokenIdCounter = 1;
    
    // Rarity distribution: Common(70%), Rare(29%), Legendary(1%)
    uint256 private constant COMMON_THRESHOLD = 70;
    uint256 private constant RARE_THRESHOLD = 99;
    
    // Pet configurations
    struct PetConfig {
        string name;
        uint256 rarity; // 0=Common, 1=Rare, 2=Legendary
        uint256 dailyReward;
    }
    
    mapping(uint256 => PetConfig) public petConfigs;
    mapping(uint256 => uint256) public tokenToPetType;
    
    // 2-phase minting mechanism
    struct MintRequest {
        address minter;
        uint256 quantity;
        uint256 blockNumber;
        bool revealed;
        uint256[] tokenIds;
    }
    
    mapping(bytes32 => MintRequest) public mintRequests;
    mapping(address => bytes32[]) public userMintRequests;
    
    event MintRequested(address indexed minter, bytes32 indexed requestId, uint256 quantity, uint256 blockNumber);
    event MintRevealed(address indexed minter, bytes32 indexed requestId, uint256[] tokenIds, uint256[] petTypes);
    event TokenURIUpdated(string newBaseURI);
    
    constructor(
        address _agToken,
        string memory _initialBaseURI
    ) ERC721("JuxGame NFT", "JUX") {
        agToken = IERC20(_agToken);
        _baseTokenURI = _initialBaseURI;
        
        // Initialize pet configurations
        _initializePetConfigs();
    }
    
    function _initializePetConfigs() private {
        // Common pets (70% chance)
        petConfigs[1] = PetConfig("Aqua Striker", 0, 4444 * 10**18);
        petConfigs[2] = PetConfig("Terra Guardian", 0, 4444 * 10**18);
        
        // Rare pets (29% chance)  
        petConfigs[3] = PetConfig("Aero Blitzer", 1, 8888 * 10**18);
        petConfigs[4] = PetConfig("Ember Drake", 1, 8888 * 10**18);
        
        // Legendary pets (1% chance)
        petConfigs[5] = PetConfig("Chrono Weaver", 2, 22222 * 10**18);
        petConfigs[6] = PetConfig("Void Lurker", 2, 22222 * 10**18);
    }
    
    /**
     * @dev Phase 1: Request mint with payment
     * @param quantity Number of NFTs to mint (1-50)
     */
    function requestMint(uint256 quantity) external nonReentrant whenNotPaused {
        require(quantity > 0 && quantity <= MAX_BATCH_SIZE, "Invalid quantity");
        
        uint256 totalCost = MINT_PRICE * quantity;
        require(agToken.transferFrom(msg.sender, address(this), totalCost), "Payment failed");
        
        bytes32 requestId = keccak256(abi.encodePacked(msg.sender, block.timestamp, block.number, quantity));
        
        mintRequests[requestId] = MintRequest({
            minter: msg.sender,
            quantity: quantity,
            blockNumber: block.number,
            revealed: false,
            tokenIds: new uint256[](0)
        });
        
        userMintRequests[msg.sender].push(requestId);
        
        emit MintRequested(msg.sender, requestId, quantity, block.number);
    }
    
    /**
     * @dev Phase 2: Reveal mint results after next block
     * @param requestId The mint request ID to reveal
     */
    function revealMint(bytes32 requestId) external nonReentrant {
        MintRequest storage request = mintRequests[requestId];
        require(request.minter == msg.sender, "Not request owner");
        require(!request.revealed, "Already revealed");
        require(block.number > request.blockNumber, "Too early to reveal");
        require(block.number <= request.blockNumber + 256, "Reveal window expired");
        
        uint256[] memory tokenIds = new uint256[](request.quantity);
        uint256[] memory petTypes = new uint256[](request.quantity);
        
        for (uint256 i = 0; i < request.quantity; i++) {
            uint256 tokenId = _tokenIdCounter++;
            uint256 petType = _generateRandomPetType(requestId, i);
            
            tokenIds[i] = tokenId;
            petTypes[i] = petType;
            
            tokenToPetType[tokenId] = petType;
            _safeMint(msg.sender, tokenId);
        }
        
        request.revealed = true;
        request.tokenIds = tokenIds;
        
        emit MintRevealed(msg.sender, requestId, tokenIds, petTypes);
    }
    
    /**
     * @dev Generate random pet type based on rarity distribution
     */
    function _generateRandomPetType(bytes32 requestId, uint256 index) private view returns (uint256) {
        uint256 randomValue = uint256(keccak256(abi.encodePacked(
            blockhash(mintRequests[requestId].blockNumber + 1),
            requestId,
            index,
            block.difficulty,
            block.timestamp
        ))) % 100;
        
        if (randomValue < COMMON_THRESHOLD) {
            // Common pets: 70% chance, return pet 1 or 2
            return (randomValue % 2) + 1;
        } else if (randomValue < RARE_THRESHOLD) {
            // Rare pets: 29% chance, return pet 3 or 4
            return ((randomValue - COMMON_THRESHOLD) % 2) + 3;
        } else {
            // Legendary pets: 1% chance, return pet 5 or 6
            return ((randomValue - RARE_THRESHOLD) % 2) + 5;
        }
    }
    
    /**
     * @dev Get user's mint requests
     */
    function getUserMintRequests(address user) external view returns (bytes32[] memory) {
        return userMintRequests[user];
    }
    
    /**
     * @dev Get mint request details
     */
    function getMintRequest(bytes32 requestId) external view returns (MintRequest memory) {
        return mintRequests[requestId];
    }
    
    /**
     * @dev Get pet configuration
     */
    function getPetConfig(uint256 petType) external view returns (PetConfig memory) {
        require(petType >= 1 && petType <= 6, "Invalid pet type");
        return petConfigs[petType];
    }
    
    /**
     * @dev Get token's pet type and config
     */
    function getTokenInfo(uint256 tokenId) external view returns (uint256 petType, PetConfig memory config) {
        require(_exists(tokenId), "Token does not exist");
        petType = tokenToPetType[tokenId];
        config = petConfigs[petType];
    }
    
    /**
     * @dev Set base URI for metadata (owner only)
     */
    function setBaseURI(string calldata newBaseURI) external onlyOwner {
        _baseTokenURI = newBaseURI;
        emit TokenURIUpdated(newBaseURI);
    }
    
    /**
     * @dev Update AG token address (owner only)
     */
    function updateAGToken(address newAGToken) external onlyOwner {
        agToken = IERC20(newAGToken);
    }
    
    /**
     * @dev Withdraw collected AG tokens (owner only)
     */
    function withdrawAGTokens() external onlyOwner {
        uint256 balance = agToken.balanceOf(address(this));
        require(balance > 0, "No tokens to withdraw");
        require(agToken.transfer(owner(), balance), "Transfer failed");
    }
    
    /**
     * @dev Emergency pause (owner only)
     */
    function pause() external onlyOwner {
        _pause();
    }
    
    /**
     * @dev Unpause (owner only)
     */
    function unpause() external onlyOwner {
        _unpause();
    }
    
    /**
     * @dev Override required by Solidity for multiple inheritance
     */
    function _beforeTokenTransfer(
        address from,
        address to,
        uint256 tokenId,
        uint256 batchSize
    ) internal override(ERC721, ERC721Enumerable) whenNotPaused {
        super._beforeTokenTransfer(from, to, tokenId, batchSize);
    }
    
    /**
     * @dev Override required by Solidity for multiple inheritance
     */
    function supportsInterface(bytes4 interfaceId)
        public
        view
        override(ERC721, ERC721Enumerable)
        returns (bool)
    {
        return super.supportsInterface(interfaceId);
    }
    
    /**
     * @dev Override to return custom base URI
     */
    function _baseURI() internal view override returns (string memory) {
        return _baseTokenURI;
    }
    
    /**
     * @dev Get tokens owned by address
     */
    function getTokensByOwner(address owner) external view returns (uint256[] memory) {
        uint256 balance = balanceOf(owner);
        uint256[] memory tokens = new uint256[](balance);
        
        for (uint256 i = 0; i < balance; i++) {
            tokens[i] = tokenOfOwnerByIndex(owner, i);
        }
        
        return tokens;
    }
}