import React, { useEffect, useRef } from 'react';

const TerminalMain = ({ messages, isLoading, connectionStatus }) => {
  const outputRef = useRef(null);
  const messagesEndRef = useRef(null);

  useEffect(() => {
    const container = outputRef.current;
    if (!container) return;
    const isAtBottom = container.scrollHeight - container.scrollTop - container.clientHeight < 80; // 近底部
    if (isAtBottom && messagesEndRef.current) {
      messagesEndRef.current.scrollIntoView({ behavior: 'smooth' });
    }
  }, [messages]);

  const formatTimestamp = (timestamp) => {
    if (!timestamp) return '';
    try {
      const date = new Date(timestamp);
      return date.toLocaleTimeString('zh-CN', {
        hour12: false,
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
      });
    } catch (error) {
      return '';
    }
  };

  const getMessageClass = (type, messageType) => {
    if (type === 'system') {
      switch (messageType) {
        case 'error': return 'system-message';
        case 'warning': return 'system-message';
        case 'success': return 'system-message';
        case 'info': return 'system-message';
        case 'command': return 'system-message';
        default: return 'system-message';
      }
    }
    return 'agent-message';
  };

  const getMessageColor = (type, messageType) => {
    if (type === 'system') {
      switch (messageType) {
        case 'error': return 'var(--terminal-error)';
        case 'warning': return 'var(--terminal-warning)';
        case 'success': return 'var(--terminal-success)';
        case 'info': return 'var(--terminal-info)';
        case 'command': return 'var(--terminal-command)';
        default: return 'var(--terminal-text-primary)';
      }
    }
    return 'var(--terminal-agent)';
  };

  const getConnectionMessage = () => {
    switch (connectionStatus) {
      case 'disconnected':
        return '🔌 未连接到A股AI分析室';
      case 'connecting':
        return '🔄 正在连接A股AI分析室...';
      case 'connected':
        return null;
      default:
        return '❓ 连接状态未知';
    }
  };

  const getRoleDisplayName = (role) => {
    const roleMap = {
      'DataAnalyst': '数据分析师',
      'TechnicalAnalyst': '技术分析师', 
      'FundamentalAnalyst': '基本面分析师',
      'IndustryAnalyst': '行业研究员',
      'MoneyFlowAnalyst': '资金流向分析师',
      'RiskManager': '风险管理师',
      'InvestmentAdvisor': '投资顾问',
      'user': '用户'
    };
    return roleMap[role] || role;
  };

  return (
    <div className="terminal-main">
      <div className="terminal-output" ref={outputRef}>
        {connectionStatus !== 'connected' && (
          <div className="connection-message">
            {getConnectionMessage()}
          </div>
        )}
        
        {messages.map((message) => (
          <div
            key={message.id}
            className={getMessageClass(message.type, message.messageType)}
            style={{ color: getMessageColor(message.type, message.messageType) }}
          >
            <span className="message-timestamp">
              [{formatTimestamp(message.timestamp)}]
            </span>
            
            {message.type === 'agent' && (
              <span className="message-author">
                {message.agent_name}({getRoleDisplayName(message.agent_role)}):
              </span>
            )}
            
            <span className="message-content">
              {message.content}
            </span>
          </div>
        ))}
        
        {isLoading && (
          <div className="loading-message">
            <span className="loading-spinner">⟳</span>
            AI分析师正在思考中...
          </div>
        )}
        
        <div ref={messagesEndRef} />
      </div>
    </div>
  );
};

export default TerminalMain;