import React, { useState, useEffect } from 'react';

const SystemLog = () => {
  const [logs, setLogs] = useState([]);
  const [isCollapsed, setIsCollapsed] = useState(true);

  const addLog = (message, type = 'info') => {
    const timestamp = new Date().toLocaleTimeString('zh-CN', {
      hour12: false,
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit'
    });
    
    const newLog = {
      id: Date.now() + Math.random(),
      timestamp,
      message,
      type
    };
    
    setLogs(prev => {
      const newLogs = [...prev, newLog];
      // 保持最近100条日志
      return newLogs.slice(-100);
    });
  };

  useEffect(() => {
    // 监听全局事件
    window.addSystemLog = addLog;
    
    // 初始化日志
    addLog('A股AI分析室系统已启动', 'success');
    addLog('正在连接后端服务...', 'info');
    
    return () => {
      // 清理全局函数
      if (window.addSystemLog === addLog) {
        window.addSystemLog = undefined;
      }
    };
  }, []);

  const getLogIcon = (type) => {
    switch (type) {
      case 'success': return '✅';
      case 'error': return '❌';
      case 'warning': return '⚠️';
      case 'trigger': return '🚀';
      default: return 'ℹ️';
    }
  };

  const getLogClass = (type) => {
    return `log-${type}`;
  };

  return (
    <div className="system-log">
      <div 
        className="section-title clickable" 
        onClick={() => setIsCollapsed(!isCollapsed)}
      >
        系统日志 {isCollapsed ? '▶' : '▼'}
      </div>
      
      {!isCollapsed && (
        <div className="log-container">
          <div className="log-header">
            <button 
              className="clear-logs-btn" 
              onClick={() => setLogs([])}
              title="清除日志"
            >
              🗑️
            </button>
            <span className="log-count">{logs.length} 条记录</span>
          </div>
          
          <div className="log-entries">
            {logs.length === 0 ? (
              <div className="no-logs">暂无日志记录</div>
            ) : (
              logs.map(log => (
                <div key={log.id} className={`log-entry ${getLogClass(log.type)}`}>
                  <span className="log-timestamp">[{log.timestamp}]</span>
                  <span className="log-icon">{getLogIcon(log.type)}</span>
                  <span className="log-message">{log.message}</span>
                </div>
              ))
            )}
          </div>
        </div>
      )}
    </div>
  );
};

export default SystemLog;