import React, { useState, useRef } from 'react';

const AgentPanel = ({ agents }) => {
  const [hoveredAgent, setHoveredAgent] = useState(null);
  const [tooltipPosition, setTooltipPosition] = useState({ top: 0, left: 0 });
  const cardRefs = useRef({});

  const getRoleShortName = (role) => {
    const shortNames = {
      'DataAnalyst': '数据',
      'TechnicalAnalyst': '技术', 
      'FundamentalAnalyst': '基本面',
      'IndustryAnalyst': '行业',
      'MoneyFlowAnalyst': '资金',
      'RiskManager': '风险',
      'InvestmentAdvisor': '投顾'
    };
    return shortNames[role] || role.slice(0, 2);
  };

  const getRoleDisplayName = (role) => {
    const roleNames = {
      'DataAnalyst': '数据分析师',
      'TechnicalAnalyst': '技术分析师', 
      'FundamentalAnalyst': '基本面分析师',
      'IndustryAnalyst': '行业研究员',
      'MoneyFlowAnalyst': '资金流向分析师',
      'RiskManager': '风险管理师',
      'InvestmentAdvisor': '投资顾问'
    };
    return roleNames[role] || role;
  };

  const getAgentColor = (role) => {
    const colors = {
      'DataAnalyst': '#3b82f6',
      'TechnicalAnalyst': '#8b5cf6',
      'FundamentalAnalyst': '#10b981',
      'IndustryAnalyst': '#f59e0b',
      'MoneyFlowAnalyst': '#f97316',
      'RiskManager': '#ef4444',
      'InvestmentAdvisor': '#6366f1'
    };
    return colors[role] || '#64748b';
  };

  const getModelDisplay = (model) => {
    const modelNames = {
      'gpt-4o': 'GPT-4o',
      'claude-sonnet-4-20250514': 'Claude Sonnet 4',
      'doubao-seed-1-6-thinking-250615': 'Doubao Thinking',
      'doubao-seed-1-6-flash': 'Doubao Flash',
      'doubao-seed-1-6-flash-250615': 'Doubao Flash',
      'deepseek-r1-250528': 'DeepSeek R1',
      'gemini-2.5-pro-preview-06-05': 'Gemini 2.5 Pro',
      'qwen3-235b-a22b-think': 'Qwen3 235B',
      'o3': 'OpenAI o3'
    };
    return modelNames[model] || model;
  };

  const formatPersonality = (personality) => {
    if (!personality) return '';
    return personality.length > 40 ? personality.substring(0, 40) + '...' : personality;
  };

  const formatExpertise = (expertise) => {
    if (!expertise || !Array.isArray(expertise)) return '';
    return expertise.slice(0, 3).join(', ');
  };

  const handleMouseEnter = (role, event) => {
    const card = cardRefs.current[role];
    if (card) {
      const rect = card.getBoundingClientRect();
      setTooltipPosition({
        top: rect.top - 10,
        left: rect.right + 10
      });
    }
    setHoveredAgent(role);
  };

  const handleMouseLeave = () => {
    setHoveredAgent(null);
  };

  if (!agents || Object.keys(agents).length === 0) {
    return (
      <div className="agent-panel">
        <div className="section-title">AI分析师</div>
        <div className="agents-loading">加载分析师中...</div>
      </div>
    );
  }

  return (
    <div className="agent-panel">
      <div className="section-title">AI分析师 ({Object.keys(agents).length}位)</div>
      
      <div className="agents-grid">
        {Object.entries(agents).map(([role, agent]) => (
          <div
            key={role}
            ref={el => cardRefs.current[role] = el}
            className="agent-card"
            style={{ borderLeft: `3px solid ${getAgentColor(role)}` }}
            onMouseEnter={(e) => handleMouseEnter(role, e)}
            onMouseLeave={handleMouseLeave}
          >
            <div className="agent-header">
              <div className="agent-avatar" style={{ backgroundColor: getAgentColor(role) }}>
                {getRoleShortName(role)}
              </div>
              <div className="agent-info">
                <div className="agent-name">{agent.name}</div>
                <div className="agent-role">{getRoleDisplayName(role)}</div>
              </div>
            </div>
            
            <div className="agent-model">
              {getModelDisplay(agent.model)}
            </div>
            
            <div className="agent-status">
              <div className="status-dot online"></div>
              <span>在线</span>
            </div>
          </div>
        ))}
      </div>

      {/* Tooltip */}
      {hoveredAgent && agents[hoveredAgent] && (
        <div 
          className="agent-tooltip"
          style={{
            position: 'fixed',
            top: tooltipPosition.top,
            left: tooltipPosition.left,
            zIndex: 1000
          }}
        >
          <div className="tooltip-header">
            <strong>{agents[hoveredAgent].name}</strong>
            <span className="tooltip-role">({getRoleDisplayName(hoveredAgent)})</span>
          </div>
          
          <div className="tooltip-section">
            <div className="tooltip-label">模型:</div>
            <div className="tooltip-value">{getModelDisplay(agents[hoveredAgent].model)}</div>
          </div>
          
          <div className="tooltip-section">
            <div className="tooltip-label">性格特点:</div>
            <div className="tooltip-value">{agents[hoveredAgent].personality}</div>
          </div>
          
          {agents[hoveredAgent].expertise && (
            <div className="tooltip-section">
              <div className="tooltip-label">专业领域:</div>
              <div className="tooltip-value">{formatExpertise(agents[hoveredAgent].expertise)}</div>
            </div>
          )}
          
          {agents[hoveredAgent].quirks && agents[hoveredAgent].quirks.length > 0 && (
            <div className="tooltip-section">
              <div className="tooltip-label">个人习惯:</div>
              <div className="tooltip-value">{agents[hoveredAgent].quirks.join(', ')}</div>
            </div>
          )}
        </div>
      )}
    </div>
  );
};

export default AgentPanel;