import React, { useState, useEffect, useRef } from 'react';
import TerminalHeader from './components/TerminalHeader';
import TerminalSidebar from './components/TerminalSidebar';
import TerminalMain from './components/TerminalMain';
import TerminalInput from './components/TerminalInput';

function App() {
  const [agents, setAgents] = useState({});
  const [messages, setMessages] = useState([]);
  const [marketData, setMarketData] = useState({});
  const [officeState, setOfficeState] = useState({});
  const [connectionStatus, setConnectionStatus] = useState('disconnected');
  const [wsError, setWsError] = useState(null);
  const [commandHistory, setCommandHistory] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [lowCostMode, setLowCostMode] = useState(false);
  
  const ws = useRef(null);
  const reconnectTimeoutRef = useRef(null);
  const reconnectAttempts = useRef(0);
  const maxReconnectAttempts = 5;

  // Add initialization messages
  useEffect(() => {
    addSystemMessage('🏢 A股AI分析室 v1.0 - 正在初始化...');
    addSystemMessage('📡 连接专业分析师团队...');
    
    // Get initial low cost mode status
    fetchLowCostModeStatus();
  }, []);

  const addSystemMessage = (content, type = 'system') => {
    const message = {
      id: Date.now() + Math.random(),
      type: 'system',
      content,
      timestamp: new Date().toISOString(),
      messageType: type
    };
    setMessages(prev => [...prev, message]);
  };

  const connectWebSocket = () => {
    if (ws.current?.readyState === WebSocket.OPEN) {
      return;
    }

    try {
      const wsUrl = process.env.NODE_ENV === 'production' 
        ? `wss://${window.location.host}/office/ws`
        : 'ws://localhost:8000/ws';
      
      ws.current = new WebSocket(wsUrl);
      setConnectionStatus('connecting');
      addSystemMessage('🔄 正在连接A股AI分析室...');

      ws.current.onopen = () => {
        console.log('WebSocket connected');
        setConnectionStatus('connected');
        setWsError(null);
        reconnectAttempts.current = 0;
        addSystemMessage('✅ 成功连接到A股AI分析室！');
        addSystemMessage('🚀 AI分析师团队已激活，正在监控市场数据...');
        addSystemMessage('💡 输入 "help" 查看可用命令');
        
        // 系统日志
        if (window.addSystemLog) {
          window.addSystemLog('WebSocket connected successfully', 'success');
        }
      };

      ws.current.onmessage = (event) => {
        try {
          const data = JSON.parse(event.data);
          handleWebSocketMessage(data);
        } catch (error) {
          console.error('Error parsing WebSocket message:', error);
        }
      };

      ws.current.onerror = (error) => {
        console.error('WebSocket error:', error);
        setWsError('连接错误');
        addSystemMessage('❌ WebSocket连接错误', 'error');
      };

      ws.current.onclose = (event) => {
        console.log('WebSocket disconnected:', event.code, event.reason);
        setConnectionStatus('disconnected');
        addSystemMessage('🔌 连接断开，正在尝试重连...', 'warning');
        
        if (reconnectAttempts.current < maxReconnectAttempts) {
          const delay = Math.min(1000 * Math.pow(2, reconnectAttempts.current), 10000);
          reconnectTimeoutRef.current = setTimeout(() => {
            reconnectAttempts.current++;
            addSystemMessage(`🔄 重连尝试 ${reconnectAttempts.current}/${maxReconnectAttempts}...`);
            connectWebSocket();
          }, delay);
        } else {
          setWsError('多次重连失败');
          addSystemMessage('💀 连接失败，请刷新页面。', 'error');
        }
      };

    } catch (error) {
      console.error('Error creating WebSocket connection:', error);
      setWsError('创建连接失败');
      addSystemMessage('💀 创建WebSocket连接失败', 'error');
    }
  };

  const handleWebSocketMessage = (data) => {
    switch (data.type) {
      case 'agents_info':
        console.log('Received agents info:', data.data);
        setAgents(data.data);
        addSystemMessage(`👥 已加载 ${Object.keys(data.data).length} 位AI分析师`);
        break;
      
      case 'chat_message':
        const message = {
          ...data.data,
          id: Date.now() + Math.random(),
          type: 'agent'
        };
        setMessages(prev => [...prev, message]);
        break;
      
      case 'market_data':
        setMarketData(data.data);
        const marketInfo = data.data;
        if (marketInfo?.sh_index?.current) {
          addSystemMessage(`📊 市场更新: 上证指数 ${marketInfo.sh_index.current.toFixed(2)}`);
        }
        break;
      
      case 'office_status':
        setOfficeState(data.data);
        break;
      
      case 'chat_history':
        // Batch load historical messages
        const historyMsgs = data.data.map(item => ({ ...item, id: Date.now() + Math.random(), type: 'agent' }));
        setMessages(prev => [...prev, ...historyMsgs]);
        break;
      
      case 'system_message':
        addSystemMessage(data.data.content, 'info');
        break;
      
      default:
        console.log('Unknown message type:', data.type);
    }
  };

  const sendMessage = (message) => {
    if (ws.current?.readyState === WebSocket.OPEN) {
      ws.current.send(JSON.stringify(message));
    } else {
      addSystemMessage('❌ 未连接到服务器', 'error');
    }
  };

  const executeCommand = (command) => {
    const trimmed = command.trim().toLowerCase();
    
    // Add to history
    setCommandHistory(prev => [...prev, command]);
    
    // Show user input
    addSystemMessage(`$ ${command}`, 'command');
    
    // System log
    if (window.addSystemLog) {
      window.addSystemLog(`Command executed: ${command}`, 'trigger');
    }
    
    setIsLoading(true);

    switch (trimmed) {
      case 'help':
        addSystemMessage('📚 可用命令:', 'info');
        addSystemMessage('  trigger market     - 触发市场分析讨论');
        addSystemMessage('  trigger risk        - 触发风险评估讨论');
        addSystemMessage('  trigger strategy    - 触发交易策略讨论');
        addSystemMessage('  trigger meeting     - 召开团队会议');
        addSystemMessage('  status             - 显示当前状态');
        addSystemMessage('  agents             - 列出所有AI分析师');
        addSystemMessage('  market             - 显示市场数据');
        addSystemMessage('  clear              - 清除终端');
        addSystemMessage('  help               - 显示帮助信息');
        addSystemMessage('');  
        addSystemMessage('💡 您也可以输入任何消息（新闻、问题等）来触发AI分析师讨论！');
        break;

      case 'trigger market':
        sendMessage({
          type: 'manual_trigger',
          topics: ['market_analysis']
        });
        addSystemMessage('🚀 正在触发市场分析讨论...', 'success');
        break;

      case 'trigger risk':
        sendMessage({
          type: 'manual_trigger',
          topics: ['risk_assessment']
        });
        addSystemMessage('⚠️ 正在触发风险评估讨论...', 'success');
        break;

      case 'trigger strategy':
        sendMessage({
          type: 'manual_trigger',
          topics: ['trading_strategy']
        });
        addSystemMessage('💡 正在触发策略讨论...', 'success');
        break;

      case 'trigger meeting':
        sendMessage({
          type: 'manual_trigger',
          topics: ['meeting', 'summary']
        });
        addSystemMessage('🏢 正在召开团队会议...', 'success');
        break;

      case 'status':
        sendMessage({ type: 'get_status' });
        addSystemMessage('📊 正在获取状态信息...', 'info');
        break;

      case 'agents':
        if (Object.keys(agents).length > 0) {
          addSystemMessage('👥 当前AI分析师:', 'info');
          Object.entries(agents).forEach(([role, agent]) => {
            addSystemMessage(`  • ${agent.name} (${role}) - ${agent.personality}`);
          });
        } else {
          addSystemMessage('❌ 分析师团队尚未加载', 'warning');
        }
        break;

      case 'market':
        if (marketData.sh_index || marketData.sz_index) {
          addSystemMessage('📈 当前市场数据:', 'info');
          if (marketData.sh_index) {
            const sh = marketData.sh_index;
            addSystemMessage(`  上证指数: ${sh.current?.toFixed(2)} (${sh.change_pct > 0 ? '+' : ''}${sh.change_pct?.toFixed(2)}%)`);
          }
          if (marketData.sz_index) {
            const sz = marketData.sz_index;
            addSystemMessage(`  深证成指: ${sz.current?.toFixed(2)} (${sz.change_pct > 0 ? '+' : ''}${sz.change_pct?.toFixed(2)}%)`);
          }
        } else {
          addSystemMessage('❌ 暂无市场数据', 'warning');
        }
        break;

      case 'clear':
        setMessages([]);
        addSystemMessage('🧹 终端已清空');
        break;
      
      default:
        // Check if it's a user input (not a command)
        if (!trimmed.startsWith('/') && trimmed.length > 0) {
          // Send as user input to trigger AI discussion
          sendMessage({
            type: 'user_input',
            content: command,
            input_type: 'general'
          });
          addSystemMessage('📝 您的输入已分享给AI分析师...', 'success');
          addSystemMessage('🤖 AI分析师正在分析，稍后将回复...', 'info');
          break;
        }

        addSystemMessage(`❌ 未知命令: ${command}`, 'error');
        addSystemMessage('💡 输入 "help" 查看可用命令');
        break;
    }

    setTimeout(() => setIsLoading(false), 500);
  };

  const getStatus = () => {
    sendMessage({
      type: 'get_status'
    });
  };

  const fetchLowCostModeStatus = async () => {
    try {
      const baseUrl = process.env.NODE_ENV === 'production' 
        ? `https://${window.location.host}/office`
        : 'http://localhost:8000';
      
      const response = await fetch(`${baseUrl}/api/low-cost-mode`);
      const data = await response.json();
      setLowCostMode(data.low_cost_mode);
    } catch (error) {
      console.error('Failed to fetch low cost mode status:', error);
    }
  };

  const toggleLowCostMode = async (enabled) => {
    try {
      const baseUrl = process.env.NODE_ENV === 'production' 
        ? `https://${window.location.host}/office`
        : 'http://localhost:8000';
      
      const response = await fetch(`${baseUrl}/api/low-cost-mode`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ enabled }),
      });
      
      const data = await response.json();
      setLowCostMode(enabled);
      
      addSystemMessage(
        `💰 低费模式已${enabled ? '开启' : '关闭'} - ${data.message}`, 
        enabled ? 'success' : 'info'
      );
      
      if (enabled) {
        addSystemMessage('⚡ 所有分析师现在使用doubao-seed-1-6-flash-250615模型以优化费用', 'info');
      } else {
        addSystemMessage('🔄 所有分析师已恢复到原始模型', 'info');
      }
      
    } catch (error) {
      console.error('Failed to toggle low cost mode:', error);
      addSystemMessage('❌ 切换低费模式失败', 'error');
      // Revert the state change
      setLowCostMode(!enabled);
    }
  };

  useEffect(() => {
    connectWebSocket();

    const statusInterval = setInterval(() => {
      if (connectionStatus === 'connected') {
        getStatus();
      }
    }, 30000); // Get status every 30 seconds

    return () => {
      clearInterval(statusInterval);
      if (reconnectTimeoutRef.current) {
        clearTimeout(reconnectTimeoutRef.current);
      }
      if (ws.current) {
        ws.current.close();
      }
    };
  }, []);

  return (
    <div className="terminal-app">
      <TerminalHeader 
        connectionStatus={connectionStatus}
        wsError={wsError}
        agentsCount={Object.keys(agents).length}
        lowCostMode={lowCostMode}
        onLowCostModeChange={toggleLowCostMode}
      />
      
      <div className="terminal-content">
        <TerminalSidebar 
          agents={agents}
          marketData={marketData}
          officeState={officeState}
        />
        
        <div className="terminal-main">
          <TerminalMain 
            messages={messages}
            isLoading={isLoading}
            connectionStatus={connectionStatus}
          />
          
          <TerminalInput 
            onExecuteCommand={executeCommand}
            commandHistory={commandHistory}
            isConnected={connectionStatus === 'connected'}
          />
        </div>
      </div>
    </div>
  );
}

export default App;