import requests
import asyncio
import json
import random
import pandas as pd
from datetime import datetime, timedelta
from typing import Dict, List, Optional
from config import TUSHARE_API_TOKEN, TUSHARE_API_BASE, SINA_API_BASE, EASTMONEY_API_BASE
import warnings
warnings.filterwarnings('ignore')

class DataSource:
    """数据源基类"""
    
    async def fetch_data(self) -> Dict:
        raise NotImplementedError

class TushareSource(DataSource):
    """Tushare Pro API数据源 - A股专业数据"""
    
    def __init__(self):
        self.base_url = TUSHARE_API_BASE
        self.token = TUSHARE_API_TOKEN
        
    def _make_request(self, api_name: str, fields: str = None, **params) -> Dict:
        """统一的API请求方法"""
        data = {
            "api_name": api_name,
            "token": self.token,
            "params": params
        }
        if fields:
            data["fields"] = fields
            
        try:
            response = requests.post(self.base_url, json=data, timeout=10)
            response.raise_for_status()
            result = response.json()
            
            if result.get("code") == 0:
                return result.get("data", {})
            else:
                print(f"Tushare API error: {result.get('msg', 'Unknown error')}")
                return {}
        except Exception as e:
            print(f"Tushare request error for {api_name}: {e}")
            return {}
    
    async def fetch_stock_basic(self) -> List[Dict]:
        """获取股票基本信息"""
        data = self._make_request(
            "stock_basic",
            fields="ts_code,symbol,name,area,industry,market,list_date",
            list_status="L",
            market="主板"
        )
        # 返回items列表，如果没有则返回空列表
        return data.get("items", []) if isinstance(data, dict) else []
    
    async def fetch_index_daily(self, ts_codes: List[str]) -> Dict:
        """获取指数日线数据"""
        result = {}
        trade_date = datetime.now().strftime("%Y%m%d")
        
        for ts_code in ts_codes:
            data = self._make_request(
                "index_daily",
                fields="ts_code,trade_date,close,pct_chg,vol,amount",
                ts_code=ts_code,
                trade_date=trade_date
            )
            if data and "items" in data and data["items"]:
                result[ts_code] = data["items"][0]
        return result
    
    async def fetch_daily_basic(self, ts_codes: List[str]) -> Dict:
        """获取股票每日指标"""
        result = {}
        trade_date = datetime.now().strftime("%Y%m%d")
        
        for ts_code in ts_codes:
            data = self._make_request(
                "daily_basic",
                fields="ts_code,trade_date,close,turnover_rate,volume_ratio,pe,pb",
                ts_code=ts_code,
                trade_date=trade_date
            )
            if data and "items" in data and data["items"]:
                result[ts_code] = data["items"][0]
        return result
    
    async def fetch_moneyflow_hsgt(self) -> Dict:
        """获取沪深港通资金流向"""
        trade_date = datetime.now().strftime("%Y%m%d")
        data = self._make_request(
            "moneyflow_hsgt",
            fields="trade_date,ggt_ss,ggt_sz,hgt,sgt,north_money,south_money",
            trade_date=trade_date
        )
        
        # 处理返回数据，提取实际的资金流向信息
        if isinstance(data, dict) and "items" in data and data["items"]:
            # 如果有数据，返回第一条记录的字段名和值的映射
            fields = data.get("fields", [])
            items = data.get("items", [])
            if fields and items and len(items[0]) >= len(fields):
                result = {}
                for i, field in enumerate(fields):
                    result[field] = items[0][i] if i < len(items[0]) else None
                return result
        
        # 如果没有数据，返回默认结构
        return {
            "trade_date": trade_date,
            "north_money": 0,  # 北向资金
            "south_money": 0,  # 南向资金
            "hgt": 0,          # 沪股通
            "sgt": 0,          # 深股通
            "ggt_ss": 0,       # 港股通(沪)
            "ggt_sz": 0        # 港股通(深)
        }
    
    async def fetch_top10_holders(self, ts_code: str) -> Dict:
        """获取前十大股东"""
        return self._make_request(
            "top10_holders",
            fields="ts_code,ann_date,end_date,holder_name,hold_amount,hold_ratio",
            ts_code=ts_code
        )

class SinaFinanceSource(DataSource):
    """新浪财经实时数据源 - 作为备用数据源"""
    
    def __init__(self):
        self.base_url = SINA_API_BASE
    
    async def fetch_realtime_quotes(self, stock_codes: List[str]) -> Dict:
        """获取实时股价数据"""
        result = {}
        try:
            # 转换股票代码格式 (000001.SZ -> sz000001)
            sina_codes = []
            for code in stock_codes:
                if code.endswith('.SZ'):
                    sina_codes.append('sz' + code[:6])
                elif code.endswith('.SH'):
                    sina_codes.append('sh' + code[:6])
            
            codes_str = ','.join(sina_codes)
            url = f"{self.base_url}/list={codes_str}"
            
            response = requests.get(url, timeout=5)
            response.encoding = 'gbk'
            
            lines = response.text.strip().split('\n')
            for i, line in enumerate(lines):
                if 'var hq_str_' in line:
                    data_part = line.split('"')[1]
                    if data_part:
                        fields = data_part.split(',')
                        if len(fields) >= 32:
                            result[stock_codes[i]] = {
                                'name': fields[0],
                                'current_price': float(fields[3]) if fields[3] else 0,
                                'prev_close': float(fields[2]) if fields[2] else 0,
                                'change_pct': 0,  # 需要计算
                                'volume': int(fields[8]) if fields[8] else 0,
                                'amount': float(fields[9]) if fields[9] else 0,
                                'update_time': fields[31] if len(fields) > 31 else ''
                            }
                            # 计算涨跌幅
                            if result[stock_codes[i]]['prev_close'] > 0:
                                change = result[stock_codes[i]]['current_price'] - result[stock_codes[i]]['prev_close']
                                result[stock_codes[i]]['change_pct'] = (change / result[stock_codes[i]]['prev_close']) * 100
        except Exception as e:
            print(f"Error fetching Sina data: {e}")
        
        return result
    
    def fetch_basic_quotes(self) -> Dict:
        """获取基本行情数据（同步方法）"""
        try:
            # 获取主要指数的基本信息
            basic_codes = ['sh000001', 'sz399001', 'sz399006']  # 上证、深证、创业板
            url = f"{self.base_url}/list={','.join(basic_codes)}"
            
            response = requests.get(url, timeout=5)
            response.encoding = 'gbk'
            
            result = {
                'sh_index': {'current': 3000.0, 'change_pct': 0.0},
                'sz_index': {'current': 10000.0, 'change_pct': 0.0},
                'cy_index': {'current': 2000.0, 'change_pct': 0.0}
            }
            
            # 简单解析数据
            lines = response.text.strip().split('\n')
            index_map = {0: 'sh_index', 1: 'sz_index', 2: 'cy_index'}
            
            for i, line in enumerate(lines[:3]):
                if 'var hq_str_' in line and i in index_map:
                    data_part = line.split('"')[1]
                    if data_part:
                        fields = data_part.split(',')
                        if len(fields) >= 4:
                            current = float(fields[3]) if fields[3] else 3000.0
                            prev_close = float(fields[2]) if fields[2] else current
                            change_pct = ((current - prev_close) / prev_close * 100) if prev_close > 0 else 0
                            
                            result[index_map[i]] = {
                                'current': current,
                                'change_pct': change_pct
                            }
            
            return result
        except Exception as e:
            # 返回默认数据
            return {
                'sh_index': {'current': 3000.0, 'change_pct': 0.0},
                'sz_index': {'current': 10000.0, 'change_pct': 0.0},
                'cy_index': {'current': 2000.0, 'change_pct': 0.0}
            }

class EastMoneySource(DataSource):
    """东方财富数据源 - 热点概念和资金流向"""
    
    def __init__(self):
        self.base_url = EASTMONEY_API_BASE
    
    async def fetch_concept_boards(self) -> List[Dict]:
        """获取概念板块数据"""
        try:
            # 这里使用模拟数据，实际应用需要解析东方财富的API
            concepts = [
                {"name": "人工智能", "change_pct": random.uniform(-3, 8), "lead_stocks": ["科大讯飞", "海康威视"]},
                {"name": "新能源汽车", "change_pct": random.uniform(-2, 6), "lead_stocks": ["比亚迪", "宁德时代"]},
                {"name": "半导体", "change_pct": random.uniform(-4, 5), "lead_stocks": ["中芯国际", "韦尔股份"]},
                {"name": "5G通信", "change_pct": random.uniform(-2, 4), "lead_stocks": ["中兴通讯", "华为产业链"]},
                {"name": "医药生物", "change_pct": random.uniform(-1, 3), "lead_stocks": ["药明康德", "恒瑞医药"]},
            ]
            return concepts
        except Exception as e:
            print(f"Error fetching concept boards: {e}")
            return []
    
    async def fetch_market_sentiment(self) -> Dict:
        """获取市场情绪指标"""
        return {
            "up_down_ratio": random.uniform(0.3, 1.8),  # 涨跌比
            "limit_up_count": random.randint(10, 150),  # 涨停个数
            "limit_down_count": random.randint(0, 30),  # 跌停个数
            "turnover_billion": random.uniform(800, 1200),  # 成交额(亿)
            "new_high_count": random.randint(5, 80),  # 创新高个数
            "new_low_count": random.randint(5, 120),  # 创新低个数
            "timestamp": datetime.utcnow().isoformat()
        }

class AStockDataManager:
    """A股数据管理器，协调各种数据源"""
    
    def __init__(self):
        self.tushare_source = TushareSource()
        self.sina_source = SinaFinanceSource()
        self.eastmoney_source = EastMoneySource()
        
        # 主要关注的指数和个股
        self.major_indices = [
            "000001.SH",  # 上证指数
            "399001.SZ",  # 深证成指  
            "399006.SZ",  # 创业板指
            "000300.SH",  # 沪深300
            "000852.SH",  # 中证1000
        ]
        
        self.focus_stocks = [
            "000858.SZ",  # 五粮液
            "000002.SZ",  # 万科A
            "600036.SH",  # 招商银行
            "000001.SZ",  # 平安银行
            "600519.SH",  # 贵州茅台
        ]
        
        # 缓存数据
        self._latest_market_data = None
        self._latest_index_data = None
        self._latest_concept_data = None
        self._latest_market_sentiment = None
    
    async def get_latest_market_data(self) -> Dict:
        """获取最新市场数据"""
        if not self._latest_market_data:
            await self.refresh_market_data()
        return self._latest_market_data
    
    async def refresh_market_data(self):
        """刷新市场数据"""
        try:
            # 获取指数数据
            index_data = await self.tushare_source.fetch_index_daily(self.major_indices)
            
            # 获取个股基本数据
            stock_data = await self.sina_source.fetch_realtime_quotes(self.focus_stocks)
            
            # 获取沪深港通资金流向
            money_flow = await self.tushare_source.fetch_moneyflow_hsgt()
            
            # 获取市场情绪
            market_sentiment = await self.eastmoney_source.fetch_market_sentiment()
            
            # 整合数据
            self._latest_market_data = {
                "indices": self._format_index_data(index_data),
                "stocks": self._format_stock_data(stock_data), 
                "money_flow": money_flow,
                "market_sentiment": market_sentiment,
                "timestamp": datetime.utcnow().isoformat()
            }
            
        except Exception as e:
            print(f"Error refreshing market data: {e}")
            # 提供默认数据
            self._latest_market_data = self._get_default_market_data()
        
        return self._latest_market_data
    
    def _format_index_data(self, raw_data: Dict) -> Dict:
        """格式化指数数据"""
        formatted = {}
        index_names = {
            "000001.SH": "上证指数",
            "399001.SZ": "深证成指",
            "399006.SZ": "创业板指", 
            "000300.SH": "沪深300",
            "000852.SH": "中证1000"
        }
        
        for ts_code, data in raw_data.items():
            if isinstance(data, list) and len(data) >= 4:
                formatted[ts_code] = {
                    "name": index_names.get(ts_code, ts_code),
                    "close": float(data[2]) if data[2] else 0,
                    "change_pct": float(data[3]) if data[3] else 0,
                    "volume": int(data[4]) if data[4] else 0,
                    "amount": float(data[5]) if len(data) > 5 and data[5] else 0
                }
        
        return formatted
    
    def _format_stock_data(self, raw_data: Dict) -> Dict:
        """格式化个股数据"""
        return raw_data
    
    async def get_latest_concept_data(self) -> List[Dict]:
        """获取最新概念板块数据"""
        if not self._latest_concept_data:
            self._latest_concept_data = await self.eastmoney_source.fetch_concept_boards()
        return self._latest_concept_data
    
    async def refresh_concept_data(self):
        """刷新概念数据"""
        self._latest_concept_data = await self.eastmoney_source.fetch_concept_boards()
        return self._latest_concept_data
    
    async def get_market_summary(self) -> Dict:
        """获取市场摘要 - 供AI分析师使用"""
        market_data = await self.get_latest_market_data()
        concept_data = await self.get_latest_concept_data()
        
        # 提取关键信息
        summary = {
            "market_overview": {
                "shanghai_index": market_data.get("indices", {}).get("000001.SH", {}),
                "shenzhen_index": market_data.get("indices", {}).get("399001.SZ", {}),
                "chinext_index": market_data.get("indices", {}).get("399006.SZ", {})
            },
            "money_flow": market_data.get("money_flow", {}),
            "market_sentiment": market_data.get("market_sentiment", {}),
            "hot_concepts": concept_data[:5],  # 取前5个热点概念
            "focus_stocks": market_data.get("stocks", {}),
            "timestamp": datetime.utcnow().isoformat()
        }
        
        return summary
    
    async def get_latest_news(self) -> List[Dict]:
        """获取最新A股新闻"""
        # 模拟A股新闻数据
        news_items = [
            {
                "id": 1,
                "title": "A股三大指数收盘涨跌不一，创业板指涨1.2%",
                "content": "今日A股市场震荡上行，上证指数微涨0.3%，深证成指上涨0.8%，创业板指表现突出涨1.2%。",
                "source": "财经网",
                "timestamp": datetime.utcnow().isoformat(),
                "category": "market"
            },
            {
                "id": 2, 
                "title": "科技股集体走强，人工智能概念领涨",
                "content": "人工智能、芯片半导体等科技板块今日表现亮眼，多只个股涨停。",
                "source": "证券时报",
                "timestamp": datetime.utcnow().isoformat(),
                "category": "sector"
            },
            {
                "id": 3,
                "title": "央行今日开展1000亿元逆回购操作",
                "content": "为维护银行体系流动性合理充裕，央行今日开展1000亿元7天期逆回购操作。",
                "source": "中国证券报",
                "timestamp": datetime.utcnow().isoformat(),
                "category": "policy"
            },
            {
                "id": 4,
                "title": "外资继续净流入，北向资金净买入50亿元",
                "content": "今日北向资金净流入50.2亿元，连续第三个交易日净流入，显示外资对A股信心增强。",
                "source": "第一财经",
                "timestamp": datetime.utcnow().isoformat(),
                "category": "capital"
            }
        ]
        return news_items
    
    async def get_latest_trends(self) -> Dict:
        """获取最新趋势数据"""
        # 获取市场数据
        market_data = await self.get_latest_market_data()
        concept_data = await self.get_latest_concept_data()
        
        # 构建趋势数据
        trends_data = {
            "market_sentiment": {
                "fear_greed_index": random.randint(20, 80),  # 恐慌贪婪指数
                "sentiment_label": random.choice(["谨慎", "中性", "乐观", "极度乐观"]),
                "confidence_level": random.uniform(0.3, 0.9)
            },
            "trending_topics": [
                "人工智能", "新能源汽车", "芯片半导体", "医药生物", 
                "5G通信", "新基建", "碳中和", "数字经济"
            ],
            "sector_performance": {
                "best_performing": random.choice(["科技", "医药", "新能源", "消费"]),
                "worst_performing": random.choice(["房地产", "传统制造", "钢铁", "煤炭"]),
                "rotation_trend": "科技板块轮动上涨"
            },
            "market_heat": {
                "overall_heat": random.uniform(0.4, 0.8),
                "volume_activity": random.uniform(0.5, 0.9),
                "investor_sentiment": random.choice(["积极", "观望", "谨慎"])
            },
            "timestamp": datetime.utcnow().isoformat()
        }
        
        return trends_data
    
    def get_market_data(self) -> Dict:
        """同步获取市场数据（用于测试）"""
        try:
            # 使用同步方式获取备用数据
            return self.sina_source.fetch_basic_quotes()
        except Exception as e:
            return self._get_default_market_data()
    
    def _get_default_market_data(self) -> Dict:
        """获取默认市场数据（当API失败时使用）"""
        return {
            "indices": {
                "000001.SH": {"name": "上证指数", "close": 3000.0, "change_pct": 0.0, "volume": 0, "amount": 0}
            },
            "stocks": {},
            "money_flow": {},
            "market_sentiment": {
                "up_down_ratio": 1.0,
                "limit_up_count": 50,
                "limit_down_count": 10,
                "turnover_billion": 1000,
                "timestamp": datetime.utcnow().isoformat()
            },
            "timestamp": datetime.utcnow().isoformat()
        }
    
    async def get_agent_context_data(self) -> Dict:
        """为AI agent提供上下文数据"""
        market_data = await self.get_latest_market_data()
        concept_data = await self.get_latest_concept_data()
        
        return {
            "market_data": market_data,
            "concept_data": concept_data,
            "topics": self._extract_discussion_topics(market_data, concept_data),
            "market_stats": market_data.get("market_sentiment", {}),
            "recent_trends": self._analyze_trends(market_data)
        }
    
    def _extract_discussion_topics(self, market_data: Dict, concept_data: List[Dict]) -> List[str]:
        """提取讨论话题"""
        topics = []
        
        # 基于指数表现
        indices = market_data.get("indices", {})
        for code, data in indices.items():
            change_pct = data.get("change_pct", 0)
            if abs(change_pct) > 1:  # 涨跌幅超过1%
                topics.append(f"{data.get('name', '')}表现异常")
        
        # 基于概念板块
        for concept in concept_data[:3]:
            if abs(concept.get("change_pct", 0)) > 2:  # 涨跌幅超过2%
                topics.append(f"{concept.get('name', '')}概念活跃")
        
        # 默认话题
        if not topics:
            topics = ["市场分析", "技术面解读", "基本面研究"]
        
        return topics
    
    def _analyze_trends(self, market_data: Dict) -> Dict:
        """分析市场趋势"""
        sentiment = market_data.get("market_sentiment", {})
        
        return {
            "market_direction": "上涨" if sentiment.get("up_down_ratio", 1) > 1 else "下跌",
            "volume_trend": "放量" if sentiment.get("turnover_billion", 1000) > 1000 else "缩量", 
            "sentiment_score": min(sentiment.get("up_down_ratio", 1) * 50, 100),
            "risk_level": "低" if sentiment.get("limit_down_count", 10) < 20 else "中"
        }

# 全局数据管理器实例
data_manager = AStockDataManager()