from __future__ import annotations

import asyncio
import json
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List

from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import FileResponse
from starlette.responses import StreamingResponse
from fastapi.staticfiles import StaticFiles
from backend.infra.logging import logger

from .config import BASE_DIR, get_settings
from .schemas import (
    AnalyzeRequest,
    AnalyzeResponse,
    SnapshotResponse,
    StrategyNavResponse,
    StrategyPositionsResponse,
    StrategyTradesResponse,
    AgentRunResponse,
    NavPoint,
    PositionItem,
    TradeItem,
    AgentRunItem,
    CompetitionOverviewResponse,
    CompetitionLeaderboardResponse,
    CompetitionParticipantDetailResponse,
    CompetitionRoundDetailResponse,
    MarketIntelResponse,
)
from backend.infra import (
    close_redis,
    close_sync_redis,
    dispose_engine,
    get_async_engine,
    get_async_session,
    get_redis_client,
)
from backend.infra.logging import log_context
from backend.infra.metrics_http import metrics_router
from backend.infra.alertmanager_adapter import build_alertmanager_router
from backend.agent.llm_service import YunwuLLMService
from backend.market.tushare_service import TushareService
from backend.infra.notifications import FeishuWebhookNotifier
from backend.infra.tracing import setup_tracing
from .utils.json_tools import to_plain_python
from backend.trading.services.reporting import fetch_nav_history, fetch_positions, fetch_recent_trades
from backend.app.agent_logs import load_recent_agent_runs
from backend.competition import CompetitionEventBus, CompetitionService

settings = get_settings()
setup_tracing("abattle-api")

app = FastAPI(title="A股智能分析助手", version="0.1.0")
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

app.include_router(metrics_router)


tushare_service = TushareService(settings.tushare_token)
yunwu_service = YunwuLLMService(
    base_url=settings.yunwu_api_base,
    api_key=settings.yunwu_api_key,
    default_model=settings.yunwu_primary_model,
    timeout=settings.request_timeout,
    fallback_models=settings.yunwu_available_models,
    responses_models=settings.yunwu_responses_models,
)
feishu_notifier = FeishuWebhookNotifier(settings.feishu_webhook_url)
app.include_router(build_alertmanager_router(feishu_notifier))

competition_service = CompetitionService(settings.default_competition_id)
competition_event_bus = CompetitionEventBus()


@app.on_event("startup")
async def startup() -> None:
    # 提前初始化关键基础设施，便于及早暴露连接问题
    get_async_engine()
    get_redis_client()
    settings.agent_log_path.mkdir(parents=True, exist_ok=True)


@app.on_event("shutdown")
async def shutdown() -> None:
    await yunwu_service.aclose()
    await close_redis()
    await dispose_engine()
    close_sync_redis()


SYSTEM_PROMPT = """
你是一名经验丰富的A股投研顾问，为用户提供客观、结构化、易读的分析。你的回答需要：
1. 首先给出一句话风格化概览，引出重点；
2. 按章节输出，章节标题使用加粗，包含“公司概览”、“行情与技术”、“资金动向”、“财务表现”、“风险提示”；
3. 每个章节使用要点式表达，引用数据时标明日期或来源字段；
4. 若数据缺失需明确说明，不要编造；
5. 收尾补充一句温馨提示，强调非投资建议。
""".strip()


@app.post("/api/analyze", response_model=AnalyzeResponse)
async def analyze_stock(payload: AnalyzeRequest) -> AnalyzeResponse:
    with log_context(endpoint="analyze_stock", query=payload.query):
        if payload.snapshot:
            plain_snapshot: Dict[str, Any] = to_plain_python(payload.snapshot)
        else:
            try:
                snapshot = tushare_service.build_snapshot(payload.query)
            except ValueError as exc:
                raise HTTPException(status_code=404, detail=str(exc)) from exc
            except Exception as exc:  # noqa: BLE001
                raise HTTPException(status_code=502, detail=f"Tushare数据获取失败: {exc}") from exc
            plain_snapshot = to_plain_python(snapshot)
        data_json = json.dumps(plain_snapshot, ensure_ascii=False, indent=2)

        stock_name = plain_snapshot["basic"].get("name", "")
        ts_code = plain_snapshot["basic"].get("ts_code", "")
        symbol = plain_snapshot["basic"].get("symbol", "")
        market = plain_snapshot["basic"].get("market", "")

        user_prompt = (
            f"请针对A股上市公司 {stock_name} ({ts_code}) 做综合分析。\n"
            "结合以下JSON数据，给出投资研究风格的结论：\n"
            f"```json\n{data_json}\n```\n"
            "分析要关注近期股价走势、估值水平、资金流向、财务亮点或隐患，并指出潜在风险。"
        )

        messages = [
            {"role": "system", "content": SYSTEM_PROMPT},
            {"role": "user", "content": user_prompt},
        ]

        try:
            analysis = await yunwu_service.chat(messages, model=payload.model, temperature=payload.temperature or 0.65)
        except Exception as exc:  # noqa: BLE001
            raise HTTPException(status_code=502, detail=f"云雾LLM请求失败: {exc}") from exc

        return AnalyzeResponse(
            query=payload.query,
            ts_code=ts_code,
            company_name=stock_name,
            exchange_symbol=f"{symbol}.{market}" if symbol and market else symbol or ts_code,
            analysis=analysis,
            snapshot=plain_snapshot,
            retrieved_at=datetime.utcnow().isoformat() + "Z",
        )


@app.get("/api/snapshot", response_model=SnapshotResponse)
async def get_snapshot(query: str) -> SnapshotResponse:
    with log_context(endpoint="get_snapshot", query=query):
        try:
            snapshot = tushare_service.build_snapshot(query)
        except ValueError as exc:
            raise HTTPException(status_code=404, detail=str(exc)) from exc
        except Exception as exc:  # noqa: BLE001
            raise HTTPException(status_code=502, detail=f"Tushare数据获取失败: {exc}") from exc

        plain_snapshot: Dict[str, Any] = to_plain_python(snapshot)
        basic = plain_snapshot.get("basic", {})
        ts_code = basic.get("ts_code", "")
        symbol = basic.get("symbol", "")
        market = basic.get("market", "")

        return SnapshotResponse(
            query=query,
            ts_code=ts_code,
            company_name=basic.get("name", ""),
            exchange_symbol=f"{symbol}.{market}" if symbol and market else symbol or ts_code,
            snapshot=plain_snapshot,
            retrieved_at=datetime.utcnow().isoformat() + "Z",
        )


@app.get("/api/strategy/{strategy_account_id}/nav", response_model=StrategyNavResponse)
async def get_strategy_nav(strategy_account_id: int, limit: int = 90) -> StrategyNavResponse:
    session_factory = get_async_session()
    async with session_factory() as session:
        rows = await fetch_nav_history(session, strategy_account_id, limit=limit)
    points = [
        NavPoint(
            date=row.date.isoformat(),
            nav=row.nav,
            total_equity=row.total_equity,
            cash=row.cash,
            pnl_daily=row.pnl_daily,
            pnl_total=row.pnl_total,
        )
        for row in rows
    ]
    return StrategyNavResponse(strategy_account_id=strategy_account_id, points=points)


@app.get("/api/strategy/{strategy_account_id}/positions", response_model=StrategyPositionsResponse)
async def get_strategy_positions(strategy_account_id: int) -> StrategyPositionsResponse:
    session_factory = get_async_session()
    async with session_factory() as session:
        rows = await fetch_positions(session, strategy_account_id)
    items = [
        PositionItem(
            ts_code=row.ts_code,
            quantity=row.quantity,
            avg_cost=row.avg_cost,
            frozen_quantity=row.frozen_quantity,
            market_value=row.market_value,
            unrealized_pnl=row.unrealized_pnl,
        )
        for row in rows
    ]
    return StrategyPositionsResponse(strategy_account_id=strategy_account_id, positions=items)


@app.get("/api/strategy/{strategy_account_id}/trades", response_model=StrategyTradesResponse)
async def get_strategy_trades(strategy_account_id: int, limit: int = 50) -> StrategyTradesResponse:
    session_factory = get_async_session()
    async with session_factory() as session:
        rows = await fetch_recent_trades(session, strategy_account_id, limit=limit)
    trades: List[TradeItem] = []
    for trade, order in rows:
        trades.append(
            TradeItem(
                trade_id=trade.id,
                order_id=order.id,
                ts_code=trade.ts_code,
                side=order.side.value if order.side else "",
                price=trade.price,
                quantity=trade.quantity,
                amount=trade.amount,
                trade_time=trade.trade_time.isoformat(),
                order_status=order.status.value if order.status else "",
            )
        )
    return StrategyTradesResponse(strategy_account_id=strategy_account_id, trades=trades)


@app.get("/api/agents/runs", response_model=AgentRunResponse)
async def get_agent_runs(limit: int = 20) -> AgentRunResponse:
    records = load_recent_agent_runs(settings.agent_log_path, limit=limit)
    items: List[AgentRunItem] = []
    for record in records:
        parsed = record.get("parsed") or {}
        final_payload = parsed.get("final") if isinstance(parsed, dict) else record.get("final")
        summary = ""
        score = None
        if isinstance(final_payload, dict):
            summary = str(final_payload.get("summary") or final_payload.get("final") or "")
            raw_score = final_payload.get("score")
            score = float(raw_score) if isinstance(raw_score, (int, float)) else None
        elif isinstance(final_payload, str):
            summary = final_payload
        else:
            summary = str(final_payload or "")

        items.append(
            AgentRunItem(
                created_at=record.get("created_at", ""),
                agent=record.get("agent", ""),
                objective=record.get("objective", ""),
                final=summary,
                score=score,
                source_file=record.get("source_file"),
            )
        )
    return AgentRunResponse(items=items)


@app.get("/api/competition/stream")
async def stream_competition() -> StreamingResponse:
    subscriber = competition_event_bus.subscribe()

    async def event_generator() -> Any:
        try:
            while True:
                try:
                    event = await asyncio.wait_for(subscriber.__anext__(), timeout=15)
                except asyncio.TimeoutError:
                    yield ": heartbeat\n\n"
                    continue
                if not isinstance(event, dict):
                    continue
                event_name = event.get("event", "update")
                payload = json.dumps(event, ensure_ascii=False)
                yield f"event: {event_name}\ndata: {payload}\n\n"
        except (StopAsyncIteration, asyncio.CancelledError):  # pragma: no cover - 连接中断
            pass
        finally:
            close = getattr(subscriber, "aclose", None)
            if close is not None:
                try:
                    await close()
                except RuntimeError:  # pragma: no cover - 竞争关闭
                    logger.debug("competition:sse_close_in_progress")

    return StreamingResponse(event_generator(), media_type="text/event-stream")


@app.get("/api/competition/overview", response_model=CompetitionOverviewResponse)
async def get_competition_overview() -> CompetitionOverviewResponse:
    session_factory = get_async_session()
    async with session_factory() as session:
        payload = await competition_service.fetch_overview(session)
    return CompetitionOverviewResponse(**payload)


@app.get("/api/competition/participants", response_model=CompetitionLeaderboardResponse)
async def get_competition_leaderboard(limit: int = 50) -> CompetitionLeaderboardResponse:
    session_factory = get_async_session()
    async with session_factory() as session:
        participants = await competition_service.fetch_leaderboard(session, limit=limit)
    return CompetitionLeaderboardResponse(participants=participants)


@app.get("/api/competition/participants/{participant_id}", response_model=CompetitionParticipantDetailResponse)
async def get_competition_participant_detail(
    participant_id: int,
    rounds_limit: int = 10,
    include_intel: bool = True,
) -> CompetitionParticipantDetailResponse:
    session_factory = get_async_session()
    async with session_factory() as session:
        detail = await competition_service.fetch_participant_detail(session, participant_id, rounds_limit=rounds_limit)
    if include_intel:
        participant = detail.get("participant", {}) if isinstance(detail, dict) else {}
        focus_symbol = participant.get("focus_symbol")
        if focus_symbol:
            try:
                intel_payload = tushare_service.build_intel(focus_symbol, detail="brief")
                detail["intel"] = to_plain_python(intel_payload)
            except Exception as exc:  # noqa: BLE001
                detail["intel"] = {"error": str(exc)}
    return CompetitionParticipantDetailResponse(**detail)


@app.get("/api/competition/rounds/{round_id}", response_model=CompetitionRoundDetailResponse)
async def get_competition_round_detail(round_id: int) -> CompetitionRoundDetailResponse:
    session_factory = get_async_session()
    async with session_factory() as session:
        detail = await competition_service.fetch_round_detail(session, round_id)
    return CompetitionRoundDetailResponse(**detail)


@app.get("/api/market/intel", response_model=MarketIntelResponse)
async def get_market_intel(
    query: str,
    news_limit: int = 5,
    announcement_limit: int = 10,
    detail: str = "brief",
) -> MarketIntelResponse:
    try:
        intel = tushare_service.build_intel(
            query,
            news_limit=news_limit,
            announcement_limit=announcement_limit,
            detail=detail,
        )
    except ValueError as exc:
        raise HTTPException(status_code=404, detail=str(exc)) from exc
    except Exception as exc:  # noqa: BLE001
        raise HTTPException(status_code=502, detail=f"情报数据获取失败: {exc}") from exc
    plain = to_plain_python(intel)
    return MarketIntelResponse(**plain)


# --- 静态资源 ---
FRONTEND_DIR = BASE_DIR / "frontend"
if FRONTEND_DIR.exists():
    app.mount("/static", StaticFiles(directory=str(FRONTEND_DIR)), name="frontend-static")

    @app.get("/", response_class=FileResponse)
    async def serve_index() -> FileResponse:
        return FileResponse(FRONTEND_DIR / "index.html")
else:
    INDEX_FALLBACK = Path(__file__).resolve().parent / "templates" / "placeholder.html"

    @app.get("/", response_class=FileResponse)
    async def placeholder() -> FileResponse:
        return FileResponse(INDEX_FALLBACK)
