"""配置加载模块。

提供一个中心化的 `Settings` 对象，负责读取 `config/.env` 中的所有敏感配置，
并为其他模块提供类型安全的访问方式。
"""

from __future__ import annotations

from functools import lru_cache
from pathlib import Path
from typing import List, Optional

from pydantic import Field
from pydantic_settings import BaseSettings, SettingsConfigDict

BASE_DIR = Path(__file__).resolve().parent.parent
ENV_FILE = BASE_DIR / "config" / ".env"


class Settings(BaseSettings):
    """顶层配置对象。"""

    model_config = SettingsConfigDict(
        env_file=ENV_FILE,
        env_file_encoding="utf-8",
        case_sensitive=False,
        extra="ignore",
    )

    # 核心环境
    laca_env: str = Field('local', alias='LACA_ENV')
    laca_trading_mode: str = Field('simulation', alias='LACA_TRADING_MODE')
    tushare_token: Optional[str] = Field(default=None, alias='TUSHARE_TOKEN')
    laca_sim_initial_cash: float = Field(1_000_000, alias='LACA_SIM_INITIAL_CASH')
    laca_sim_commission_rate: float = Field(0.0003, alias='LACA_SIM_COMMISSION_RATE')

    # SSH（兼容旧系统，可选）
    ssh_primary_server: str = Field('', alias='SSH_PRIMARY_SERVER')
    ssh_laca_server: str = Field('', alias='SSH_LACA_SERVER')

    # Binance（旧系统，后续将移除）
    binance_api_key: str = Field('', alias='BINANCE_API_KEY')
    binance_secret_key: str = Field('', alias='BINANCE_SECRET_KEY')

    # 市场资讯（旧系统，后续将移除）
    tokensinsight_api_key: str = Field('', alias='TOKENSINSIGHT_API_KEY')

    # LLM - Yunwu
    yunwu_api_base: str = Field(..., alias="YUNWU_API_BASE")
    yunwu_api_key: str = Field(..., alias="YUNWU_API_KEY")
    yunwu_primary_model: str = Field(..., alias="YUNWU_PRIMARY_MODEL")
    yunwu_secondary_model: str = Field(..., alias="YUNWU_SECONDARY_MODEL")
    yunwu_default_models: str = Field(..., alias="YUNWU_DEFAULT_MODELS")
    yunwu_nano_api_key: str = Field(..., alias="YUNWU_nano_API_KEY")
    yunwu_nano_default_model: str = Field(..., alias="YUNWU_nano_DEFAULT_MODEL")

    # LLM - Doubao
    doubao_api_base: str = Field(..., alias="DOUBAO_API_BASE")
    doubao_api_key: str = Field(..., alias="DOUBAO_API_KEY")
    doubao_default_model: str = Field(..., alias="DOUBAO_DEFAULT_MODEL")

    # Notifications
    telegram_bot_token: str = Field(default="", alias="TELEGRAM_BOT_TOKEN")
    telegram_chat_id: str = Field(default="", alias="TELEGRAM_CHAT_ID")
    feishu_webhook_url: str = Field(default="", alias="FEISHU_WEBHOOK_URL")

    @property
    def yunwu_models(self) -> List[str]:
        """将逗号分隔的模型列表转换为 List。"""

        return [m.strip() for m in self.yunwu_default_models.split(",") if m.strip()]


@lru_cache()
def get_settings() -> Settings:
    """使用 LRU 缓存确保配置对象在进程内只初始化一次。"""

    if not ENV_FILE.exists():
        raise FileNotFoundError("config/.env 不存在，请先添加配置文件")
    return Settings()


__all__ = ["Settings", "get_settings"]
