"""A 股 LLM 推理器 - 基于 EnhancedLLMReasoner 适配 A 股市场。

这个模块扩展了 EnhancedLLMReasoner，针对 A 股市场进行优化：
1. 使用 A 股专用 Prompt 模板
2. 适配 A 股数据结构（AccountSnapshot）
3. 集成 A 股风控规则（涨跌停、T+1、停牌等）
"""

from __future__ import annotations

import json
from datetime import date, datetime, timezone
from typing import Any, Dict, List, Optional

import structlog

from laca.agent.ashare_context import AShareObservation
from laca.agent.enhanced_llm_reasoner import EnhancedLLMReasoner
from laca.agent.events import EventType
from laca.agent.trading_state import TradingMode
from laca.agent.reasoner import PlanStep, ReasonerResult, TradingPlan, TradingRequest
from laca.config.settings import Settings, get_settings

logger = structlog.get_logger("ashare_reasoner")


class AShareReasoner(EnhancedLLMReasoner):
    """A 股专用 LLM 推理器。"""

    def __init__(
        self,
        *,
        settings: Optional[Settings] = None,
        provider: str = "yunwu",
        model: Optional[str] = None,
        temperature: float = 0.3,
        max_tokens: int = 2000,
        max_reasoning_rounds: int = 3,
        enable_function_calling: bool = True,
        permissions: Optional[set] = None,
    ) -> None:
        super().__init__(
            settings=settings,
            provider=provider,
            model=model,
            temperature=temperature,
            max_tokens=max_tokens,
            max_reasoning_rounds=max_reasoning_rounds,
            enable_function_calling=enable_function_calling,
            permissions=permissions,
        )
        logger.info(
            "ashare_reasoner.initialized",
            provider=self.provider,
            model=self.model,
            enable_function_calling=enable_function_calling,
        )

    async def plan(self, observation: AShareObservation) -> ReasonerResult:
        """基于 A 股观测数据进行推理和决策。"""

        logger.info(
            "ashare_reasoner.planning_started",
            event_type=observation.event.event_type.value,
            total_assets=observation.snapshot.total_assets,
            positions_count=len(observation.snapshot.positions),
            current_model=self.model,
            current_provider=self.provider,
        )

        # 开始新的对话
        initial_context = {
            "market_context": self._extract_ashare_market_context(observation),
            "portfolio_context": self._extract_ashare_portfolio_context(observation),
            "risk_context": self._extract_ashare_risk_context(observation),
        }

        conversation_id = self.conversation_manager.start_conversation(
            user_intent="A 股投资决策分析", initial_context=initial_context
        )

        try:
            # 执行多轮推理
            final_decision = await self._multi_round_reasoning_ashare(
                observation, conversation_id
            )

            # 转换为交易计划
            trading_plan = await self._create_ashare_plan_from_decision(
                observation, final_decision, conversation_id
            )

            # 生成结果
            reasoning_chain = self.conversation_manager.get_reasoning_chain(
                conversation_id=conversation_id
            )
            notes = self._generate_notes(final_decision, reasoning_chain)

            logger.info(
                "ashare_reasoner.planning_completed",
                conversation_id=conversation_id,
                final_action=final_decision.get("action"),
                reasoning_steps=len(reasoning_chain),
                tool_calls=len(
                    self.conversation_manager.get_tool_call_history(conversation_id)
                ),
            )

            return ReasonerResult(plan=trading_plan, notes=notes)

        except Exception as exc:
            logger.error(
                "ashare_reasoner.planning_failed",
                conversation_id=conversation_id,
                error=str(exc),
            )
            # 返回保守决策
            fallback_plan = self._create_ashare_fallback_plan(observation, str(exc))
            return ReasonerResult(
                plan=fallback_plan, notes=[f"A 股推理失败，使用保守决策: {str(exc)}"]
            )

    def _extract_ashare_market_context(
        self, observation: AShareObservation
    ) -> Dict[str, Any]:
        """提取 A 股市场上下文。"""
        return {
            "quotes": observation.market.quotes,
            "limit_prices": observation.market.limit_prices,
            "suspended_stocks": observation.market.suspended_stocks,
            "timestamp": datetime.now(timezone.utc).isoformat(),
        }

    def _extract_ashare_portfolio_context(
        self, observation: AShareObservation
    ) -> Dict[str, Any]:
        """提取 A 股组合上下文。"""
        return {
            "total_assets": observation.snapshot.total_assets,
            "available_cash": observation.snapshot.available_cash,
            "positions_count": len(observation.snapshot.positions),
            "market_value": observation.snapshot.market_value,
            "unrealized_pnl": observation.snapshot.unrealized_pnl,
        }

    def _extract_ashare_risk_context(
        self, observation: AShareObservation
    ) -> Dict[str, Any]:
        """提取 A 股风险上下文。"""
        return {
            "cash_ratio": observation.metrics.get("cash_ratio", 0.0),
            "risk_metrics": observation.metrics,
        }

    async def _multi_round_reasoning_ashare(
        self, observation: AShareObservation, conversation_id: str
    ) -> Dict[str, Any]:
        """执行 A 股专用的多轮推理。"""

        trading_mode = self._resolve_trading_mode(observation)

        # 第一轮：初始分析
        self.conversation_manager.add_reasoning_step(
            "analysis", "开始 A 股市场和投资组合分析", confidence=1.0, conversation_id=conversation_id
        )

        initial_prompt = self._build_ashare_initial_prompt(observation, trading_mode)
        analysis_result = await self._call_llm_with_tools(initial_prompt, conversation_id)

        # 记录分析结果
        self.conversation_manager.add_reasoning_step(
            "analysis",
            f"A 股市场分析完成: {analysis_result.get('analysis_summary', '')}",
            confidence=analysis_result.get("confidence", 0.5),
            metadata={"analysis": analysis_result},
            conversation_id=conversation_id,
        )

        if trading_mode is not TradingMode.TRADING:
            summary = analysis_result.get("analysis_summary") or "非交易时段，保持观察"
            final_decision = {
                "action": "HOLD",
                "ts_code": None,
                "quantity": 0,
                "price": None,
                "reason": f"当前为分析模式（{trading_mode.value}），暂不执行下单。{summary}",
                "risk_level": analysis_result.get("risk_level", "MEDIUM"),
                "confidence": analysis_result.get("confidence", 0.6),
                "analysis_summary": summary,
                "mode": trading_mode.value,
                "decision_summary": summary,
                "is_final_decision": True,
                "needs_more_data": analysis_result.get("needs_more_data", False),
                "recommended_tools": analysis_result.get("recommended_tools", []),
            }

            self.conversation_manager.add_reasoning_step(
                "execution",
                "分析模式 - 暂不下单",
                confidence=analysis_result.get("confidence", 0.6),
                metadata={"final_decision": final_decision},
                conversation_id=conversation_id,
            )

            return final_decision

        # 多轮推理循环（简化版，直接使用 1 轮）
        reasoning_prompt = self._build_ashare_reasoning_prompt(
            observation, analysis_result, trading_mode
        )
        final_decision = await self._call_llm_with_tools(reasoning_prompt, conversation_id)

        # 记录推理步骤
        self.conversation_manager.add_reasoning_step(
            "execution",
            f"A 股最终决策: {final_decision.get('action', 'HOLD')}",
            confidence=final_decision.get("confidence", 0.3),
            metadata={"final_decision": final_decision},
            conversation_id=conversation_id,
        )

        return final_decision

    def _resolve_trading_mode(self, observation: AShareObservation) -> TradingMode:
        """解析观测中的交易模式，默认返回交易模式。"""

        raw_mode = (observation.metadata or {}).get("trading_mode")
        if isinstance(raw_mode, str):
            try:
                return TradingMode(raw_mode)
            except ValueError:
                logger.warning("ashare_reasoner.invalid_trading_mode", raw=raw_mode)
        return TradingMode.TRADING

    def _build_ashare_initial_prompt(
        self, observation: AShareObservation, mode: TradingMode
    ) -> str:
        """构建 A 股初始分析 Prompt。"""

        mode_hint = {
            TradingMode.TRADING: "当前处于交易时段，请识别可执行的买卖机会并评估风险。",
            TradingMode.ANALYST: (
                "当前不在交易时段，你处于分析员模式：重点梳理资讯、风险、潜在机会与下一交易日的准备事项，不要给出立即执行的买卖指令。"
            ),
            TradingMode.PAUSED: "系统暂停交易，仅保持宏观监控。",
        }[mode]

        return f"""
你是一个专业的 A 股市场投资顾问，需要对当前市场情况进行深入分析。

**当前运行模式**：{mode_hint}

**当前情况**：
{self._format_ashare_observation(observation)}

**分析任务**：
1. 评估 A 股市场趋势、热点与潜在风险（结合当前模式调整侧重点）
2. 评估现有持仓状态（如有）及风险暴露
3. 识别潜在的交易机会或调整需求，给出前置条件
4. 输出需要跟进的待办事项（如情报补充、风险复核、策略调整）
5. 考虑 A 股特有规则：T+1、涨跌停、停牌、交易时间

**A 股市场特点**：
- T+1：当日买入次日可卖
- 涨跌停：±10%（ST ±5%）
- 交易时间：9:30-11:30, 13:00-15:00
- 买入单位：100 股整数倍

请以 JSON 格式返回分析结果：
{{
    "market_trend": "上涨/下跌/震荡",
    "risk_level": "LOW/MEDIUM/HIGH",
    "portfolio_health": "良好/一般/需要调整",
    "opportunities": ["机会股票代码1", "机会股票代码2"],
    "risks": ["风险1", "风险2"],
    "analysis_summary": "简要分析总结",
    "confidence": 0.8,
    "needs_more_data": false,
    "recommended_tools": ["get_stock_info", "get_current_price", "其他工具"],
    "follow_ups": ["待办事项1", "待办事项2"]
}}

如果需要更多数据，请设置 needs_more_data 为 true。
"""

    def _build_ashare_reasoning_prompt(
        self,
        observation: AShareObservation,
        context: Dict[str, Any],
        mode: TradingMode,
    ) -> str:
        """构建 A 股推理 Prompt。"""

        mode_guideline = (
            "在交易模式下，根据分析结果给出可执行的买卖或持有决策。"
            if mode is TradingMode.TRADING
            else "当前处于分析模式，保持 HOLD 并输出下一步的分析/准备要点。"
        )

        return f"""
基于之前的 A 股市场分析，现在需要做出最终投资决策。

当前运行模式：{mode.value} —— {mode_guideline}

**分析结果**：
{json.dumps(context, ensure_ascii=False, indent=2)}

**决策要求**：
必须给出明确的投资行动（BUY/SELL/HOLD）。

**如果考虑 BUY**：
- 调用 calculate_position_size() 获取建议仓位
- 检查是否有足够资金
- 确保买入数量为 100 股整数倍
- 检查是否达到持仓上限（3 只）

**如果考虑 SELL**：
- 调用 get_t1_sellable_quantity() 检查 T+1 可卖数量
- 评估持仓盈亏状态
- 考虑市场趋势变化

**如果考虑 HOLD**：
- 说明为何不交易的理由

**风险管理**：
- LOW 风险：10-15% 仓位
- MEDIUM 风险：5-10% 仓位
- HIGH 风险：3-5% 仓位

请以 JSON 格式返回最终决策：
{{
    "action": "BUY/SELL/HOLD",
    "ts_code": "000001.SZ",
    "quantity": 100,
    "price": 10.50,
    "reason": "详细决策理由",
    "risk_level": "LOW/MEDIUM/HIGH",
    "confidence": 0.8,
    "is_final_decision": true,
    "decision_summary": "决策总结"
}}
"""

    def _format_ashare_observation(self, observation: AShareObservation) -> str:
        """格式化 A 股观测数据。"""

        snapshot = observation.snapshot
        metrics = observation.metrics

        # 记忆上下文
        memory_section = ""
        if observation.memory_context:
            memory_section = f"""
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
## 📚 历史记忆与上下文

{observation.memory_context}
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

"""

        return f"""
{memory_section}**账户状态**：
- 总资产：{snapshot.total_assets:.2f} 元
- 可用资金：{snapshot.available_cash:.2f} 元
- 冻结资金：{snapshot.frozen_cash:.2f} 元
- 持仓市值：{snapshot.market_value:.2f} 元
- 未实现盈亏：{snapshot.unrealized_pnl:.2f} 元 ({snapshot.unrealized_pnl_pct:.2f}%)
- 现金比例：{metrics.get('cash_ratio', 0.0):.2%}

**持仓情况**：
{self._format_ashare_positions(snapshot.positions, observation.market)}

**市场行情**：
{self._format_ashare_market(observation.market)}

**触发事件**：
{self._format_event_context(observation.event)}
"""

    def _format_ashare_positions(self, positions, market) -> str:
        """格式化 A 股持仓。"""
        if not positions:
            return "无持仓"

        lines = []
        for pos in positions:
            # 获取市场价格
            current_price = market.quotes.get(pos.ts_code, {}).get("price", pos.cost_price)

            lines.append(
                f"- {pos.ts_code}: {pos.total_quantity} 股 "
                f"(可卖 {pos.available_quantity} 股, T+1 限制 {pos.total_quantity - pos.available_quantity} 股), "
                f"成本价 {pos.cost_price:.2f}, 当前价 {current_price:.2f}, "
                f"盈亏 {pos.unrealized_pnl:.2f} 元 ({pos.unrealized_pnl_pct:+.2f}%)"
            )
        return "\n".join(lines)

    def _format_ashare_market(self, market) -> str:
        """格式化 A 股行情。"""
        if not market or not market.quotes:
            return "暂无行情数据"

        lines = []
        for ts_code, quote in list(market.quotes.items())[:5]:
            price = quote.get("price", 0.0)
            limit_info = market.limit_prices.get(ts_code, {})
            up_limit = limit_info.get("up_limit")
            down_limit = limit_info.get("down_limit")

            suspended = " (停牌)" if ts_code in market.suspended_stocks else ""
            limit_str = (
                f", 涨停 {up_limit:.2f}, 跌停 {down_limit:.2f}"
                if up_limit and down_limit
                else ""
            )

            lines.append(f"- {ts_code}: {price:.2f}{limit_str}{suspended}")

        return "\n".join(lines)

    def _format_event_context(self, event) -> str:
        """格式化事件上下文。"""
        if event.event_type == EventType.MANUAL_COMMAND:
            return "手动命令触发"
        elif event.event_type == EventType.SCHEDULED_TICK:
            return "定时巡检"
        elif event.event_type == EventType.AUTO_SIGNAL:
            payload = event.payload or {}
            return f"自动信号触发 - {payload.get('reason', '未知')}"
        else:
            return f"{event.event_type.value}"

    async def _create_ashare_plan_from_decision(
        self,
        observation: AShareObservation,
        decision: Dict[str, Any],
        conversation_id: str,
    ) -> TradingPlan:
        """从 A 股决策创建交易计划。"""

        # 获取推理链和工具调用历史
        reasoning_chain = self.conversation_manager.get_reasoning_chain(
            conversation_id=conversation_id
        )
        tool_history = self.conversation_manager.get_tool_call_history(conversation_id)

        # 构建元数据
        metadata = {
            "enhanced_reasoning": True,
            "conversation_id": conversation_id,
            "reasoning_steps": len(reasoning_chain),
            "tool_calls": len(tool_history),
            "decision": decision,
            "llm_provider": self.provider,
            "llm_model": self.model,
            "market_type": "A_SHARE",
        }

        # 创建交易请求
        trades = []
        action = decision.get("action", "HOLD").upper()

        if action in ("BUY", "SELL"):
            ts_code = decision.get("ts_code") or decision.get("symbol")
            quantity = decision.get("quantity", 0)
            price = decision.get("price")

            if ts_code and quantity > 0:
                # A 股买入必须 100 股整数倍
                if action == "BUY":
                    quantity = (quantity // 100) * 100

                trades.append(
                    TradingRequest(
                        symbol=ts_code,
                        side=action,
                        quantity=quantity,
                        price=price,
                    )
                )

        # 创建计划步骤
        steps = [
            PlanStep(
                name="ashare_decision",
                action=f"A 股 {action} 决策: {decision.get('reason', '未知')}",
            )
        ]

        if trades:
            steps.append(
                PlanStep(
                    name="ashare_trade",
                    action=f"{action} {trades[0].symbol} × {trades[0].quantity} @ {trades[0].price}",
                )
            )

        plan = TradingPlan(
            summary=f"[A 股 LLM 决策] {action} {decision.get('ts_code', 'N/A')} - {decision.get('reason', '')[:50]}",
            trades=trades,
            steps=steps,
            preconditions={},
            metadata=metadata,
        )

        return plan

    def _parse_llm_content(self, content: str) -> Dict[str, Any]:
        """解析 LLM 返回的内容（A 股版本）。

        覆盖父类方法，使用 A 股特定的 fallback 决策。
        """
        import json

        try:
            # 尝试解析JSON
            if "```json" in content:
                json_str = content.split("```json")[1].split("```")[0].strip()
            elif "```" in content:
                json_str = content.split("```")[1].split("```")[0].strip()
            else:
                json_str = content.strip()

            result = json.loads(json_str)

            # 验证必需字段
            required_fields = ["action"]
            if all(field in result for field in required_fields):
                return result
            else:
                missing = [f for f in required_fields if f not in result]
                logger.warning(
                    "llm_content.missing_fields",
                    missing_fields=missing,
                    content_preview=content[:200]
                )
                raise ValueError(f"缺少必需字段: {missing}")

        except json.JSONDecodeError as exc:
            logger.error(
                "llm_content.json_decode_error",
                error=str(exc),
                content_length=len(content),
                content_preview=content[:500]
            )
        except Exception as exc:
            logger.error(
                "llm_content.parse_error",
                error_type=type(exc).__name__,
                error=str(exc),
                content_preview=content[:500]
            )

        # 如果解析失败，返回 A 股特定的基本结构
        logger.warning(
            "llm_content.using_fallback_decision",
            reason="解析失败，使用 A 股默认 HOLD 决策"
        )
        return {
            "action": "HOLD",
            "ts_code": None,
            "symbol": None,
            "quantity": 0,
            "price": 0.0,
            "reason": f"LLM 输出解析失败（内容长度: {len(content)}字符）",
            "confidence": 0.1,
            "risk_level": "MEDIUM",
            "is_final_decision": True
        }

    def _create_ashare_fallback_plan(
        self, observation: AShareObservation, error_msg: str
    ) -> TradingPlan:
        """创建 A 股降级决策。"""

        return TradingPlan(
            summary=f"[A 股推理降级] HOLD - {error_msg}",
            trades=[],
            steps=[PlanStep(name="hold", action="A 股推理失败，保持观望")],
            preconditions={},
            metadata={
                "enhanced_reasoning": True,
                "fallback": True,
                "error": error_msg,
                "llm_provider": self.provider,
                "llm_model": self.model,
                "market_type": "A_SHARE",
                "reasoning_steps": 0,
                "tool_calls": 0,
                "decision": {
                    "action": "HOLD",
                    "ts_code": None,
                    "symbol": None,
                    "quantity": 0,
                    "price": 0.0,
                    "reason": f"A 股推理失败: {error_msg[:100]}",
                    "risk_level": "UNKNOWN",
                    "confidence": 0.0,
                },
                "confidence": 0.0,
                "risk_level": "UNKNOWN",
            },
        )


__all__ = ["AShareReasoner"]
